///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

// Demonstrates:
//   - Computing histogram for a sub-volume (ROI).
//   - Displaying histogram using MeshViz axis/chart classes.

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/Gui/view/PoSceneView.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoClipPlane.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoMaterial.h>

#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoBaseColor.h> 
#include <Inventor/nodes/SoTranslation.h> 

#include <VolumeViz/nodes/SoVolumeGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoObliqueSlice.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoUniformGridClipping.h>
#include <VolumeViz/readers/SoVRGenericFileReader.h>
#include <VolumeViz/readers/SoVRSegyFileReader.h>

#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoROI.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoGeometryPriority.h> 
#include <LDM/nodes/SoDataRange.h>
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMDataAccess.h> 
#include <LDM/SoVolumeHistogram.h>

#include <MeshViz/nodes/PoDomain.h> 
#include <MeshViz/graph/PoSingleHistogram.h> 

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////
#define VOLUME_FILENAME "$OIVHOME/examples/data/Medical/files/head.ldm"

///////////////////////////////////////////////////////////////////////////////
const int m_Resolution = 0;
const int m_histoSize  = 256;

static float     m_histo256[m_histoSize];
static double    m_DataMin, m_DataMax;

static SbVec3i32     m_LdmDim;
static SbBox3f       m_Extent;
static SoROIManip*   m_ROIManip;
static SoVolumeData* m_VolumeData;
static SoCamera*     m_3DCamera;
static PoSingleHistogram* m_histo;

static SoSeparator*  m_root;

static PoSceneView* m_2DScene;
static SoSeparator* m_3DScene;

///////////////////////////////////////////////////////////////////////////////

static void BuildSceneGraph();

static void Build3DScene();

static void Build2DScene();

static void ROIChangedCB(void *userData, SoDragger *dragger);

static void UpdateHistogram();

static void NormalizeHisto256( int64_t* hvalues, size_t N, float histo256_f[256] );

///////////////////////////////////////////////////////////////////////////////
int
main(int, char **argv)
{
  // Initialize Inventor and Xt
  Widget myWindow = SoXt::init(argv[0]) ;
  if (myWindow == NULL) exit(1) ;
  SoVolumeRendering::init();
  PoMeshViz::init() ;
  InventorMedical::init();

  // Create scene graph (under m_root)
  BuildSceneGraph();

  // Create viewer
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);  
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    viewer->setTitle( "Compute Histogram" );
    viewer->setDecoration(FALSE);
    viewer->setViewing(FALSE);
    viewer->setSceneGraph(m_root);

  // Adjust camera
  viewer->getCamera()->orientation.setValue( SbVec3f(0.8f, 0.36f, 0.47f), 1.435f ); // Rotate for more interesting view.
  viewer->viewAll();
  MedicalHelper::dollyZoom( 1.5f, viewer->getCamera() ); // Fill viewport
  viewer->saveHomePosition();

  // Run then cleanup
  viewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;
  m_root = NULL;
  InventorMedical::finish();
  PoMeshViz::finish() ;
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
/// Build the scenegraph to demo computing and displyaing subset of volume
/// histogram in real time.
void BuildSceneGraph()
{
    // Create scene root
    m_root = new SoSeparator();

    // Perspective camera to view the data
    m_3DCamera = new SoPerspectiveCamera();
    m_root->addChild(m_3DCamera);

    // Add Open Inventor Logo and Medical Gnomon at left-bottom corner.
    m_root->addChild( MedicalHelper::exampleLogoNode() );
    m_root->addChild( new Gnomon() );

    // Add some instructions at the top of the window.
    TextBox* text = new TextBox();
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::TOP;
      text->position.setValue( 0, 0.98f, 0 );
      text->addLine( "Click and drag ROI box. Histogram will update automatically." );
      m_root->addChild(text);

    //-----------------------------------------------------------------
    // Build 3D scene
    Build3DScene();

    //-----------------------------------------------------------------
    // Build 2D scene
    Build2DScene();

    // Call callback function to update histogram 
    ROIChangedCB( NULL, m_ROIManip->getDragger() );
}

///////////////////////////////////////////////////////////////////////////////
// Create a sub-scenegraph that represent 3D scene
void Build3DScene()
{
    //---------------------------------------------------------
    // Build 3D scene
    m_3DScene = new SoSeparator();
    m_3DScene->setName( "3DScene" );
    m_root->addChild(m_3DScene);

    SoSeparator* m_VolumeRoot = new SoSeparator();
    m_3DScene->addChild(m_VolumeRoot);

    // Load data set.
    m_VolumeData = new SoVolumeData();
      m_VolumeData->fileName = VOLUME_FILENAME;
      MedicalHelper::dicomAdjustVolume( m_VolumeData );
      m_VolumeRoot->addChild(m_VolumeData);

    // Get data characteristics (min/max, dimensions and extent)
    m_VolumeData->getMinMax( m_DataMin, m_DataMax );
    m_LdmDim = m_VolumeData->data.getSize();
    m_Extent = m_VolumeData->extent.getValue();

    // Set range of values to map into the color map.
    SoDataRange* m_DataRange = new SoDataRange();
      m_VolumeRoot->addChild(m_DataRange);
      m_DataRange->min = m_DataMin;
      m_DataRange->max = 112;

    // Color map.
    SoTransferFunction* m_TransferFunction = new SoTransferFunction();
      m_VolumeRoot->addChild(m_TransferFunction);
      m_TransferFunction->predefColorMap = SoTransferFunction::STANDARD;
      m_TransferFunction->minValue       = 80;

    // Define volume rendering under new SoSeparator.  As a result:
    // 1. Volume rendering settings do not affect the slices, and
    // 2. The ROIManip only clips the volume rendering, not the slices.
    SoSeparator* volSep = new SoSeparator();
      m_VolumeRoot->addChild(volSep);

    //  Add region-of-interest (ROI) with built-in user interface.
    //  Initialize to display a subset of the volume.
    m_ROIManip = new SoROIManip();
      volSep->addChild(m_ROIManip);
      m_ROIManip->constrained = true;
      m_ROIManip->boxOn = true;
      m_ROIManip->box.setValue( m_LdmDim / 4, 3 * m_LdmDim / 4 );
      m_ROIManip->subVolume.setValue( SbVec3i32(0,0,0), m_LdmDim - SbVec3i32(1,1,1));

    // Set a method to be called when the ROI is modified.
    m_ROIManip->getDragger()->addValueChangedCallback( ROIChangedCB );

    SoVolumeRenderingQuality* m_VolRendQ = new SoVolumeRenderingQuality();
      m_VolRendQ->interpolateOnMove = TRUE;
      m_VolRendQ->deferredLighting  = TRUE;
      volSep->addChild(m_VolRendQ);

    SoVolumeRender* m_VolRend = new SoVolumeRender();
      m_VolRend->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volSep->addChild(m_VolRend);

    // Add Orthoslices to the scenegraph.
    // We add the Orthoslices after the ROIManip, so the ROIManip has priority
    // in handling mouse events.
    // Oct 2016: Somehow that's not working with v9.6.2 :-( Have to make slices unpickable.
    //SoPickStyle pickStyle = new SoPickStyle();
    //  pickStyle.style.Value = SoPickStyle.Styles.UNPICKABLE;
    //  m_VolumeRoot.AddChild(pickStyle);
    //SoSwitch m_SwitchSlices = new SoSwitch();
    //m_SwitchSlices.whichChild.Value = SoSwitch.WhichChild.SO_SWITCH_ALL;
    //for (uint i = 0; i < 3; ++i)
    //{
    //  SoOrthoSlice _slice = new SoOrthoSlice();
    //  _slice.axis.Value = ((i % 3 == 0) ? SoOrthoSlice.AxisType.X : ((i % 3 == 1) ? SoOrthoSlice.AxisType.Y : SoOrthoSlice.AxisType.Z));
    //  _slice.sliceNumber.Value = (uint)(i == 2 ? 64 : 128);
    //  _slice.alphaUse.Value = SoSlice.AlphaUses.ALPHA_AS_IS;
    //  _slice.SetName("Slice_" + i.ToString());
    //  m_SwitchSlices.AddChild(_slice);
    //}
    //m_VolumeRoot.AddChild(m_SwitchSlices);
}

///////////////////////////////////////////////////////////////////////////////
// Create a sub-scenegraph for the histogram plot.
void Build2DScene()
{
    m_2DScene = new PoSceneView();
    m_2DScene->setName( "2DScene" );
    m_root->addChild(m_2DScene);

    SoSeparator* m_2DRoot = new SoSeparator();

    // Histogram plot should not affect viewAll.
    SoBBox* bbox = new SoBBox();
      bbox->mode = SoBBox::NO_BOUNDING_BOX;
      m_2DRoot->addChild(bbox);

    SoOrthographicCamera* orthoCam = new SoOrthographicCamera();

    m_2DScene->setPart( "scene", m_2DRoot );
    m_2DScene->setPart( "cameraKit.camera", m_3DCamera );
    m_2DScene->isBorderVisible     = TRUE;
    m_2DScene->isBackgroundVisible = TRUE;
    // Set the viewport so histogram plot does not obscure gnomon.
    m_2DScene->viewportOrigin.setValue(0.15f, 0.0f);
    m_2DScene->viewportSize.setValue(0.85f, 0.2f);

    SoAnnotation* histRoot = new SoAnnotation();
    m_2DRoot->addChild(histRoot);

    // Add camera to avoid manipulation in 2D PoSceneView
    histRoot->addChild(orthoCam);

    SoTranslation* pTrans = new SoTranslation();
      pTrans->translation.setValue(0, -0.9f, 0);

    SoPickStyle* pickStyle = new SoPickStyle();
      pickStyle->style = SoPickStyle::UNPICKABLE;

    PoDomain* pDomain = new PoDomain();
    pDomain->min.setValue(0.0f, 0.0f, 0.0f);
    pDomain->max.setValue(1.8f, 1.0f, 1.0f);

    // Set default value to histogram bins
    for (int i = 0; i < m_histoSize; ++i)
        m_histo256[i] = (float)i / 256.0f;

    // Make the histogram plot semi-transparent so it can overlay the data.
    float app = 1.2f * (1.0f / 0.2f);
    m_histo = new PoSingleHistogram( SbVec2f(-app, 0), app, PoHistogram::X, m_histoSize, m_histo256, NULL);
    m_histo->valueVisibility = PoHistogram::VISIBILITY_OFF;
    m_histo->nameVisibility  = PoHistogram::VISIBILITY_OFF;
    m_histo->set( "barApp.material", "diffuseColor 0.0 0.5 1.0 transparency 0.5");
    m_histo->barSpaceValue = 0.0f;

    histRoot->addChild(pTrans);
    histRoot->addChild(pickStyle);
    histRoot->addChild(pDomain);
    histRoot->addChild(m_histo);
}

///////////////////////////////////////////////////////////////////////////////
// Called when ROI is changed
void ROIChangedCB(void* /*userData*/, SoDragger* /*dragger*/)
{
  UpdateHistogram();
}

///////////////////////////////////////////////////////////////////////////////
void UpdateHistogram()
{
  //SbBox3i32   subvol = m_ROIManip->subVolume.getValue();
  SbBox3i32   subvol = m_ROIManip->box.getValue();

  // Display ROI dimensions in user interface
  int xmin,ymin,zmin,xmax,ymax,zmax;
  subvol.getBounds( xmin, ymin, zmin, xmax, ymax, zmax );
  //updateROIdisplay( xmin, ymin, zmin, xmax, ymax, zmax );

  // get data informations
  SbBox3i32 box(SbVec3i32(xmin, ymin, zmin), SbVec3i32(xmax, ymax, zmax));
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  SoLDMDataAccess& dataAccess(m_VolumeData->getLdmDataAccess());
  SoDataSet* DS = dataAccess.getDataSet();
  SoDataSet::DataType DT = DS->getDataType();
  //query size of data buffer
  info = dataAccess.getData(m_Resolution, box,(SoBufferObject*)NULL);
  SoRef<SoCpuBufferObject> bufferObj = new SoCpuBufferObject;
  bufferObj->setSize(info.bufferSize);

  info = dataAccess.getData(m_Resolution, box, bufferObj.ptr());
  if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT) {
      //printf("buf: %d\n",info.bufferSize);
  } else
      SoError::post("Error while data access request");

  SbVec3i32 buffDim = info.bufferDimension;
  int32_t nx,ny,nz;

  buffDim.getValue(nx,ny,nz);
  int buffElements = nx*ny*nz;

  SoVolumeHistogram aHisto(DT);
  aHisto.setInputValueRange(m_DataMin,m_DataMax);
  aHisto.addValues(bufferObj.ptr(),buffElements);

  size_t numBins = aHisto.getHistoSize();
  int64_t* hvalues = aHisto.getHistogram();

  NormalizeHisto256(hvalues, numBins, m_histo256);

  float* app = m_histo->value.startEditing();
  memcpy( app, m_histo256, sizeof(float)*256 );
  m_histo->value.finishEditing();
}

///////////////////////////////////////////////////////////////////////////////
// Convert histo into 256 entries normalized between 0 and 1
void NormalizeHisto256(int64_t* hvalues, size_t N, float histo256_f[256] )
{
    size_t i,j;
    //move histo into 256 values
    int64_t hvalues256[256];
    if (N==256) {
        memcpy(hvalues256,hvalues,sizeof(int64_t)*256);
    } else { //65536
        int cc=0;
        for (i=0;i<256;i++) {
            int64_t app = 0;
            for (j=0;j<(N/256);j++) {
                app+=hvalues[cc];
                cc++;
            }
            hvalues256[i] = app;
        }
    }

    //normalize histo
    int64_t imin,imax,app;
    double dmin,ddelta,dapp;
    //determine min and max
    imax=imin=hvalues256[0];
    for (i=0;i<256;i++) {
        app=hvalues256[i];
        if (app<imin) imin=app;
        if (app>imax) imax=app;
    }
    dmin = (double)imin;
    ddelta = (double)(imax-imin);

    for (i=0;i<256;i++) {
        dapp = (double)(hvalues256[i]);
        dapp = (dapp - dmin)/ddelta;
        histo256_f[i] = (float)(dapp);
    }
}
