///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
* Example program.
* Purpose : Demonstrate how to use GPU combining feature
* author : Jerome Hummel
* date :22 November 2005
*
* Updated : Pascal Estrade
* Date of modiciation : September 26 2014.
*
* This program combines two volumes - data volume1 and "mask" volume2
* The mask volume is used to selectively make voxels completely
* transparent, effectively clipping volume1 at voxel precision.
*
* To achieve this effect we define the mask volume to contain only the
* values zero and one.  We create a custom shader (GLSL function).
* Any voxel where volume2 contains zero will be forced to zero, while
* all voxels where volume2 contains one will have the value from volume1.
* Finally we make sure that value zero maps to completely transparent
* in the transfer function.
*----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoMultiDataSeparator.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

// set to 1 tu use ldm file instead of generating in memory data (usefull
// when recording ART)
#define USE_LDM 0

#define FILENAME "$OIVHOME/examples/data/Medical/files/head.ldm"

// Comment out the following line in order to apply lighting and
// pre-integration to the combined data
#define LIGHTING_AND_PREINTEGRATION 

///////////////////////////////////////////////////////////////////////////////
// Main function
int main(int argc, char **argv)
{
    // Create the window
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow) return 0;

    SoVolumeRendering::init();
    InventorMedical::init();

    const char *filename = (argc >= 2 ? argv[1] : FILENAME);

    // Assemble the scene graph
    SoRef<SoSeparator> root = new SoSeparator;

    // Camera (must have explicit camera to use Gnomon node)
    root->addChild( new SoPerspectiveCamera() );
    
    // MUST use a multidata separator when combining volumes
    SoMultiDataSeparator *mds = new SoMultiDataSeparator;
      root->addChild(mds);

    // Add some specular reflection to the rendering
    SoMaterial *material = new SoMaterial();
      material->ambientColor.setValue(0,0,0);
      material->diffuseColor.setValue(1,1,1);
      material->specularColor.setValue(1,1,1);
      material->shininess.setValue(0.5);
      mds->addChild( material );

    // Change complexity when the dataset moves.
    SoInteractiveComplexity *intercatComplexity = new SoInteractiveComplexity();
      intercatComplexity->fieldSettings.set1Value(0,"SoComplexity value 0.3 0.9");
      intercatComplexity->fieldSettings.set1Value(1,"SoVolumeRender samplingAlignment BOUNDARY_ALIGNED BOUNDARY_ALIGNED");
      intercatComplexity->fieldSettings.set1Value(2,"SoVolumeRenderingQuality preIntegrated FALSE TRUE");
      intercatComplexity->fieldSettings.set1Value(3,"SoVolumeRender interpolation LINEAR CUBIC");
      intercatComplexity->refinementDelay = 0;
      mds->addChild( intercatComplexity );

    // Data 1
    SoVolumeData* pVolData1 = new SoVolumeData;
      pVolData1->fileName     = filename;
      SbBox3f size            = pVolData1->extent.getValue();
      SbVec3i32 vol1dim       = pVolData1->data.getSize();
      mds->addChild( pVolData1 );

    // Data 2: mask volume
    SoVolumeData* pVolData2 = new SoVolumeData();
#if USE_LDM
    pVolData2->fileName = "$OIVHOME/examples/data/Medical/files/mask.ldm";
#else
    int testXdim = vol1dim[0];
    int testYdim = vol1dim[1];
    int testZdim = vol1dim[2];
    int numTestBytes = testXdim * testYdim * testZdim;
    unsigned char *testData = new unsigned char[numTestBytes];
    int counter = 0;
    for (int k = 0; k < testZdim; k++){
        for (int j = 0; j < testYdim; j++){
            for (int i = 0; i < testXdim; i++){
                double phase = sin(4*3.1*j/testYdim) + 4*i*i/(testZdim*testZdim);//same for each given j
                float waveFactor = 23.;
                float t = char(127 + 127*sin((testZdim - k)/waveFactor + phase ));//now make it vary on k
                testData[counter++] = (t > -10 && t < 50) ? 0 : 1;
            }
        }
    }
    pVolData2->data.setValue(SbVec3i32(testXdim, testYdim, testZdim), pVolData1->getDataType(), 0, testData, SoSFArray::NO_COPY );
    pVolData2->ldmResourceParameters.getValue()->tileDimension.setValue(pVolData1->ldmResourceParameters.getValue()->tileDimension.getValue());
#endif
    // Mask volume must have a unique dataSetId and same extent as data volume
    pVolData2->dataSetId = 2;
    pVolData2->extent.setValue( size );
    mds->addChild( pVolData2 );

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
      pTransFunc->predefColorMap = SoTransferFunction::GREY;
      pTransFunc->minValue = 40;
      pTransFunc->maxValue = 250;
      pTransFunc->transferFunctionId = 0;
      mds->addChild( pTransFunc );

    SoTransferFunction* pTransFunc1 = new SoTransferFunction;
      pTransFunc1->predefColorMap = SoTransferFunction::SEISMIC;
      pTransFunc1->transferFunctionId = 1;
#ifndef LIGHTING_AND_PREINTEGRATION
      mds->addChild( pTransFunc1 );
#endif

    // Initialize and set the volume shader program
    SoVolumeRenderingQuality* pVolShader = new SoVolumeRenderingQuality;
    
    // Specify how to combine data with each other
    SoFragmentShader* fragmentShader = new SoFragmentShader;

    // Data1 in texture unit 1
    fragmentShader->addShaderParameter1i("data1",1);
    // Data2 in texture unit 2
    fragmentShader->addShaderParameter1i("data2",2);

    // Specify to what part the shader prog is intended to. Here fragments only.
#ifdef LIGHTING_AND_PREINTEGRATION
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/Medical/Analysis/medicalGPUDataCompose/GPUDataCompose_combine_frag.glsl");
    pVolShader->shaderObject.set1Value(SoVolumeShader::DATA_COMBINE_FUNCTION, fragmentShader);
    pVolShader->preIntegrated = TRUE;
    pVolShader->deferredLighting=TRUE;
    pVolShader->ambientOcclusion=TRUE;
#else
    // TF in texture unit 0
    fragmentShader->sourceProgram.setValue("$OIVHOME/examples/source/Medical/Analysis/medicalGPUDataCompose/GPUDataCompose_volume_frag.glsl");
    pVolShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
#endif
    mds->addChild( pVolShader );

    // Node in charge of drawing the volume
    SoVolumeRender* pVolRender = new SoVolumeRender;
      pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
      pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      mds->addChild( pVolRender );

    // Draw volume bounding box
    root->addChild( MedicalHelper::createBoundingBox( size ) );

    // Define OIV logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation marker
    root->addChild( new Gnomon() );
    
    // Set up viewer:
    SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);
    viewer->setSceneGraph(root.ptr());
    viewer->setTitle("GPU Data Masking");
    viewer->setDecoration(FALSE);
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, viewer->getCamera() );
    viewer->viewAll();
    viewer->saveHomePosition();
    viewer->show();

    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;

    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}


