///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the getDAta API from VolumeViz
author : Jerome Hummel
January 2004
Updaded by Pascal Estrade (Sep 2014)
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMResourceManager.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/draggers/SoJackDragger.h>

#include <Inventor/algorithms/SoArithmetic.h>
#include <Inventor/algorithms/SoConversion.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

///////////////////////////////////////////////////////////////////////////////

bool toggleViewData = true;
bool toggleInterpolation = true;
int  resolution = 0;

SoXtExaminerViewer* myViewer2 = NULL;
SoSeparator*            root2 = NULL;
SoText2*                 menu = NULL;
SoVolumeData*        pVolData = NULL;
SoROIManip*         pROIManip = NULL;
SoJackDragger*   planeDragger = NULL;

SoArithmetic*      arithmetic = NULL;
SoConversion*       converter = NULL;

///////////////////////////////////////////////////////////////////////////////
void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle data display
  if (SO_KEY_PRESS_EVENT(event, D)) {
    toggleViewData = !toggleViewData;
  }

  // Toggle interpolation
  else if (SO_KEY_PRESS_EVENT(event, I)) {
    toggleInterpolation = !toggleInterpolation;
  }

  // increase resolution
  else if (SO_KEY_PRESS_EVENT(event, UP_ARROW)) {
    if (resolution)
      resolution--;
  }

  // decrease resolution
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) {
    resolution++;
  }

  // YZ plane
  else if (SO_KEY_PRESS_EVENT(event, X)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,0,1), 1.570796325f );
  }

  // XZ plane
  else if (SO_KEY_PRESS_EVENT(event, Y)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,1,0), 1.570796325f );
  }

  // XY plane
  else if (SO_KEY_PRESS_EVENT(event, Z)) {
    planeDragger->rotation = SbRotation( SbVec3f(1,0,0), 1.570796325f );
  }

  // Change Data Access Mode
  else if (SO_KEY_PRESS_EVENT(event, M))
  {
    int mode = pVolData->getLdmDataAccess().getGetDataMode();
    mode = (mode + 1)%3;
    pVolData->getLdmDataAccess().setGetDataMode(SoLDMDataAccess::GetDataMode(mode));
  }
  else {
    return;
  }

  pROIManip->touch(); // to force viewer2 to be updated
}

///////////////////////////////////////////////////////////////////////////////
void
updateView2( void* /*userData*/ )
{
  SbVec3i32 dim = pVolData->data.getSize();

  SbPlane plane;
  { // get plane
    SbVec3f planePosition = planeDragger->translation.getValue();
    // rotate the plane's normal by the dragger rotation
    SbRotation rotation = planeDragger->rotation.getValue();
    SbVec3f planeNormal;
    rotation.multVec( SbVec3f(0,1,0), planeNormal );
    plane = SbPlane( planeNormal, planePosition );

    // plane equation in data coordinates
    SbMatrix mx;
    mx.makeIdentity();
    SbVec3f volmin, volmax;
    pVolData->extent.getValue().getBounds( volmin, volmax );
    mx[0][0] = dim[0]/(volmax[0]-volmin[0]); mx[3][0] = -mx[0][0]*volmin[0];
    mx[1][1] = dim[1]/(volmax[1]-volmin[1]); mx[3][1] = -mx[1][1]*volmin[1];
    mx[2][2] = dim[2]/(volmax[2]-volmin[2]); mx[3][2] = -mx[2][2]*volmin[2];
    plane.transform(mx);
  }

  SbBox3i32 box(pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());
  SoVolumeData::LDMDataAccess::DataInfoPlane info;

  // First call to getData with no buffer to retrieve needed information
  info = pVolData->getLdmDataAccess().getData( resolution, box, plane );

  bool firstTime = (root2 == NULL);
  if (firstTime) {

    SoSeparator *root = new SoSeparator;

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( eventCB );

    SoLightModel *lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    root->addChild( lightModel );

    { // menu
      SoFont *font = new SoFont;
      font->name = "Courier New";
      font->size = 12;
      root->addChild( font);

      SoBaseColor *menuColor = new SoBaseColor;
      menuColor->rgb.setValue( 1, 1, 0);
      root->addChild( menuColor );

      SoTranslation *menuPos = new SoTranslation;
      menuPos->translation.setValue( -0.95f, 0.95f, 0. );
      root->addChild( menuPos );

      menu = new SoText2;
      menu->string.set1Value( 0, "D   : Toggle data" );
      menu->string.set1Value( 1, "I   : Toggle interpolation" );
      menu->string.set1Value( 2, "UP  : Increase resolution" );
      menu->string.set1Value( 3, "DOWN: Decrease resolution" );
      menu->string.set1Value( 4, "XYZ : Axis alignment" );
      menu->string.set1Value( 5, "M       : Toggle DataAccess mode" );
      root->addChild( menu );
    }

    root->addChild( new SoPerspectiveCamera );

    root2 = new SoSeparator;
    root->addChild( root2 );

    myViewer2->setSceneGraph( root );
  }

  root2->removeAllChildren();

  switch (pVolData->getLdmDataAccess().getGetDataMode() )
  {
  case SoLDMDataAccess::CACHE:
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (CACHE)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(20);
    break;
  case SoLDMDataAccess::DIRECT:
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (DIRECT)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  case SoLDMDataAccess::DIRECT_AND_PREFETCH:
    menu->string.set1Value( 5, "M       : Toggle DataAccess mode (DIRECT_AND_PREFETCH" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  }


  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  root2->addChild( lineStyle );

  SoSeparator *volSep = new SoSeparator;
  root2->addChild( volSep );
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(dim[0]/2.), (float)(dim[1]/2.), (float)(dim[2]/2.) );
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = (float)dim[0];
    vol->height = (float)dim[1];
    vol->depth  = (float)dim[2];
    volSep->addChild( vol );

  SoSeparator *roiSep = new SoSeparator;
  root2->addChild( roiSep );
  {
    SbVec3i32 roiMin = box.getMin();
    SbVec3i32 roiDim = box.getMax() - roiMin + SbVec3i32(1,1,1);

    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(roiMin[0] + roiDim[0]/2.), (float)(roiMin[1] + roiDim[1]/2.), (float)(roiMin[2] + roiDim[2]/2.) );
    roiSep->addChild( volTr );

    SoBaseColor *roiColor = new SoBaseColor;
    roiColor->rgb.setValue( 1, 0, 0 );
    roiSep->addChild( roiColor );

    SoCube *roi = new SoCube;
    roi->width  = (float)roiDim[0];
    roi->height = (float)roiDim[1];
    roi->depth  = (float)roiDim[2];
    roiSep->addChild( roi );
  }

  if (info.numPolygonPoints > 0) {
    int i;
    SbVec3f vertices[6];

    SoBaseColor *quadColor = new SoBaseColor;
    quadColor->rgb.setValue( 0.3f, 0.7f, 0.5f );
    root2->addChild( quadColor );

    SoFaceSet *quad = new SoFaceSet;
    for (i = 0; i < 4; i++) {
      vertices[i][0] = (float)info.quadCoord[i][0];
      vertices[i][1] = (float)info.quadCoord[i][1];
      vertices[i][2] = (float)info.quadCoord[i][2];
    }
    SoVertexProperty *vp = new SoVertexProperty;
    vp->vertex.setValues(0, 4, vertices);
    quad->vertexProperty = vp;
    quad->numVertices.set1Value( 0, 4 );
    root2->addChild( quad );

    SoBaseColor *polygonColor = new SoBaseColor;
    polygonColor->rgb.setValue( 1.0f, 0.8f, 0.6f );
    root2->addChild( polygonColor );

    SoDrawStyle *lineWidthStyle = new SoDrawStyle;
    lineWidthStyle->style = SoDrawStyle::LINES;
    lineWidthStyle->lineWidth = 5.f;
    root2->addChild( lineWidthStyle );

    SoFaceSet *polygon = new SoFaceSet;
    for (i = 0; i < info.numPolygonPoints; i++) {
      vertices[i][0] = (float)info.polygonCoord[i][0];
      vertices[i][1] = (float)info.polygonCoord[i][1];
      vertices[i][2] = (float)info.polygonCoord[i][2];
    }
    vp = new SoVertexProperty;
    vp->vertex.setValues(0, info.numPolygonPoints, vertices);
    polygon->vertexProperty = vp;
    polygon->numVertices.set1Value( 0, info.numPolygonPoints );
    root2->addChild( polygon );
  }

  if (toggleViewData && info.bufferSize > 0) 
  {
    SoComplexity *complexity = new SoComplexity;
    complexity->textureQuality = toggleInterpolation ? 1 : 0.1f;
    root2->addChild( complexity );

    SoTexture2 *tex2 = new SoTexture2;

    SoRef<SoBufferObject> bufferObject=NULL;

      SoRef<SoCpuBufferObject> cpuBufferObject = new SoCpuBufferObject;
      cpuBufferObject->setSize(info.bufferSize);
      info = pVolData->getLdmDataAccess().getData( resolution, box, plane, cpuBufferObject.ptr() );
      bufferObject = cpuBufferObject.ptr();

    if (info.errorFlag == SoVolumeData::LDMDataAccess::CORRECT)
    {
      SoBufferObject *resultBufferObject = bufferObject.ptr();

      // convert data to byte if needed.
      int dataSize = pVolData->getDataSize();
      if (pVolData->getDataType() != SoVolumeData::UNSIGNED_BYTE)
      {
        int64_t n = info.bufferSize / dataSize;
        double vmin,vmax;        
        pVolData->getMinMax(vmin,vmax);

        // scale data to get range in byte
        arithmetic->scale(
          bufferObject.ptr(),pVolData->getDataType(),
          bufferObject.ptr(),pVolData->getDataType(),
          (float)((vmax-vmin)/256.0)
          );

        // shift data to get min to 0
        arithmetic->shift(
          bufferObject.ptr(),pVolData->getDataType(),
          bufferObject.ptr(),pVolData->getDataType(),
          (float)(-vmin)
          );

        // convert data to byte
        resultBufferObject = bufferObject->createInstance();
        resultBufferObject->setSize(n*sizeof(unsigned char));
        converter->convert(
          bufferObject.ptr(),pVolData->getDataType(),
          resultBufferObject,SbDataType::UNSIGNED_BYTE,
          n
        );
      }
      tex2->image.setValue( 
        info.bufferDimension, 1, SoSFImage::UNSIGNED_BYTE,
        resultBufferObject, SoSFImage::NO_COPY_AND_DELETE 
        );
    }
    root2->addChild( tex2 );

    SoBaseColor *texColor = new SoBaseColor;
    texColor->rgb.setValue( 1.0f, 1.0f, 1.0f );
    root2->addChild( texColor );

    SoDrawStyle *texStyle = new SoDrawStyle;
    texStyle->style = SoDrawStyle::FILLED;
    texStyle->lineWidth = 5.f;
    root2->addChild( texStyle );

    SoFaceSet *texpolygon = new SoFaceSet;
    SbVec3i32 *quad = info.quadCoord;
    SbVec2f u( (float)(quad[1][info.uAxis]-quad[0][info.uAxis]), (float)(quad[1][info.vAxis]-quad[0][info.vAxis]) );
    SbVec2f v( (float)(quad[2][info.uAxis]-quad[1][info.uAxis]), (float)(quad[2][info.vAxis]-quad[1][info.vAxis]) );
    u /= u.dot(u);
    v /= v.dot(v);
    SbVec3f vertices[6];
    SbVec2f texcoord[6];
    for (int i = 0; i < info.numPolygonPoints; i++) {
      vertices[i][0] = (float)info.polygonCoord[i][0];
      vertices[i][1] = (float)info.polygonCoord[i][1];
      vertices[i][2] = (float)info.polygonCoord[i][2];
      SbVec2f uv( vertices[i][info.uAxis]-quad[0][info.uAxis], vertices[i][info.vAxis]-quad[0][info.vAxis] );
      texcoord[i][0] = uv.dot(u);
      texcoord[i][1] = uv.dot(v);
    }
    SoVertexProperty *vp = new SoVertexProperty;
    vp->vertex.setValues( 0, info.numPolygonPoints, vertices );
    vp->texCoord.setValues( 0, info.numPolygonPoints, texcoord );
    texpolygon->vertexProperty = vp;
    texpolygon->numVertices.set1Value( 0, info.numPolygonPoints );
    root2->addChild( texpolygon );
  }

  if (firstTime) {
    myViewer2->viewAll();
    firstTime = false;
  }
}

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  SoVolumeRendering::init();
  InventorMedical::init();

  // Initialize Inventor Arithmetic module
  arithmetic = new SoArithmetic();

  // Initialize Inventor Conversion module
  converter = new SoConversion();

  // Node to hold the volume data
  pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/Medical/files/3DHEAD.ldm";
  //getting data is faster in ALWAYS mode
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Plane manipulator
  planeDragger = new SoJackDragger;
  planeDragger->rotation = SbRotation( SbVec3f(1,0,0), 1.570796325f );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;

  // VR Quality
  SoVolumeRenderingQuality *volQuality = new SoVolumeRenderingQuality();
  volQuality->deferredLighting = TRUE;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
  SbVec3i32 dimensions = pVolData->data.getSize();

  pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = TRUE;
  pROIManip->boxOn = FALSE;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
  pVolRender->interpolation = SoVolumeRender::CUBIC;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoRef<SoSeparator> root = new SoSeparator;
  root->addChild( eventCB );
  root->addChild( planeDragger );
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  root->addChild( volQuality );
  root->addChild( pVolRender );

  // Define OIV logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(myWindow);
  myViewer1->setSceneGraph(root.ptr());
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("GetData( plane )");
  myViewer2->setDecoration(FALSE);
  myViewer2->show();

  // feed backs
  SoNodeSensor *roiSensor = new SoNodeSensor( (SoSensorCB *)&updateView2, NULL );
  roiSensor->attach( pROIManip );
  roiSensor->trigger(); // force an update for the first frame

  planeDragger->addMotionCallback( (SoDraggerCB *)&updateView2, NULL );

  //
  SoXt::show(myWindow);
  SoXt::mainLoop();

  root = NULL;
  delete myViewer1;
  delete myViewer2;
  delete converter;
  delete arithmetic;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


