///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use the getData API from VolumeViz
author : Jerome Hummel
January 2004
Updaded by Pascal Estrade (Sep 2014)
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/manips/SoROIManip.h>
#include <LDM/SoLDMResourceManager.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoVertexProperty.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/STL/vector>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

///////////////////////////////////////////////////////////////////////////////

using std::vector;

bool  toggleViewData = true;
int   resolution = 0;
float amplitude = 1.0f;
SbVec3i32* myPoints = NULL;

SoXtExaminerViewer* myViewer2 = NULL;
SoSeparator*           root2  = NULL;
SoVolumeData*        pVolData = NULL;
SoROIManip*         pROIManip = NULL;
SoJackDragger*   planeDragger = NULL;
SoText2 *                menu = NULL;
vector<SbVec3i32> points;

#define NPOINTSY 10

////////////////////////////////////////////////////////////////////////
void
EventCB (void *, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Toggle data display
  if (SO_KEY_PRESS_EVENT(event, D)) {
    toggleViewData = !toggleViewData;
  }

  // increase resolution
  else if (SO_KEY_PRESS_EVENT(event, UP_ARROW)) {
    if (resolution)
      resolution--;
  }

  // decrease resolution
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) {
    resolution++;
  }

  // increase amplitude
  else if (SO_KEY_PRESS_EVENT(event, PAGE_UP)) {
    amplitude *= 1.3f;
  }

  // decrease amplitude
  else if (SO_KEY_PRESS_EVENT(event, PAGE_DOWN)) {
    amplitude /= 1.3f;
  }

  // YZ plane
  else if (SO_KEY_PRESS_EVENT(event, X)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,0,1), 1.570796325f );
  }

  // XZ plane
  else if (SO_KEY_PRESS_EVENT(event, Y)) {
    planeDragger->rotation = SbRotation( SbVec3f(0,1,0), 1.570796325f );
  }

  // XY plane
  else if (SO_KEY_PRESS_EVENT(event, Z)) {
    planeDragger->rotation = SbRotation( SbVec3f(1,0,0), 1.570796325f );
  }
  else if (SO_KEY_PRESS_EVENT(event, M))
  {
    int mode = pVolData->getLdmDataAccess().getGetDataMode();
    mode = (mode + 1)%3;
    pVolData->getLdmDataAccess().setGetDataMode(SoLDMDataAccess::GetDataMode(mode));
  }
  else {
    return;
  }

  pROIManip->touch(); // to force viewer2 to be updated
}

////////////////////////////////////////////////////////////////////////
//
void addData( SbBox3i32 box, SoVolumeData::LDMDataAccess::DataInfoPolyLine& info,
              int numPoints, SbVec3i32* points, SoSeparator* root2)
{
  SoBaseColor *lineColor = new SoBaseColor;
  lineColor->rgb.setValue( 0.3f, 0.7f, 0.5f );
  root2->addChild( lineColor );

  if (info.bufferSize<=0)
    return;

  SoRef<SoCpuBufferObject> cpuBufferObject=new SoCpuBufferObject();
  cpuBufferObject->setSize(info.bufferSize);

  SoVolumeData::LDMDataAccess::DataInfoPolyLine newInfo(numPoints);
  pVolData->getLdmDataAccess().getData( newInfo, resolution, box, numPoints, points, cpuBufferObject.ptr() );

  SbVec3f vsegment[2];
  if( numPoints && toggleViewData )
  {
    int offset = 0;
    int segmentCount = 0;
    for(int k = 0; k < numPoints-1; k++){

      if(newInfo.segmentBufferDim[k] == 0)
        continue;

      vsegment[0] = SbVec3f( (float)(newInfo.segmentCoord[segmentCount][0][0]), (float)(newInfo.segmentCoord[segmentCount][0][1]), (float)(newInfo.segmentCoord[segmentCount][0][2]));
      vsegment[1] = SbVec3f( (float)(newInfo.segmentCoord[segmentCount][1][0]), (float)(newInfo.segmentCoord[segmentCount][1][1]), (float)(newInfo.segmentCoord[segmentCount][1][2]));

      SbVec3f dsegment = vsegment[1] - vsegment[0];

      SbVec3f axis = dsegment.getClosestAxis();
      SbVec3f upvector;
      if (axis[0] != 0) {
        if (dsegment[0] > 0)
          upvector.setValue( -dsegment[1], dsegment[0], 0 );
        else
          upvector.setValue( dsegment[1], -dsegment[0], 0 );
      }
      else if (axis[1] != 0) {
        if (dsegment[1] > 0)
          upvector.setValue( dsegment[1], -dsegment[0], 0 );
        else
          upvector.setValue( -dsegment[1], dsegment[0], 0 );
      }
      else {
        if (dsegment[2] > 0)
          upvector.setValue( 0, dsegment[2], -dsegment[1] );
        else
          upvector.setValue( 0, -dsegment[2], dsegment[1] );
      }
      upvector.normalize();

      SoLineSet *curve = new SoLineSet;
      SoVertexProperty *vp = new SoVertexProperty;

      int n = newInfo.segmentBufferDim[k];
      int i ;
      if(segmentCount > 0 &&
        newInfo.segmentCoord[segmentCount-1][1] == newInfo.segmentCoord[segmentCount][0]){
        offset --; //last byte of previous seg = first byte of current seg (end = start)
      }
      if (n>0)
      {
        void * buffer = cpuBufferObject->map(SoCpuBufferObject::READ_ONLY);
        for (i = 0; i < n; i++) {
          double value = 0.;
          switch (pVolData->getDataType()) {
          case SoVolumeData::UNSIGNED_BYTE:  value = (double)(((unsigned char  *)buffer)[i+offset]) /       0xff; break;
          case SoVolumeData::UNSIGNED_SHORT: value = (double)(((unsigned short *)buffer)[i+offset]) /     0xffff; break;
          case SoVolumeData::UNSIGNED_INT32: value = (double)(((unsigned int   *)buffer)[i+offset]) / 0xffffffff; break;
          case SoVolumeData::SIGNED_BYTE:  value = (double)(((char  *)buffer)[i+offset]) /       0xff + 0.5; break;
          case SoVolumeData::SIGNED_SHORT: value = (double)(((short *)buffer)[i+offset]) /     0xffff + 0.5; break;
          case SoVolumeData::SIGNED_INT32: value = (double)(((int   *)buffer)[i+offset]) / 0xffffffff + 0.5; break;
          case SoVolumeData::FLOAT: value = (double)(((float *)buffer)[i+offset]) / 50000. + 0.5; break;
          case SoVolumeData::DOUBLE: value = (((double *)buffer)[i+offset]) / 50000. + 0.5; break;
          }
          float u = (float) i / (n-1);
          float v = 5.f + amplitude/10.f * (float)value * 250.f;
          SbVec3f p = vsegment[0] +  u * dsegment + v * upvector;
          vp->vertex.set1Value( i, p );
        }
        cpuBufferObject->unmap();
      }
      curve->vertexProperty = vp;
      curve->numVertices.set1Value( 0, n );

      root2->addChild( curve );

      offset += n;
      segmentCount++;
    }
  }
}


////////////////////////////////////////////////////////////////////////
//
void addPolyLine(SoVolumeData::LDMDataAccess::DataInfoPolyLine* info, SoSeparator* root2){

  SoBaseColor *lineColor = new SoBaseColor;
  lineColor->rgb.setValue( 0.3f, 0.7f, 0.0f );
  root2->addChild( lineColor );

  SoDrawStyle *segmentWidthStyle = new SoDrawStyle;
  segmentWidthStyle->style = SoDrawStyle::LINES;
  segmentWidthStyle->lineWidth = 4.f;
  root2->addChild( segmentWidthStyle );

  SbVec3f vsegment[2];
  if(info->numSegment){

    int segmentCount = 0;
    for(int k = 0; k < info->numSegment; k++){

      if(info->segmentBufferDim[k] == 0)
        continue;

      vsegment[0] = SbVec3f((float)(info->segmentCoord[segmentCount][0][0]), (float)(info->segmentCoord[segmentCount][0][1]), (float)(info->segmentCoord[segmentCount][0][2]) );
      vsegment[1] = SbVec3f((float)(info->segmentCoord[segmentCount][1][0]), (float)(info->segmentCoord[segmentCount][1][1]), (float)(info->segmentCoord[segmentCount][1][2]) );

      SoLineSet *segment = new SoLineSet;
      SoVertexProperty *vp = new SoVertexProperty;
      vp->vertex.setValues( 0, 2, vsegment );
      segment->vertexProperty = vp;
      segment->numVertices.set1Value( 0, 2 );
      root2->addChild( segment );

      segmentCount ++;
    }
  }

  SoBaseColor *lineColor2 = new SoBaseColor;
  lineColor->rgb.setValue( 0.0f, 0.5f, 0.5f );
  root2->addChild( lineColor2 );
  SoDrawStyle *segmentWidthStyle2 = new SoDrawStyle;
  segmentWidthStyle2->style = SoDrawStyle::LINES;
  segmentWidthStyle2->lineWidth = 1.f;
  root2->addChild( segmentWidthStyle2 );

  SbVec3f temp= SbVec3f((float)points[0][0],(float)points[0][1],(float)points[0][2]) ;;

  for(unsigned int k=1; k < points.size(); k++){
    vsegment[0] = temp;
    vsegment[1] = SbVec3f((float)points[k][0],(float)points[k][1],(float)points[k][2]) ;

    SoLineSet *segment = new SoLineSet;
    SoVertexProperty *vp = new SoVertexProperty;
    vp->vertex.setValues( 0, 2, vsegment );
    segment->vertexProperty = vp;
    segment->numVertices.set1Value( 0, 2 );
    root2->addChild( segment );
    temp = SbVec3f((float)points[k][0],(float)points[k][1],(float)points[k][2]) ;
  }

  SoDrawStyle *lineWidthStyle = new SoDrawStyle;
  lineWidthStyle->style = SoDrawStyle::LINES;
  root2->addChild( lineWidthStyle );
}

////////////////////////////////////////////////////////////////////////
//
void
updateView2( void* /*userData*/ )
{
  unsigned int i;
  static int first = 1;
  static vector<SbVec3f> vect;
  static SbVec3f initDraggerPos;
  
  SbVec3i32 dim = pVolData->data.getSize();
  SbVec3f draggerPos = planeDragger->translation.getValue();
  if(points.size() == 0){

    // create stack of points
    int border = dim[0]/20;
    int x = dim[0]-border;
    int y = dim[1]-border;
    int z = dim[2]-border;

    SbVec3i32 point0(border,0,z/2);
    //printf("[%d, %d]\n",point0[0],point0[1]);
    points.push_back(point0);

    int step = y/NPOINTSY;
    if(step == 0) step = 1;
    for(i=1; i < 2*NPOINTSY; i++){
      SbVec3i32 point0 = points[points.size()-1];
      SbVec3i32 point1 = point0;

      //increment in X
      if(i%2 != 0){
        if(point1[0] == border)
          point1[0] = x - border;
        else
          point1[0] = border;
      }
      //increment in Y
      else{
        point1[1] += step;
      }
      points.push_back(point1);
    }

    //remember original vect only first time
    if(first){
      first = 0;
      initDraggerPos = draggerPos;
      for(unsigned int k=0; k < points.size(); k++){
        SbVec3f point0 = pVolData->getLdmDataAccess().voxelToXYZ(points[k]);
        vect.push_back(point0 - draggerPos);
      }
    }
  }
#define PIO2 1.570796326794896619231f
    vector<SbVec3i32> temp;
    for(i = 0 ; i < points.size(); i++){
      pVolData->getLdmDataAccess().voxelToXYZ(points[i]);//world coord

      SbVec3f vectTemp = vect[i];//original vector going from dragger pos to point

      SbRotation rot(SbVec3f(0,0,1),-PIO2);
      SbRotation rotation = planeDragger->rotation.getValue();
      rotation =  rotation*rot; //cancel original dragger rotation
      SbVec3f lineDirection;
      rotation.multVec( vectTemp, lineDirection );//rotate original vect by dragger's rotation

      SbVec3f diff = draggerPos - initDraggerPos;
      SbVec3f final = /*draggerPos +*/ lineDirection + diff;//rotated point in world

      SbVec3i32 finals = pVolData->getLdmDataAccess().XYZToVoxel(final);//rotated point in data
      temp.push_back(finals);
    }
#undef PIO2

  points = temp;
  bool firstTime = (root2 == NULL);
  if (firstTime) {

    SoSeparator *root = new SoSeparator;
    root->ref();

    SoEventCallback *eventCB = new SoEventCallback;
    eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( eventCB );

    SoLightModel *lightModel = new SoLightModel;
    lightModel->model = SoLightModel::BASE_COLOR;
    root->addChild( lightModel );

    { // menu
      SoFont *font = new SoFont;
      font->name = "Courier New";
      font->size = 12;
      root->addChild( font);

      SoBaseColor *menuColor = new SoBaseColor;
      menuColor->rgb.setValue( 1, 1, 0);
      root->addChild( menuColor );

      SoTranslation *menuPos = new SoTranslation;
      menuPos->translation.setValue( -0.95f, 0.95f, 0. );
      root->addChild( menuPos );

      menu = new SoText2;
      menu->string.set1Value( 0, "D       : Toggle data" );
      menu->string.set1Value( 1, "UP      : Increase resolution" );
      menu->string.set1Value( 2, "DOWN    : Decrease resolution" );
      menu->string.set1Value( 3, "PAGEUP  : Increase amplitude" );
      menu->string.set1Value( 4, "PAGEDOWN: Decrease amplitude" );
      menu->string.set1Value( 5, "XYZ     : Axis alignment" );
      menu->string.set1Value( 6, "M       : Toggle DataAccess mode" );
      root->addChild( menu );
    }

    root->addChild( new SoPerspectiveCamera );

    root2 = new SoSeparator;
    root->addChild( root2 );

    myViewer2->setSceneGraph( root );
  }

  root2->removeAllChildren();

  switch (pVolData->getLdmDataAccess().getGetDataMode() )
  {
  case SoLDMDataAccess::CACHE:
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (CACHE)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(20);
    break;
  case SoLDMDataAccess::DIRECT:
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (DIRECT)" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  case SoLDMDataAccess::DIRECT_AND_PREFETCH:
    menu->string.set1Value( 6, "M       : Toggle DataAccess mode (DIRECT_AND_PREFETCH" );
    pVolData->ldmResourceParameters.getValue()->maxMainMemory.setValue(0);
    break;
  }

  SoDrawStyle *lineStyle = new SoDrawStyle;
  lineStyle->style = SoDrawStyle::LINES;
  root2->addChild( lineStyle );

  SoSeparator *volSep = new SoSeparator;
  root2->addChild( volSep );
  {
    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(dim[0]/2.), (float)(dim[1]/2.), (float)(dim[2]/2.) );
    volSep->addChild( volTr );

    SoBaseColor *volColor = new SoBaseColor;
    volColor->rgb.setValue( 0.9f, 0.9f, 0.9f );
    volSep->addChild( volColor );

    SoCube *vol = new SoCube;
    vol->width  = (float)dim[0];
    vol->height = (float)dim[1];
    vol->depth  = (float)dim[2];
    volSep->addChild( vol );
  }

  SbBox3i32 box(pROIManip->subVolume.getValue().getMin(), pROIManip->subVolume.getValue().getMax());
  SoSeparator *roiSep = new SoSeparator;
  root2->addChild( roiSep );
  {
    SbVec3i32 roiMin = box.getMin();
    SbVec3i32 roiDim = box.getMax() - roiMin + SbVec3i32(1,1,1);

    SoTranslation *volTr = new SoTranslation;
    volTr->translation = SbVec3f( (float)(roiMin[0] + roiDim[0]/2.), (float)(roiMin[1] + roiDim[1]/2.), (float)(roiMin[2] + roiDim[2]/2.) );
    roiSep->addChild( volTr );

    SoBaseColor *roiColor = new SoBaseColor;
    roiColor->rgb.setValue( 1, 0, 0 );
    roiSep->addChild( roiColor );

    SoCube *roi = new SoCube;
    roi->width  = (float)roiDim[0];
    roi->height = (float)roiDim[1];
    roi->depth  = (float)roiDim[2];
    roiSep->addChild( roi );
  }

  size_t numPoints = points.size();
  if( myPoints )
    delete [] myPoints;

  myPoints = new SbVec3i32[numPoints];
  for(i=0;i < numPoints;i++){
    myPoints[i] = points[i];
  }

  //ask for buffer size
  SoVolumeData::LDMDataAccess::DataInfoPolyLine infoPolyline((int)numPoints);

  // First call to getData with no buffer to retrieve needed information
  pVolData->getLdmDataAccess().getData( infoPolyline, resolution, box, (int)numPoints, myPoints );

  //draw geometry
  addPolyLine(&infoPolyline, root2);

  //ask for data
  addData(box, infoPolyline, (int)numPoints, myPoints, root2);

  if (firstTime) {
    myViewer2->viewAll();
    firstTime = false;
  }
}

////////////////////////////////////////////////////////////////////////
//
// main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  SoVolumeRendering::init();
  InventorMedical::init();

  // Node to hold the volume data
  pVolData = new SoVolumeData();
  pVolData->fileName = "$OIVHOME/examples/data/Medical/files/3DHEAD.ldm";
  //getting data is faster in ALWAYS mode
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoVolumeData::SoLDMResourceParameters::ALWAYS;
  pVolData->ldmResourceParameters.getValue()->tileHalfLife = 0;

  // Track the keyboard events
  SoEventCallback *eventCB = new SoEventCallback;
  eventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );

  // Plane manipulator
  planeDragger = new SoJackDragger;
  planeDragger->rotation = SbRotation( SbVec3f(0,0,1), 1.570796325f );

  // Use a predefined colorMap with the SoTransferFunction
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::SEISMIC;
  pTransFunc->minValue = 15;
  pTransFunc->maxValue = 255;

  // VR Quality
  SoVolumeRenderingQuality *volQuality = new SoVolumeRenderingQuality();
  volQuality->deferredLighting = TRUE;

  // ROI / ROIManip
  // Initialize both ROI box and subvolume to be the entire volume.
  // Constrain the ROIManip to stay inside the volume.
  SbVec3i32 dimensions = pVolData->data.getSize();

  pROIManip = new SoROIManip();
  pROIManip->box.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->subVolume.setValue( SbVec3i32(0,0,0), dimensions - SbVec3i32(1,1,1) );
  pROIManip->constrained = FALSE;
  pROIManip->boxOn = FALSE;

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
  pVolRender->interpolation = SoVolumeRender::CUBIC;

  // Assemble the scene graph
  // Note: SoVolumeRender must appear after the SoVolumeData node.
  SoRef<SoSeparator> root = new SoSeparator;
  root->addChild( eventCB );
  root->addChild( planeDragger );
  root->addChild( pVolData );
  root->addChild( pROIManip );
  root->addChild( pTransFunc );
  root->addChild( volQuality );
  root->addChild( pVolRender );

  // Define OIV logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Set up viewer 1:
  SoXtExaminerViewer *myViewer1 = new SoXtExaminerViewer(myWindow);
  myViewer1->setSceneGraph(root.ptr());
  myViewer1->setTitle("Volume rendering");
  myViewer1->show();

  // Set up viewer 2:
  myViewer2 = new SoXtExaminerViewer(myWindow, "", FALSE);
  myViewer2->setTitle("GetData( polyline )");
  myViewer2->setDecoration(FALSE);
  myViewer2->show();

  // feed backs
  SoNodeSensor *roiSensor = new SoNodeSensor( (SoSensorCB *)&updateView2, NULL );
  roiSensor->attach( pROIManip );
  roiSensor->trigger(); // force an update for the first frame

  planeDragger->addMotionCallback( (SoDraggerCB *)&updateView2, NULL );

  //
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer1;
  delete myViewer2;
  root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}


