///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/engines/SoTimeCounter.h>
#include <Inventor/sensors/SoNodeSensor.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <DialogViz/dialog/SoMessageDialog.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/DicomInfo.h>

///////////////////////////////////////////////////////////////////////////////

#define DEFAULT_SCENE "$OIVHOME/examples/source/Medical/Dental/medicalDentalCurveUnfolding/dentalCurveUnfolding.iv"
#define DATA_PATH     "$OIVHOME/examples/data/Medical/files/"

SoXtExaminerViewer*       _viewer;
SoSeparator*              _root;
SoVolumeRenderingQuality* _unfoldingPosition;
SoTimeCounter*            _translationTimer;

void myKeyPressCB(void *userData, SoEventCallback *eventCB);

SbBool postRenderCallback(void*, SoXtRenderArea*);

///////////////////////////////////////////////////////////////////////////////
int
main(int /*argc*/, char **argv)
{
  // Initialize all needed modules
  Widget myWindow = SoXt::init(argv[0]); 
  SoVolumeRendering::init();
  InventorMedical::init();

  // First of all read the scene
  SbString filename = DEFAULT_SCENE;

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    std::cerr << "Cannot open " << filename.toLatin1() << std::endl;
    new SoMessageDialog( filename.toLatin1(), "Unable to open:", SoMessageDialog::MD_ERROR );
    return -1;
  }

  std::cout << "Press 'U' to start/stop unfolding animation.\n";

  // Create the scene root
  _root = new SoSeparator();

  // Create event callback for key press event
  SoEventCallback* activateClippingCB = new SoEventCallback();
    activateClippingCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), myKeyPressCB );
    _root->addChild(activateClippingCB);

  // Manually create a camera to view the scene.
  // Because Gnomon contains a camera, SoXtExaminerViewer will not automatically insert a camera in sceneGraph.
  _root->addChild( new SoOrthographicCamera() );

  // Add Open Inventor logo
  _root->addChild( MedicalHelper::exampleLogoNode() );

  // Scene orientation
  _root->addChild( new Gnomon() );

  // DICOM annotation
  _root->addChild( MedicalHelper::exampleDicomAnnotation("$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm") );

  // Instructions
  TextBox* text = new TextBox();
    text->position.setValue(0, -0.99f, 0); // Normalized device coords -1..1
    text->alignmentH = TextBox::CENTER;
    text->alignmentV = TextBox::BOTTOM;
    text->addLine( "Press 'U' to start and stop the animation" );
    _root->addChild( text );

  //-----------------------------------------------------------------
  // Add light to the scene (after gnomon so we don't overlight it)
  _root->addChild( new SoDirectionalLight() );
            
  // Load scene graph for cutting
  SoSeparator* clipSep = SoDB::readAll(&myInput);
  if (clipSep == NULL) {
    new SoMessageDialog("Cannot load scenegraph file","Error",SoMessageDialog::MD_ERROR);
    return -1;
  }
  _root->addChild(clipSep);

  // Find volume rendering quality node so we modify shader parameter
  _unfoldingPosition = MedicalHelper::find<SoVolumeRenderingQuality>(clipSep);
            
  // Define animation timer 
  _translationTimer = new SoTimeCounter();
    _translationTimer->ref();
    _translationTimer->on = FALSE;

  // Viewer
  _viewer = new SoXtExaminerViewer(myWindow);
    _viewer->setTitle("Dental Unfolding (panorama)");
    _viewer->setDecoration(FALSE);
    _viewer->setSize( MedicalHelper::exampleWindowSize() );
    _viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    _viewer->setSceneGraph( _root );
    _viewer->setPostRenderCallback(postRenderCallback);

  // Adjust camera
  _viewer->getCamera()->orientation.setValue(SbVec3f(1, 0, 0), 0.78f);
  _viewer->viewAll();
  _viewer->saveHomePosition();

  // Run then cleanup
  _viewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  _translationTimer->unref();
  _translationTimer = NULL;
  delete _viewer;
  _root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


///////////////////////////////////////////////////////////////////////////////
void myKeyPressCB(void* /*userData*/, SoEventCallback* eventCB)
{
  const SoEvent *event = eventCB->getEvent();

   // Start/stop unfolding animation
  if (SO_KEY_PRESS_EVENT(event,  U))
  {
    SbBool state = _translationTimer->on.getValue();
    if (state == FALSE) {
      _translationTimer->on = TRUE;

      // To activate node sensor.
      _unfoldingPosition->touch();
    }
    else {
      _translationTimer->on = FALSE;
    }
  } 

  // Stop animation (doesn't work in viewing mode)
  if (SO_KEY_PRESS_EVENT(event,  S)) 
  {
    _translationTimer->on.setValue( FALSE );
  } 
}


///////////////////////////////////////////////////////////////////////////////
SbBool
postRenderCallback(void*, SoXtRenderArea*)
{
  if (_translationTimer->on.getValue() == TRUE) 
  {
    static float index = 0.0f;
    static float increment = 0.2f;
    // To modify 
    (dynamic_cast<SoShaderParameter1f*>((dynamic_cast<SoFragmentShader*>(_unfoldingPosition->shaderObject[2]))->parameter[2]))->value = index;
    
    _viewer->scheduleRedraw();

    index += increment;
    if ( index > 8 ) increment = -0.2f;
    if( index < 0 ) increment = 0.2f;
  }

  return FALSE;
}
