///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

//!oiv_include <VolumeViz/vvizfnc_frag.h>
//!oiv_include <VolumeViz/vvizGetData_frag.h>
//!oiv_include <VolumeViz/vvizTransferFunction_frag.h>
//!oiv_include <VolumeViz/vvizStructure.h>

#define SPLINE_SIZE 50

// WARNING: This is a proof of concept. It use some hard-coded values and code might
// not be bug free.

uniform vec3[SPLINE_SIZE] spline;
uniform vec3 splineNormal;

// variable used in medicalDentalCurveUnfolding code to change the position of the jaw unfolding.
uniform float unfoldPosition;

// Convert from spline space to texture space.
// spline space: 
// - x along spline
// - z normal to spline plane
// - y along spline plane (y = cross (z, x))
// (0, 0, 0) represent the 1st point of spline, (1, 0, 0) is the last point of spline.
// (0, 0, 1) is in op of 1st point of spline.
vec3 splineToVolume(in VVizDataSetId dataset, vec3 ptInPlineSpace)
{
  // ptInPlineSpace.x is bewteen spline[pointId] and spline[pointId+1]
  int pointId = int(floor(ptInPlineSpace.x * SPLINE_SIZE-1));
  // get position of ptInPlineSpace in [pointId, pointId+1]
  float pointIdFract = fract(ptInPlineSpace.x * SPLINE_SIZE-1);

  // get point over spline (in model model space)
  vec3 pointInVolume  = mix(spline[pointId], spline[pointId+1], pointIdFract);

  // linear interpolation between tangent[n-1] and tangent[n]. This Avoid some artefacts.
  vec3 splineY1 = cross(splineNormal, spline[pointId] - spline[pointId-1]);
  vec3 splineY2 = cross(splineNormal, spline[pointId+1] - spline[pointId]);
  vec3 splineY = mix(splineY1, splineY2, pointIdFract);

  // here the 100 factor is a hard coded value to properly scale data. It depend of original dataset extent.
  // To simplify, here we hard code a value that fit well with given dataset but ideally it should
  // be computed from dataset extent.
  pointInVolume += 100*ptInPlineSpace.y * normalize(splineY);
  pointInVolume += 100*ptInPlineSpace.z * normalize(splineNormal);

  // back to texSpace
  return VVizObjectToTextureVec(dataset, pointInVolume);
}

VVIZ_DATATYPE VVizGetData(in VVizDataSetId dataset, vec3 tcoord)
{
  vec3 newCoord = tcoord;
  // small space change so that spline is in middle of volume and not in lower bottom corner.
  newCoord.yz = 2.0 * (newCoord.yz - vec2(0.5));
  newCoord = splineToVolume(dataset, newCoord);

  // small fancy effect: interpolat between original coordinates and new coordinates.
  // Linked to SoVolumeRenderingQuality::gradientThreashold slider for convenience.
  // Changind this slider value in [0, 10] will wrap/unwrap the volume.
  vec3 mixedTexCoord = mix(newCoord, tcoord, 0.1*unfoldPosition);
  
  // get the data value at this position
  VVIZ_DATATYPE sf = VVizGetRawData(dataset, mixedTexCoord);

  return sf;
}

