﻿///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Jan 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to import DICOM medical data set and how to get
DICOM information.
Description : Main
-------------------------------------------------------------------------*/
// Inventor Headers
#include <Inventor/Xt/SoXt.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoComplexity.h>

// VolumeViz headers
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

/// LDM headers
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/manips/SoROIManip.h>

// Medical headers
#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////
// Data Set
// This file contains the list of dicom we want to load.
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

// color map can be edited within Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

/*********************************************************************************/
int main(int /*argc*/, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow)
    return 0;

  SoVolumeRendering::init();
  InventorMedical::init();

  // ROOTs
  SoRef<SoSeparator> root = new SoSeparator();

  SoOrthographicCamera *localCamera = new SoOrthographicCamera();
  localCamera->position.setValue(SbVec3f(7.90892f, -244.637f, -17.1747f));
  localCamera->orientation.setValue(SbVec3f(1, 0, 0), 1.53932f);
  root->addChild(localCamera);

  // Volume Rendering Separator =========================================
  SoSeparator *rootVR = new SoSeparator();
  rootVR->setName("root_VR");
  root->addChild(rootVR);

  // Node to hold the volume data
  SoVolumeData *volData = new SoVolumeData();
    volData->fileName = VOLUME_FILENAME;
    MedicalHelper::dicomAdjustVolume( volData );

  // To give the repository where to find list of dicom.
  // Create a reader to import dicom under the following folder and uncomment following code :
  //SoVRDicomFileReader *DicomReader = new SoVRDicomFileReader();
  //DicomReader->setDirectory("$OIVHOME/examples/data/Medical/dicomSample/");
  //volData->setReader(*DicomReader);

  rootVR->addChild(volData);
    
  // Add specular to change rendering quality.
  SoMaterial *volumeMaterial = new SoMaterial();
    volumeMaterial->specularColor.setValue(0.26f, 0.26f, 0.26f);
    rootVR->addChild(volumeMaterial);

  // Load the colorMap
  SoTransferFunction *pVRTransFunc = new SoTransferFunction();
    pVRTransFunc->loadColormap(COLORMAP_FILENAME);
    rootVR->addChild(pVRTransFunc);

  // remap data range 
  // Note: Our colormap is tuned for a specific data range so we don't use MedicalHelper.
  SoDataRange *VRRange = new SoDataRange();
    VRRange->min = 176;
    VRRange->max = 476;
    rootVR->addChild(VRRange);

  // Change complexity when the dataset moves.
  SoInteractiveComplexity *intercatComplexity = new SoInteractiveComplexity();
    intercatComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
    intercatComplexity->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    intercatComplexity->refinementDelay = 0;
    rootVR->addChild(intercatComplexity);
    rootVR->addChild(new SoComplexity());

  // Property node which allows SoVolumeRender to draw High Quality volumes.  
  SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality();
    pVRVolQuality->interpolateOnMove = true;
    pVRVolQuality->deferredLighting = true;
    pVRVolQuality->preIntegrated = true;
    pVRVolQuality->ambientOcclusion = true;
    rootVR->addChild(pVRVolQuality);

  // Node in charge of drawing the volume
  SoVolumeRender *pVolRender = new SoVolumeRender();
    pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
    pVolRender->lowScreenResolutionScale = 2;
    pVolRender->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
    pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    rootVR->addChild(pVolRender);

  // Medical Gnomon.
  Gnomon *gnomon = new Gnomon();
  root->addChild(gnomon);

  // Dicom annotation
  root->addChild( 
    MedicalHelper::exampleDicomAnnotation( IMAGE_FILENAME ) );

  // Define Open Inventor logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Set up viewer :
  SoXtExaminerViewer* viewer3D = new SoXtExaminerViewer(myWindow);
    viewer3D->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    viewer3D->setDecoration(false);
    viewer3D->setSize( MedicalHelper::exampleWindowSize() );
    viewer3D->setTitle("Medical Volume Viewer");
    viewer3D->setSceneGraph(root.ptr());

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::CORONAL, viewer3D->getCamera(), volData );
  viewer3D->saveHomePosition();

  // Run then cleanup
  viewer3D->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer3D;
  root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}
