// SceneExaminer with patch for 2-finger panning in touch() method.

#include "MySceneExaminer.h"

#include <cmath>
#include <Inventor/events/SoMouseWheelEvent.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/touch/events/SoTouchEvent.h>
#include <Inventor/gestures/events/SoScaleGestureEvent.h>
#include <Inventor/gestures/events/SoRotateGestureEvent.h>
#include <Inventor/gestures/events/SoDoubleTapGestureEvent.h>
#include <Inventor/touch/SoTouchManager.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/elements/SoViewportRegionElement.h>
#include <Inventor/ViewerComponents/SoCameraInteractor.h>
#include <Inventor/SoPickedPoint.h>

MySceneExaminer::MySceneExaminer()
  : m_isTouchOrbitActivated(false)
{
}

void
MySceneExaminer::touch( SoTouchEvent* touchEvent, SoHandleEventAction* action )
{
  // Unfortunately these state variables are declared 'private' :-(
  // We re-use the names to minimize deviation of this code from SceneExaminer.
  SceneExaminer::InteractionMode m_activeMode = this->getInteractionMode();
  SceneExaminer::NavigationMode  m_navigationMode = this->getNavigationMode();
  bool m_isPanEnabled   = this->isPanEnabled();
  bool m_isOrbitEnabled = this->isOrbitEnabled();

  if (m_activeMode == SceneExaminer::NAVIGATION)
  {
    const SbViewportRegion& vpRegion = SoViewportRegionElement::get( action->getState() );
    SbVec2f touchNormPosition = touchEvent->getNormalizedPosition(vpRegion);
    SoTouchEvent::State state = touchEvent->getState();
    int numFinger = touchEvent->getTouchManager()->getFingerNumber();

    if ( numFinger == 1)
    {
      switch ( m_navigationMode )
      {
      case PLANE:
        if ( m_isPanEnabled && ( state == SoTouchEvent::DOWN || state == SoTouchEvent::MOVE ) )
        {
          // 1 finger down or moved
          m_cameraInteractor->translate(vpRegion.normalize(SbVec2s(touchEvent->getDisplacement() * 0.5f)), vpRegion);
          action->setHandled();
        }
        break;
      case ORBIT:
      default:
        if ( m_isOrbitEnabled )
        {
          if ( state == SoTouchEvent::DOWN )
          {
            // one finger down
            m_cameraInteractor->activateOrbiting(touchNormPosition);
            m_cameraInteractor->setRotationCenter(m_cameraInteractor->getFocalPoint());
            m_isTouchOrbitActivated = true;
            action->setHandled();
          }
          else if ( state == SoTouchEvent::MOVE && m_isTouchOrbitActivated )
          {
            // one finger moved
            m_cameraInteractor->orbit(touchNormPosition);
            m_cameraInteractor->adjustClippingPlanes(this, vpRegion);
            action->setHandled();
          }
        }
      }
    }
    else if ( numFinger == 2 )
    {
      if ( m_isPanEnabled && ( state == SoTouchEvent::DOWN || state == SoTouchEvent::MOVE ) )
      {
        // 2 fingers down or moved
#ifdef ORIGINAL
        m_cameraInteractor->translate(vpRegion.normalize(SbVec2s(touchEvent->getDisplacement() * 0.5f)), vpRegion);
#else
        SbVec2f displacement = 0.5f * touchEvent->getDisplacement();
        const SbVec2i32& vpSize = vpRegion.getViewportSizePixels_i32();
        SbVec2f normDisp( displacement[0]/vpSize[0], displacement[1]/vpSize[1] );
        m_cameraInteractor->translate( normDisp, vpRegion );
#endif
        action->setHandled();
      }
      else if ( state == SoTouchEvent::UP )
      {
        // one finger is on the screen but one has been lifted,
        // orbiting is temporarily disabled until the next touch down event.
        m_isTouchOrbitActivated = false;
      }
    }
  }
  else
  {
    // picking mode
    SoEvent* event_out = convertTouchEvent(touchEvent);
    action->setEvent(event_out);
  }
}

///////////////////////////////////////////////////////////////////////////////
// Must copy this method because it's declared private in parent class. :-(

SoEvent*
MySceneExaminer::convertTouchEvent(SoTouchEvent* touchEvent)
{
  SoMouseButtonEvent* mbe = &m_touchMouseButtonEvent;
  SoLocation2Event* lct = &m_touchLocation2Event;
  SoTouchEvent::State state = touchEvent->getState();

  if ( state == SoTouchEvent::DOWN)
  {
    mbe->setTime(touchEvent->getTime());
    mbe->setPosition(touchEvent->getPosition());
    mbe->setButton(SoMouseButtonEvent::BUTTON1);
    mbe->setState(SoMouseButtonEvent::DOWN);
    return mbe;
  }

  if ( state == SoTouchEvent::MOVE)
  {
    lct->setTime(touchEvent->getTime());
    lct->setPosition(touchEvent->getPosition());
    lct->setEventSource(SoLocation2Event::MOUSE_MOVE);
    return lct;
  }

  if ( state == SoTouchEvent::UP)
  {
    mbe->setTime(touchEvent->getTime());
    mbe->setPosition(touchEvent->getPosition());
    mbe->setButton(SoMouseButtonEvent::BUTTON1);
    mbe->setState(SoMouseButtonEvent::UP);
    return mbe;
  }

  SoDebugError::post( __FUNCTION__, "Unknown Touch Event" );
  return NULL;
}
