///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include "OGLWindow.h"

#include <Inventor/sys/SoGLX.h>
#include <unistd.h>
#include <X11/keysym.h>

#include <Inventor/SoDB.h>
#include <Inventor/sys/SoGL.h>
#include <Inventor/sys/SoGLU.h>

class OGLWindowImpl
{
public:
  OGLWindowImpl(int width, int height);

  ~OGLWindowImpl();

  void render();

  void mainLoop();

  void setDrawCallback(DrawCB callback);
  
  void setKeyCallback(KeyCB callback);

  void bindContext();
  
  void unbindContext();

  void update();

  SoGLContext* m_context;
  DrawCB m_cb;
  KeyCB m_keyCb;
  
  Display* m_display;
  Window m_window;
  XVisualInfo* m_visualInfo;
};


OGLWindow::OGLWindow(int width, int height)
{
  m_impl = new OGLWindowImpl(width, height);
}

OGLWindow::~OGLWindow()
{
  delete m_impl;
}

void 
OGLWindow::render()
{
  m_impl->render();
}

void
OGLWindow::update()
{
  m_impl->update();
}

void
OGLWindow::mainLoop()
{
  m_impl->mainLoop();
}

void 
OGLWindow::setDrawCallback(DrawCB callback)
{
  m_impl->setDrawCallback(callback);
}

void 
OGLWindow::setKeyCallback(KeyCB callback)
{
  m_impl->setKeyCallback(callback);
}


void 
OGLWindow::bindContext()
{
  m_impl->bindContext();
}

void 
OGLWindow::unbindContext()
{
  m_impl->unbindContext();
}

// Callback used by GLX window
static Bool waitForNotify(Display *, XEvent *e, char *arg)
{
  return (e->type == MapNotify) && (e->xmap.window == (Window)arg);
}


OGLWindowImpl::OGLWindowImpl(int width, int height)
{
  m_context = NULL;
  m_window = 0;
  m_display = NULL;
  m_visualInfo = NULL;

  Colormap cmap;
  XSetWindowAttributes swa;
  XEvent event;
  static int attributeList[] = {
    GLX_RGBA,
    GLX_RED_SIZE, 1,
    GLX_GREEN_SIZE, 1,
    GLX_BLUE_SIZE, 1,
    GLX_DEPTH_SIZE, 1,
    GLX_DOUBLEBUFFER,        
    None,
  };

  // Open the X display 
  m_display = XOpenDisplay(0);

  // Initialize the GLX visual and context
  m_visualInfo = glXChooseVisual(m_display, DefaultScreen(m_display), 
				 attributeList);

  // Create the X color map
  cmap = XCreateColormap(m_display, 
			 RootWindow(m_display, m_visualInfo->screen), 
			 m_visualInfo->visual, AllocNone);

  // Create and map the X window
  swa.colormap = cmap;
  swa.border_pixel = 0;
  swa.event_mask = StructureNotifyMask | KeyPressMask | VisibilityChangeMask | ExposureMask;
  m_window = XCreateWindow(m_display, 
                         RootWindow(m_display, m_visualInfo->screen),
                         100,
                         100,
                         width,
                         height,
                         0,
                         m_visualInfo->depth,
                         InputOutput,
                         m_visualInfo->visual, 
                         (CWBorderPixel | CWColormap | CWEventMask),
                         &swa);
  XMapWindow(m_display, m_window);
  XIfEvent(m_display, &event, waitForNotify, (char *)m_window);

  SoGLFormat format;
  format.setDisplay( m_display );
  format.setVisualInfo( m_visualInfo );
  
  m_context = new SoGLContext( format, m_window );
}

OGLWindowImpl::~OGLWindowImpl()
{
  if (m_context)
  {
    SoGLContext::invalidateDisplay(m_context->getDisplay());
    m_context->unref();
  }

  if (m_window)
    XDestroyWindow(m_display, m_window);
  if (m_display)
    XCloseDisplay(m_display);
}

void 
OGLWindowImpl::render()
{
  SoDB::processEvents();

  if (m_cb)
    m_cb();

  m_context->swapBuffers();
}

void
OGLWindowImpl::update()
{
  if (m_window)
    XClearArea(m_display, m_window, 0, 0, 10, 10, TRUE);
}

void
OGLWindowImpl::mainLoop()
{
  Display* dpy = m_display;
  XEvent event;
  while (1)
  {
    do
    {
      XNextEvent( dpy, &event );
      switch (event.type)
      {
        case Expose:
	  if (event.xexpose.count != 0)
	    break;
	  render();
	  break;
        case KeyPress:
	  {
	    KeySym keysym = XLookupKeysym( &event.xkey, 0 );
	    if (m_keyCb)
	    {
	      if (keysym == (KeySym)XK_Left)
		m_keyCb(KEY_LEFT);
	      else if (keysym == (KeySym)XK_Right)
		m_keyCb(KEY_RIGHT);
	      else if (keysym == (KeySym)XK_Up)
		m_keyCb(KEY_UP);
	      else if (keysym == (KeySym)XK_Down)
		m_keyCb(KEY_DOWN);
	      else if (keysym == (KeySym)XK_space)
		m_keyCb(KEY_SPACE);
	      else if (keysym == (KeySym)XK_j)
		m_keyCb(KEY_J);
	      else if (keysym == (KeySym)XK_p)
		m_keyCb(KEY_P);
	    }
	  if (keysym == (KeySym)XK_Escape)
	    return;
          break;
        }
        default:
          break;
      }
    }
    while (XPending( dpy ));
  }
}

void 
OGLWindowImpl::setDrawCallback(DrawCB callback)
{
  m_cb = callback;
}

void 
OGLWindowImpl::setKeyCallback(KeyCB callback)
{
  m_keyCb = callback;
}

void 
OGLWindowImpl::bindContext()
{
  m_context->bind();
}

void 
OGLWindowImpl::unbindContext()
{
  m_context->unbind();
}
