///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/
#include "OGLWindow.h"

#include <Inventor/sys/SoGL.h>
#include <Inventor/sys/SoGLU.h>
#include <stdio.h>

#include <Inventor/SoDB.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoMatrixTransform.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/elements/SoDataSetIdElement.h>

#include <Inventor/actions/SoGLRenderAction.h>

#include <Inventor/elements/SoModelMatrixElement.h>
#include <Inventor/elements/SoProjectionMatrixElement.h>
#include <Inventor/elements/SoViewingMatrixElement.h>
#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/elements/SoInteractionElement.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include <Inventor/SoSceneManager.h>

#define WINWIDTH 1024
#define WINHEIGHT 800

const double ANGLE_INCR = 0.02094395102393195492308428922186; // radians

////////////////////////////////////////////////////////////////////////
// Global variables

static float floorObj[81][3];

static SoSeparator  *    gpRoot;  // Root of OIV scene graph
static SoRotationXYZ*    gpRot;   // Rotation for animating scene
static SoSceneManager*   g_sceneManager; 
static SbBool            g_isInteracting = FALSE;

static SoVolumeRenderingQuality *gpVolQuality = NULL;

// Define initial view parameters
float fovy     = 45; // degrees - similar to default OIV view frustum
float aspect   = (float)WINWIDTH / WINHEIGHT; // View frustum and viewport aspect should match!
float nearDist =  5;
float farDist  = 20;
float camXpos  =  0;
float camYpos  =  0;
float camZpos  = 10;

SoPerspectiveCamera* gCamera;
SoMatrixTransform* gTransform;

////////////////////////////////////////////////////////////////////////
// Forward decls

void drawScene();
void KeyCallback(const SimpleKeyCode);

// OpenGL functions
void drawGLScene();
void buildFloor();
void drawFloor();

// Open Inventor functions
void buildOivScene( SoSeparator *root );
void drawOivScene();
SoSeparator *makeVolBBox( SbBox3f volSize ); // Utility function

////////////////////////////////////////////////////////////////////////
// Render callback
////////////////////////////////////////////////////////////////////////
void renderCallback(void* userData, SoSceneManager* /*mgr*/)
{
  OGLWindow* win = (OGLWindow*)userData;
  if (win)
    win->update();
}


////////////////////////////////////////////////////////////////////////
int main(int argc, char ** argv)
{
  // Initialize Open Inventor
  SoDB::init();

  SoVolumeRendering::init();
  InventorMedical::init();

  // Create and initialize OpenGL window
  OGLWindow* window = new OGLWindow( WINWIDTH, WINHEIGHT );
  window->setDrawCallback(drawScene);
  window->setKeyCallback(KeyCallback);
  window->bindContext();

  // Make sure depth test is enabled
  glEnable(GL_DEPTH_TEST);

  // Build the floor geometry with OpenGL
  buildFloor();

  //Setup the camera
  gCamera = new SoPerspectiveCamera;
  gCamera->heightAngle = fovy;
  gCamera->aspectRatio = aspect;
  gCamera->nearDistance = nearDist;
  gCamera->farDistance = farDist;
  gCamera->position = SbVec3f(camXpos,camYpos,camZpos);

  // Build Open Inventor scene
  gpRoot = new SoSeparator; // global variable
  gpRoot->ref();
  buildOivScene( gpRoot );

  // Create and initialize the Open Inventor render action
  g_sceneManager = new SoSceneManager();
  g_sceneManager->setViewportRegion(SbViewportRegion(WINWIDTH, WINHEIGHT));
  g_sceneManager->setSceneGraph(gpRoot);
  g_sceneManager->setRenderCallback(renderCallback, window);
  g_sceneManager->getGLRenderAction()->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  g_sceneManager->activate();

  SbBool onlyOneDraw = FALSE;
  if (argc == 2 && !strcmp(argv[1], "--noMainLoop"))
    onlyOneDraw = TRUE;

  // Event loop
  // Handle some arrow keys to modify the Open Inventor scene
  printf( "To rotate press left/right arrow keys...\n" );
  printf( "To move forward/backward press up/down arrow keys...\n" );
  printf( "To reset position press <space> key...\n" );
  printf( "To enable Jittering press <J> key...\n" );
  printf( "To enable PreIntegrated press <P> key...\n" );
  printf( "To enable/disable Interaction Mode press <I> key...\n" );
  printf( "To Quit press <ESC> key...\n" );

  if (onlyOneDraw)
    window->render();
  else
    window->mainLoop();

  window->unbindContext();

  delete window;

  // Cleanup Open Inventor resources
  gpRoot->unref();
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDB::finish();
  return 0;
}


void IKey()
{
  g_isInteracting = g_isInteracting?FALSE:TRUE;
}

void leftKey()
{
  double angle = gpRot->angle.getValue();
  gpRot->angle = float(angle + ANGLE_INCR);
}

void rightKey()
{
  double angle = gpRot->angle.getValue();
  gpRot->angle = float(angle - ANGLE_INCR);
}

void spaceKey()
{
  gpRot->angle = 0;
}

void upKey()
{ // Right arrow
  SbVec3f newPos = gCamera->position.getValue();
  newPos[2] += (float)ANGLE_INCR;
  gCamera->position = newPos;
}

void downKey()
{ // Right arrow
  SbVec3f newPos = gCamera->position.getValue();
  newPos[2] -= (float)ANGLE_INCR;
  gCamera->position = newPos;
}

void JKey()
{
  gpVolQuality->jittering = ! gpVolQuality->jittering.getValue();
  printf( "Jittering: %s\n", gpVolQuality->jittering.getValue() ? "ON" : "OFF" );
}

void PKey()
{
  gpVolQuality->preIntegrated = ! gpVolQuality->preIntegrated.getValue();
  printf( "PreIntegrated: %s\n", gpVolQuality->preIntegrated.getValue() ? "ON" : "OFF" );
}

void KeyCallback(const SimpleKeyCode key)
{
  if (key == KEY_LEFT)
  { // Left arrow
    leftKey();
  }
  else if (key == KEY_RIGHT)
  { // Right arrow
    rightKey();
  }
  else if (key == KEY_SPACE)
  {
    spaceKey();
  }
  else if (key == KEY_UP)
  { // Right arrow
    upKey();
  }
  else if (key == KEY_DOWN)
  { // Right arrow
    downKey();
  }
  else if (key == KEY_J)
  {
    JKey();
  }
  else if (key == KEY_P)
  {
    PKey();
  }
  else if (key == KEY_I)
  {
    IKey();
  }

  g_sceneManager->scheduleRedraw();
}

////////////////////////////////////////////////////////////////////////
void
pushGLMatrices()
{
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
}

////////////////////////////////////////////////////////////////////////
void
popGLMatrices()
{
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}

////////////////////////////////////////////////////////////////////////
void drawScene() 
{
  // Clear framebuffer
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Rerender OpenGL scene
  pushGLMatrices();
  drawGLScene();
  popGLMatrices();

  // Render Open Inventor scene graph
  drawOivScene();
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// OpenGL Functions

////////////////////////////////////////////////////////////////////////
void drawGLScene() 
{
  // Set up the camera using OpenGL.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective( fovy, aspect, nearDist, farDist );

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glTranslatef( -camXpos, -camYpos, -camZpos );

  // Rerender the floor using OpenGL
  glPushMatrix();
  glTranslatef(0.0f, -3.0f, 0.0f);
  glColor3f(0.0f, 0.7f, 0.0f);
  glLineWidth(2.0);
  glDisable(GL_LIGHTING);
  drawFloor();
  glEnable(GL_LIGHTING);
  glPopMatrix();
}

////////////////////////////////////////////////////////////////////////
// Build a floor that will be rendered using OpenGL.
void buildFloor()
{
  int a = 0;
  for (float i = -5.0f; i <= 5.0f; i += 1.25f) {
    for (float j = -5.0f; j <= 5.0f; j += 1.25f, a++) {
      floorObj[a][0] = j;
      floorObj[a][1] = 0.0f;
      floorObj[a][2] = i;
    }
  }
}

////////////////////////////////////////////////////////////////////////
// Draw the lines that make up the floor, using OpenGL
void
drawFloor()
{
  int i;

  glBegin(GL_LINES);
  for (i=0; i<4; i++) {
    glVertex3fv(floorObj[i*18]);
    glVertex3fv(floorObj[(i*18)+8]);
    glVertex3fv(floorObj[(i*18)+17]);
    glVertex3fv(floorObj[(i*18)+9]);
  }

  glVertex3fv(floorObj[i*18]);
  glVertex3fv(floorObj[(i*18)+8]);
  glEnd();

  glBegin(GL_LINES);
  for (i=0; i<4; i++) {
    glVertex3fv(floorObj[i*2]);
    glVertex3fv(floorObj[(i*2)+72]);
    glVertex3fv(floorObj[(i*2)+73]);
    glVertex3fv(floorObj[(i*2)+1]);
  }
  glVertex3fv(floorObj[i*2]);
  glVertex3fv(floorObj[(i*2)+72]);
  glEnd();
}

////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////
// Open Inventor functions

////////////////////////////////////////////////////////////////////////
void buildOivScene(SoSeparator *root)
{
  root->addChild(gCamera);

  // Node to reference the volume data
  // Note: We could use $OIVHOME in the path but that only works in OIV 8.1 or later...
  // pVolData->fileName = "$OIVHOME/examples/data/VolumeViz/3dhead.ldm";
  SoVolumeData *pVolData = new SoVolumeData();
  SbString datapath = SoPreferences::getString( "OIVHOME", "." );
  datapath += "/examples/data/Medical/files/3DHEAD.ldm";
  pVolData->fileName = datapath;

  // Scale up the volume size by adjusting the volume extent
  const float SCALE_FACTOR = 3;

  // We know the OpenGL view is setup to look at 0,0,0
  // so adjust the volume extent to span -1..1 (* the scale factor above)
  const SbBox3f   volext = pVolData->extent.getValue();
  float sizex, sizey, sizez;
  volext.getSize( sizex, sizey, sizez );
  float maxext = SbMathHelper::Max( sizex, SbMathHelper::Max( sizey, sizez ));
  SbVec3f maxExtent( sizex/maxext, sizey/maxext, sizez/maxext );
  maxExtent *= SCALE_FACTOR;
  SbVec3f minExtent = -1 * maxExtent;
  SbBox3f volExtent(minExtent,maxExtent);
  pVolData->extent.setValue( volExtent );

  // Set reasonable parameters for this data volume (so we can run with pre-8.1 OIV)
  SoLDMGlobalResourceParameters::setLoadNotificationRate(100);
  SoLDMGlobalResourceParameters::setTex3LoadRate(64);
  SoLDMGlobalResourceParameters::setMaxMainMemory(512);
  SoLDMGlobalResourceParameters::setMaxTexMemory(256);
  pVolData->ldmResourceParameters.getValue()->loadPolicy = SoLDMResourceParameters::ALWAYS;

  // Set the data range
  // This is the range of voxel values that will be mapped into the transfer function.
  // May not be needed for 8-bit voxels, but usually required for larger data types.
  SoDataRange *pDataRange = new SoDataRange;

  // Set the color map (transfer function).
  // Just use a predefined color map for now.
  // Make voxels in range 0..minValue-1 completely transparent (they're noise).
  SoTransferFunction *pTransFunc = new SoTransferFunction;
  pTransFunc->predefColorMap = SoTransferFunction::GLOW;
  pTransFunc->minValue = 28;

  // Create a material node to manage the "global" alpha scale factor
  SoMaterial *pVolMaterial = new SoMaterial;
  pVolMaterial->diffuseColor = SbColor(1,1,1);
  pVolMaterial->transparency = 0.7f;

  // Image quality settings
  SoVolumeRenderingQuality* pVolQuality = new SoVolumeRenderingQuality;
  pVolQuality->preIntegrated = TRUE;
  pVolQuality->jittering     = TRUE;
  gpVolQuality = pVolQuality;

  // Node in charge of drawing the volume
  // Increase number of slices to improve image quality
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::MANUAL;
  pVolRender->numSlices = 512;
  pVolRender->samplingAlignment = SoVolumeRender::VIEW_ALIGNED;

  // Rotate the volume so the "head" is right side up.
  // (specific to 3dhead.ldm data set)
  SoRotationXYZ* pVolRotate = new SoRotationXYZ;
  pVolRotate->axis = SoRotationXYZ::Z;
  pVolRotate->angle = float(3.1415926535897932384626433832795);

  // Additional rotation that will be modified for animation
  gpRot = new SoRotationXYZ;
  gpRot->axis = SoRotationXYZ::Y;

  // ---------- Assemble scene graph ----------

  // Some light in the scene
  // Note: This should match the OpenGL scene lighting
  //       (but there isn't any in this simple example)
  SoDirectionalLight *pDirLight = new SoDirectionalLight;
  pDirLight->direction.setValue( SbVec3f(.2f, -.2f, -.9797958971f) );
  root->addChild( pDirLight );

  // Volume rotation and animation rotation nodes
  // In this case they apply to whole scene.
  root->addChild( pVolRotate );
  root->addChild( gpRot );

  // Volume rendering nodes
  // Because SoVolumeRenderingQuality is a shader, it must 
  // in a separator in order to not be applied on the whole scenegraph
  SoSeparator *pVolSep = new SoSeparator;
  pVolSep->addChild( pVolData );
  pVolSep->addChild( pVolMaterial );
  pVolSep->addChild( pDataRange );
  pVolSep->addChild( pTransFunc );
  pVolSep->addChild( pVolQuality );
  pVolSep->addChild( pVolRender );
  root->addChild( pVolSep );

  // Some extra geometry just for example...
  const float CUBE_SIZE = 0.5f;
  SoSeparator *pGeomSep = new SoSeparator;
  SoTransform *pTran = new SoTransform;
  SoMaterial  *pMatl = new SoMaterial;
  SoCube      *pCube = new SoCube;
  pMatl->diffuseColor.setValue( 0, 0, 1 );
  pTran->translation = maxExtent - SbVec3f(CUBE_SIZE/2,CUBE_SIZE/2,CUBE_SIZE/2);
  pCube->width  = CUBE_SIZE;
  pCube->height = CUBE_SIZE;
  pCube->depth  = CUBE_SIZE;
  pGeomSep->addChild( pTran );
  pGeomSep->addChild( pMatl );
  pGeomSep->addChild( pCube );
  root->addChild( pGeomSep );

  // Draw outline of volume extent to check clipping
  // Note: makeVolBBox returns an SoSeparator
  root->addChild( makeVolBBox( volExtent ) );

  // Define OIV logo
  root->addChild( MedicalHelper::exampleLogoNode() );
}

////////////////////////////////////////////////////////////////////////
void drawOivScene()
{
  if ( g_sceneManager == NULL )
    return;

  // setup interactive mode
    SoGLRenderAction* renderAction = g_sceneManager->getGLRenderAction();
    SoState* state = renderAction->getState();
    if (state == NULL)
    {
        renderAction->setUpState();
        state = renderAction->getState();
    }
    SoInteractionElement::set(state, g_isInteracting);

  // - Render
  g_sceneManager->render();
}

///////////////////////////////////////////////////////////////////////
// Create a wireframe box showing bounding box of volume
//
SoSeparator *makeVolBBox( SbBox3f volSize )
{
  // The box will be easier to see without lighting and with wide lines
  SoLightModel *pLModel = new SoLightModel;
  pLModel->model = SoLightModel::BASE_COLOR;

  SoDrawStyle *pStyle = new SoDrawStyle;
  pStyle->lineWidth = 2;

  // The box should be unpickable so manip can be used inside it
  SoPickStyle *pPickable = new SoPickStyle;
  pPickable->style = SoPickStyle::UNPICKABLE;

  // Create a cube with the geometric size of the volume
  float xmin, xmax, ymin, ymax, zmin, zmax;
  volSize.getBounds( xmin,ymin, zmin, xmax, ymax, zmax );
  SoVertexProperty *pProp = new SoVertexProperty;
  pProp->vertex.set1Value( 0, SbVec3f(xmin,ymin,zmin) );
  pProp->vertex.set1Value( 1, SbVec3f(xmax,ymin,zmin) );
  pProp->vertex.set1Value( 2, SbVec3f(xmax,ymax,zmin) );
  pProp->vertex.set1Value( 3, SbVec3f(xmin,ymax,zmin) );
  pProp->vertex.set1Value( 4, SbVec3f(xmin,ymin,zmax) );
  pProp->vertex.set1Value( 5, SbVec3f(xmax,ymin,zmax) );
  pProp->vertex.set1Value( 6, SbVec3f(xmax,ymax,zmax) );
  pProp->vertex.set1Value( 7, SbVec3f(xmin,ymax,zmax) );
  pProp->orderedRGBA.set1Value( 0, 0xFF0000FF );

  // Draw it with a line set
  int coordIndices[] = { 0, 1, 2, 3, 0, -1, 4, 5, 6, 7, 4, -1, 0, 4, -1, 1, 5, -1, 2, 6, -1, 3, 7 };
  int numCoordIndices = sizeof(coordIndices) / sizeof(int);
  SoIndexedLineSet *pLines = new SoIndexedLineSet;
  pLines->vertexProperty = pProp;
  pLines->coordIndex.setValues( 0, numCoordIndices, coordIndices );

  // Assemble scene graph
  SoSeparator *pBoxSep = new SoSeparator;
  pBoxSep->addChild( pLModel );
  pBoxSep->addChild( pPickable );
  pBoxSep->addChild( pStyle );
  pBoxSep->addChild( pLines );
  return pBoxSep;
}

