///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how use PlaneGeometryIntersection (From medicalExample 
library)node to generate capping on a surface object (STL, iv, Catia ... 
all format suported within Open Inventor).
Description : Main
-------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoIndexedFaceSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoOrthographicCamera.h>

#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoGetPrimitiveCountAction.h>
#include <Inventor/actions/SoWriteAction.h>

#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/manips/SoClipPlaneManip.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/SoInput.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoMenuFileSelection.h> // For platform independent open file dialog

// Medical headers
#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/PlaneGeometryIntersection.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

#include <Inventor/STL/algorithm>
#include <Inventor/STL/iostream>
#include <Inventor/STL/vector>


/////////////////////////////////////////////////////////////
// Application state

static SoClipPlaneManip*			m_clipManip     = NULL;
static PlaneGeometryIntersection* m_intersect		= NULL;
static SoMaterial*					m_intersectMatl = NULL;

static SbBox3f m_defaultBbox;
static SbVec3f m_defaultPlane;
static float m_defaultScaleFactor;

static SoMenuFileSelection* m_fileSelection = NULL; // Open file dialog
static SoXtExaminerViewer*  m_viewer        = NULL;

/////////////////////////////////////////////////////////////
// Constants / Declarations

// Default Data Set
const char *DEFAULT_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/HUANG-GUANG-ZE.surf.iv";

#ifdef WIN32
SbBool viewerEventCB( void *userData, MSG *msg );
#endif

void EventCB(void *userData, SoEventCallback *node );

/////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
  std::cout << "Press 'A' to toggle automatic mode (enabled by default).\n";
  std::cout << "      'F' to select a new file to load.\n";
  std::cout << "      'I' to compute intersection when not in automatic mode.\n";
  std::cout << "      'P' to toggle between curve mode and polygon mode (curve by default).\n";
  std::cout << "      'R' to reset the clip plane dragger to its initial position.\n";
  std::cout << "      'T' to toggle transparency for the intersection polygon.\n";
  std::cout << std::endl;

  Widget myWindow = SoXt::init(argv[0]);
  InventorMedical::init();
  SoDialogViz::init();

  SoRef<SoSeparator> root = new SoSeparator();

  SoOrthographicCamera* camera = new SoOrthographicCamera();
  root->addChild(camera);

  // Catch keyboard events for simple user interface
  SoEventCallback* evCB = new SoEventCallback;
    evCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), EventCB, NULL );
    root->addChild( evCB );

  // This sub-graph contains the clip plane manip and geometry to be clipped.
  // The intersection geometry should be in a completely separate sub-graph.
  SoSeparator* scene = new SoSeparator();
    scene->setName( "Scene" );
    root->addChild( scene );
 
  // Insert a clip plane + manipulator
  SoClipPlaneManip* manip = new SoClipPlaneManip();
    manip->setName( "ClipPlaneManip" );
    scene->addChild( manip );

  // Offset the scene polygons so intersection curves will be clearly on top.
  SoPolygonOffset* offset = new SoPolygonOffset();
    offset->units = 1;
    offset->factor = 1;
    scene->addChild( offset );

  // This sub-sub-graph will have its contents replaced when a new file is loaded.
  SoSeparator* sceneGeometry = new SoSeparator();
    sceneGeometry->setName( "SceneGeometry" );
    scene->addChild( sceneGeometry );

  // Try to read the file
  SoSeparator* geometry = NULL;
  SbString filename = (argc > 1) ? argv[1] : DEFAULT_FILENAME;
  if (SbFileHelper::isAccessible( filename )) {
    SoInput in;
    if (in.openFile( filename )) {
      geometry = SoDB::readAll( &in );
      in.closeFile();
    }
  }
  else {
    std::cout << "Unable to open file '" << filename << "'\n";
  }

  // If that failed, create our favorite geometry
  if (geometry == NULL) {
    geometry = new SoSeparator();
    
    SoSphere* geom = new SoSphere();
      geometry->addChild( geom );
  }
  geometry->setName("Geometry");  // for IvTune
  sceneGeometry->addChild( geometry );

  // Use the scene bounds to setup the clip plane dragger.
  SoGetBoundingBoxAction gbba( SbViewportRegion(500,500) );
  gbba.apply( sceneGeometry );
  m_defaultBbox = gbba.getBoundingBox();
  m_defaultPlane = SbVec3f(0, 0, -1);
  m_defaultScaleFactor = 0.5f;
  manip->setValue( m_defaultBbox, m_defaultPlane, m_defaultScaleFactor );

  // Report the complexity of the scene
  SoGetPrimitiveCountAction gpca;
  gpca.apply( sceneGeometry );
  int numTris = gpca.getTriangleCount();
  std::cout << "Scene: " << numTris << " triangles.\n";

  //---------------------------------------------------------------------------
  // Intersection geometry should not be affected by the clip plane.
  SoSeparator* intGeometry = new SoSeparator();
    intGeometry->setName( "IntGeometry" );
    root->addChild( intGeometry );

  // Exaggerate the point size and line width for the intersection
  SoDrawStyle* capStyle = new SoDrawStyle();
    capStyle->style = SoDrawStyle::FILLED;
    capStyle->lineWidth = 3;
    capStyle->pointSize = 10;
    intGeometry->addChild( capStyle );

  // Make the intersection red for visibility
  SoMaterial* capMatl = new SoMaterial();
    capMatl->diffuseColor.setValue( 1, 0, 0 );
    intGeometry->addChild( capMatl );

  // Extract and display the intersection geometry.
  // NOTE: Be careful that the specified scene graph does NOT include the
  //       clip plane dragger.
  PlaneGeometryIntersection* intersect = new PlaneGeometryIntersection();
    intersect->scene = sceneGeometry;
    intersect->plane.connectFrom( &(manip->plane) );
    intersect->automatic = TRUE;
    intersect->polygon   = FALSE;
    intGeometry->addChild( intersect );

  // OIV Logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Medical Gnomon.
  Gnomon *gnomon = new Gnomon();
    root->addChild(gnomon);

  // Notes
  TextBox* textBox = new TextBox();
    textBox->position.setValue(-0.8f, -0.99f, 0);  // Normalized Device Coordinates (-1..1) - Leave room for gnomon
    textBox->alignmentV = TextBox::BOTTOM;
    textBox->addLine("Press:");
    textBox->addLine("  'P' to toggle between curve mode and polygon mode (curve by default).");
    textBox->addLine("  'R' to reset the clip plane dragger to its initial position.");
    textBox->addLine("  'T' to toggle transparency for the intersection polygon.");
    root->addChild(textBox);

  // Create a viewer
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);
#ifdef WIN32
  viewer->setEventCallback( viewerEventCB, (void*)viewer );
#endif
  viewer->setDecoration(FALSE);
  viewer->setViewing(FALSE);
  viewer->setSize( MedicalHelper::exampleWindowSize() );
  viewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  viewer->setSceneGraph(root.ptr());
  viewer->setTitle( "Capping" );

  // Adjust camera for more interesting initial view
  viewer->getCamera()->orientation.setValue( SbVec3f(0.828883f, -0.303221f, -0.470117f), 1.27283f );
  viewer->viewAll();
  MedicalHelper::dollyZoom( 1.5f, viewer->getCamera() ); // Fill viewport
  viewer->saveHomePosition();
  viewer->show();

  //------ Values that must be visible for simple UI -----
  m_viewer    = viewer;
  m_clipManip = manip;
  m_intersect = intersect;
  m_intersectMatl = capMatl;

  // Loop, then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;
  root = NULL;
  InventorMedical::finish();
  SoXt::finish();
  return 0;
}
/////////////////////////////////////////////////////////////

#ifdef WIN32
SbBool viewerEventCB( void *userData, MSG *msg )
{
  if (msg->message == WM_KEYDOWN) {
    int key = (int)(msg->wParam);

    SoWinViewer* viewer = (SoWinViewer*)userData;
    SoNode* scenegraph = ((SoWinViewer*)userData)->getSceneGraph();

    //---------------------------------------------------------------
    if (key == 'A') {
      // Toggle automatic mode
      SbBool newState = ! m_intersect->automatic.getValue();
      m_intersect->automatic = newState;
      std::cout << "Automatic: " << (newState ? "Yes" : "No") << std::endl;
    }
    //---------------------------------------------------------------
    else if (key == 'I') {
      // Trigger recomputation of intersection
      // (not useful in automatic mode of course...)
      SbPlane plane = m_intersect->plane.getValue();
      std::cout << "Plane " << plane.getNormal() << " : " << plane.getDistanceFromOrigin() << std::endl;
      m_intersect->recompute();
    }
    //---------------------------------------------------------------
    else if (key == 'P') {
      // Toggle curve/polygon mode
      SbBool newState = ! m_intersect->polygon.getValue();
      m_intersect->polygon = newState;
      std::cout << "Draw: " << (newState ? "Face" : "Curve") << std::endl;
    }
    //---------------------------------------------------------------
    else if (key == 'R') {
      // Reset dragger and clip plane
      m_clipManip->setValue( m_defaultBbox, m_defaultPlane, m_defaultScaleFactor );
    }
    //---------------------------------------------------------------
    else if (key == 'T') {
      // Toggle transparency for intersection polygon
      float transp = m_intersectMatl->transparency[0];
      m_intersectMatl->transparency.setValue( (transp != 0) ? 0 : 0.5f );
    }
    //---------------------------------------------------------------
    else if (key == 'W') {
      // Write intersection geometry to a .iv file (mainly for debugging).
      SoSeparator* capSep = (SoSeparator*)SoNode::getByName( "CapSep" );
      if (capSep == NULL) return FALSE;
      SoWriteAction wa;
      wa.getOutput()->openFile( "temp.iv" );
      wa.apply( capSep );
      wa.getOutput()->closeFile();
      std::cout << "Wrote intersection geometry to 'temp.iv'\n";
    }
  }
  return FALSE;
}
#endif

/////////////////////////////////////////////////////////////

void EventCB(void* /*userData*/, SoEventCallback* node )
{
  const SoEvent *event = node->getEvent();
  //-------------------------------------------------------------------
  // Pressing 'F' -> Open File
  if (SoKeyboardEvent::isKeyPressEvent(event, SoKeyboardEvent::F)) {
    node->setHandled();
    // Create file selection dialog if necessary.
    if (m_fileSelection == NULL) {
      m_fileSelection = new SoMenuFileSelection();
      m_fileSelection->filter.set1Value( 0, "*.*" ); // Allow any file (too many extensions to list all)
      m_fileSelection->filter.set1Value( 1, "All files" );
    }
    // Display file selection dialog
    m_fileSelection->menuEvent( NULL, 0 );

    // Get selected filename
    SbString temppath = m_fileSelection->fileDirectory.getValue();
    const SbString& filename = m_fileSelection->filename.getValue();
    temppath += "/";
    temppath += filename;
    SbString filepath = SbFileHelper::cleanUpPath( temppath );

    // If no file was selected, return.
    if (filepath.isEmpty()) {
      std::cout << "No file was selected...\n";
      return;
    }

    // If selected file cannot be opened, return.
    if (! SbFileHelper::isAccessible( filepath )) { 
      std::cout << "Unable to open '" << filepath.toLatin1() << "'\n";
      return;
    }

    // Load geometry
    SoInput in;
    SoNode* geometry = NULL;
    if (in.openFile( filepath )) {
      geometry = SoDB::readAll( &in );
      in.closeFile();
    }
    if (geometry == NULL) {
      std::cout << "Error reading '" << filepath.toLatin1() << "'\n";
      return;
    }

    // Replace geometry in scene graph
    SoSeparator* sceneGeometry = (SoSeparator*)SoNode::getByName( "SceneGeometry" );
    if (sceneGeometry != NULL) {
      // Disable intersection computation until everything is setup again.
      m_intersect->scene = NULL;

      sceneGeometry->removeAllChildren();
      sceneGeometry->addChild( geometry );

      // Use the scene bounds to setup the clip plane dragger.
      SoGetBoundingBoxAction gbba( SbViewportRegion(500,500) );
      gbba.apply( geometry );
      SbBox3f bbox = gbba.getBoundingBox();
      m_defaultBbox = gbba.getBoundingBox();
      m_defaultPlane = SbVec3f(0, 0, -1);
      m_defaultScaleFactor = 0.5f;
      m_clipManip->setValue(m_defaultBbox, m_defaultPlane, m_defaultScaleFactor);

      // Report the complexity of the scene
      SoGetPrimitiveCountAction gpca;
      gpca.apply( sceneGeometry );
      int numTris = gpca.getTriangleCount();
      std::cout << "Scene: " << numTris << " triangles.\n";

      // Re-enable intersection and recompute so the viewAll uses the correct scene bbox
      m_intersect->scene = sceneGeometry;
      m_intersect->recompute();

      // Reset the viewer
      m_viewer->viewAll();
      m_viewer->saveHomePosition();
    }
  }
}
