﻿///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**
** Demonstrates using a custom fragment shader to "cut" a volume open.
**
**=======================================================================*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/DicomInfo.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/manips/SoClipPlaneManip.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/engines/SoTimeCounter.h>

#include <DialogViz/dialog/SoMessageDialog.h>

///////////////////////////////////////////////////////////////////////////////
// Data Set
// This file contains the list of dicom we want to load.
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm";

const SbString SHADER_FILENAME = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalCutting/medicalCuttingCustomComputeColor_frag.glsl";

// color map can be edited within Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

///////////////////////////////////////////////////////////////////////////////
// Application state

static SoXtExaminerViewer*       m_viewer;
static SoSeparator*              m_root;
static SoVolumeRenderingQuality* m_shaderNode;
static SoTimeCounter*            m_animationTimer;
static SoInteractiveComplexity*  m_interactComplexity = NULL;

///////////////////////////////////////////////////////////////////////////////

void myKeyPressCB(void *userData, SoEventCallback *eventCB);

SbBool postRenderCallback(void*, SoXtRenderArea*);

///////////////////////////////////////////////////////////////////////////////
int
main(int /*argc*/, char **argv)
{
  // Initialize
  Widget myWindow = SoXt::init(argv[0]); 
  SoVolumeRendering::init();
  SoDialogViz::init();
  InventorMedical::init();

  if (! SbFileHelper::isAccessible(VOLUME_FILENAME)) 
  {
    std::cerr << "Cannot open " << VOLUME_FILENAME.toLatin1() << std::endl;
    new SoMessageDialog( VOLUME_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
    return -1;
  }

  std::cout << "Press 'C' to start and stop the cutting animation\n";

  // Create the scene root
  m_root = new SoSeparator();

  // Create event callback for key press events
  SoEventCallback* activateClippingCB = new SoEventCallback();
    activateClippingCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), myKeyPressCB );
    m_root->addChild(activateClippingCB);

  // Camera
  m_root->addChild( new SoPerspectiveCamera() );

  // Create the volume rendering scene graph
  SoSeparator* volSep = new SoSeparator();
    volSep->setName("VolSep");
    m_root->addChild(volSep);

  // Node to hold the volume data
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = VOLUME_FILENAME;
    MedicalHelper::dicomAdjustVolume( volData );
    volSep->addChild(volData);

  // Add specular to change rendering quality.
  SoMaterial *volumeMaterial = new SoMaterial();
    volumeMaterial->specularColor.setValue(0.26f, 0.26f, 0.26f);
    volSep->addChild(volumeMaterial);

  // Load the colorMap
  SoTransferFunction *pVRTransFunc = new SoTransferFunction();
    pVRTransFunc->loadColormap(COLORMAP_FILENAME);
    volSep->addChild(pVRTransFunc);

  // Set range of values to map into the color map
  SoDataRange *VRRange = new SoDataRange();
    VRRange->min = 176;
    VRRange->max = 476;
    volSep->addChild(VRRange);

  // Change complexity when the camera moves.
  m_interactComplexity = new SoInteractiveComplexity();
    m_interactComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
    m_interactComplexity->fieldSettings.set1Value(1, "SoVolumeRenderingQuality preIntegrated FALSE TRUE");
    m_interactComplexity->fieldSettings.set1Value(2, "SoVolumeRender interpolation LINEAR CUBIC");
    m_interactComplexity->refinementDelay = 0;  // Don't wait before returning to full quality rendering.
    volSep->addChild(m_interactComplexity);
    volSep->addChild(new SoComplexity());       // Complexity node that will be controlled by m_interactComplexity

  // Property node which allows SoVolumeRender to draw High Quality volumes.  
  SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
    volQual->interpolateOnMove = TRUE;
    volQual->deferredLighting = TRUE;
    volQual->preIntegrated = TRUE;
    volQual->ambientOcclusion = TRUE;
    volSep->addChild(volQual);

  // Attach custom shader program to do the "cutting".
  // (SoVolumeRenderingQuality is also a shader node.)
  m_shaderNode = volQual;
  SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceType = SoShaderObject::FILENAME;
    fragmentShader->sourceProgram.setValue( SHADER_FILENAME );
    fragmentShader->addShaderParameter1f( "cuttingPosition", 0.0f );
    m_shaderNode->shaderObject.set1Value( SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader );

  // Node in charge of drawing the volume
  SoVolumeRender* volRender = new SoVolumeRender();
    volRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volRender->opacityThreshold = 0.1f;
    volSep->addChild(volRender);

  // Rest of scene graph
  // Add Open Inventor logo
  m_root->addChild( MedicalHelper::exampleLogoNode() );

  // Scene orientation
  m_root->addChild( new Gnomon() );

  // DICOM annotation
  m_root->addChild( MedicalHelper::exampleDicomAnnotation( IMAGE_FILENAME ) );

  // Instructions
  TextBox* text = new TextBox();
    text->position.setValue(0, -0.98f, 0); // Normalized device coords -1..1
    text->alignmentH = TextBox::CENTER;
    text->alignmentV = TextBox::BOTTOM;
    text->addLine( "Press 'C' to start and stop the cutting animation" );
    m_root->addChild( text );
            
  // Define animation timer (initially NOT running - enabled by key press)
  m_animationTimer = new SoTimeCounter();
    m_animationTimer->ref();
    m_animationTimer->on = FALSE;

  // Viewer
  m_viewer = new SoXtExaminerViewer(myWindow);
    m_viewer->setTitle("Cutting With Shaders");
    m_viewer->setDecoration(FALSE);
    m_viewer->setSize( MedicalHelper::exampleWindowSize() );
    m_viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    m_viewer->setSceneGraph( m_root );
    m_viewer->setPostRenderCallback(postRenderCallback);

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::CORONAL, m_viewer->getCamera(), volData );
    m_viewer->viewAll();
    m_viewer->saveHomePosition();

  // Run then cleanup
  m_viewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  m_animationTimer->unref();
  delete m_viewer;
  m_root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


///////////////////////////////////////////////////////////////////////////////
void myKeyPressCB(void* /*userData*/, SoEventCallback* eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Start/stop cutting animation
  if (SO_KEY_PRESS_EVENT(event,  C))
  {
    SbBool state = m_animationTimer->on.getValue();
    if (state == FALSE) {
      m_animationTimer->on.setValue( TRUE );

      m_interactComplexity->interactiveMode = SoInteractiveComplexity::FORCE_INTERACTION;
    }
    else {
      m_animationTimer->on.setValue( FALSE );

      m_interactComplexity->interactiveMode = SoInteractiveComplexity::AUTO;
    }
  } 

  // Stop cutting animation (doesn't work in viewing mode)
  else if (SO_KEY_PRESS_EVENT(event,  S))
  {
    m_animationTimer->on.setValue( FALSE );
  } 
}


///////////////////////////////////////////////////////////////////////////////
SbBool
postRenderCallback(void*, SoXtRenderArea*)
{
  if (m_animationTimer->on.getValue() == TRUE) 
  {
    // Update the shader parameter
    static float index     = 0.0f;
    static float increment = 0.2f;
    //(dynamic_cast<SoShaderParameter1f*>((dynamic_cast<SoFragmentShader*>(_cutPosition->shaderObject[3]))->parameter[0]))->value = index;
    
    SoFragmentShader*    shader = (SoFragmentShader*)m_shaderNode->shaderObject[SoVolumeShader::FRAGMENT_COMPUTE_COLOR];
    SoShaderParameter1f* param  = (SoShaderParameter1f*)shader->parameter[0];
    param->value = index;

    m_viewer->scheduleRedraw();

    index += increment;
    if ( index >= 10 )
      increment = -0.2f;
    else if ( index <= 0 )
      increment = 0.2f;
  }

  return FALSE;
}
