///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>

#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

const SbString DATAFILE = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalEdge2D/scene.iv";

SoVolumeRenderingQuality* g_vrq;
SoTopLevelDialog *myTopLevelDialog;

///////////////////////////////////////////////////////////////////////////////
class InterfaceAuditor : public SoDialogAuditor
{
public:
  InterfaceAuditor(SoTopLevelDialog* topDialog){ m_top = topDialog; }

private:
  SoTopLevelDialog* m_top;


  void dialogCheckBox(SoDialogCheckBox* cpt)
  {
    SbBool state = cpt->state.getValue();

    if (cpt->auditorID.getValue() == "edgeimage")
    {
      g_vrq->edgeDetect2D = state;
    }
    else if (cpt->auditorID.getValue() == "edgeluminance")
    {
      int bitmask = g_vrq->edgeDetect2DMethod.getValue();
      g_vrq->edgeDetect2DMethod = state ? (bitmask | SoVolumeRenderingQuality::LUMINANCE) : (bitmask&~SoVolumeRenderingQuality::LUMINANCE);
    }
    else if (cpt->auditorID.getValue() == "edgedepth")
    {
      int bitmask = g_vrq->edgeDetect2DMethod.getValue();
      g_vrq->edgeDetect2DMethod = state ? (bitmask | SoVolumeRenderingQuality::DEPTH) : (bitmask&~SoVolumeRenderingQuality::DEPTH);
    }
    else if (cpt->auditorID.getValue() == "edgegrad")
    {
      int bitmask = g_vrq->edgeDetect2DMethod.getValue();
      g_vrq->edgeDetect2DMethod = state ? (bitmask | SoVolumeRenderingQuality::GRADIENT) : (bitmask&~SoVolumeRenderingQuality::GRADIENT);

      SoDialogViz* dv = myTopLevelDialog->searchForAuditorId(SbString("surfacescalar"));
      dv->enable = state;
    }
    else if (cpt->auditorID.getValue() == "surfacescalar")
    {
      g_vrq->surfaceScalarExponent = state ? 8.f : 0.f;
    }
  }

};

/*********************************************************************************/
int main(int, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow)
    return 0;

  SoVolumeRendering::init();
  SoDialogViz::init();
  InventorMedical::init();

  SoSeparator* scene = MedicalHelper::readFile(DATAFILE.toLatin1());
  if (!scene)
  {
    // Use a dialog so the windows remain visible allowing to read the error message.
    new SoMessageDialog( DATAFILE, "Failed to open:", SoMessageDialog::MD_ERROR );
    return 0;
  }

  SbString InterfaceName = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalEdge2D/interface.iv";
  Widget parent = MedicalHelper::buildInterface(myWindow, InterfaceName.getString(), "Viewer", &myTopLevelDialog);
  myTopLevelDialog->addAuditor(new InterfaceAuditor(myTopLevelDialog));

  SoRef<SoSeparator> root = new SoSeparator;

  // Camera
  root->addChild( new SoPerspectiveCamera() );

  // Increase the rendering quality when you stop to move the dataset.
  SoInteractiveComplexity* cplx = new SoInteractiveComplexity;
  cplx->fieldSettings.set1Value(0, "SoVolumeRender interpolation LINEAR CUBIC");

  root->addChild(cplx);
  root->addChild(scene);

  // Define OIV logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Orientation marker
  root->addChild( new Gnomon() );

  SoTransferFunction *localTransferFunction = MedicalHelper::find<SoTransferFunction>(scene);
  if (localTransferFunction)
    localTransferFunction->loadColormap("$OIVHOME/examples/data/Medical/resources/volrenGlow.am");

  g_vrq = MedicalHelper::find<SoVolumeRenderingQuality>(scene);

  if (!g_vrq)
  {
    std::cerr << "SoVolumeRenderingQuality not found" << std::endl;
    new SoMessageDialog( "SoVolumeRenderingQuality not found", "Error", SoMessageDialog::MD_ERROR );
    return 0;
  }

  g_vrq->edgeDetect2DMethod = SoVolumeRenderingQuality::DEPTH;
  g_vrq->surfaceScalarExponent = 8.f;

  // Set up viewer:
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(parent);
  viewer->setSceneGraph(root.ptr());
  viewer->setDecoration(false);
  viewer->setSize( MedicalHelper::exampleWindowSize() );
  viewer->setAntialiasing( 0.5f );
  viewer->show();
  // Adjust camera to standard Coronal/Anterior view
  MedicalHelper::orientView( MedicalHelper::CORONAL, viewer->getCamera() );
  viewer->viewAll();
  viewer->saveHomePosition();

  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;

  root = NULL;

  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}


