///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Sep 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Gamma Correction.
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoTranslation.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/draggers/SoOrthoSliceDragger.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

// Data Set
const SbString IMAGE_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

const SbString SHADER_FILENAME = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalGammaCorrection/gammaCorrection_frag.glsl";

///////////////////////////////////////////////////////////////////////////////
// Application state

static float    m_gammaValue = 2.2f;

static SoFragmentShader* m_fragmentShader = NULL;

static TextBox* m_annotationBox = NULL;

///////////////////////////////////////////////////////////////////////////////

static void updateGammaValue();

static void myKeyPressCB(void *usrData, SoEventCallback *sender);

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    // Create the window, initialize OpenIventor and VolumeViz.
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();	
    InventorMedical::init();

    std::cout << "Press up/down arrow keys to change gamma value.\n";

    // Create the scene graph.
    SoRef<SoSeparator> root = new SoSeparator();

    // Handle some events
    SoEventCallback *myEventCB = new SoEventCallback();
      myEventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), myKeyPressCB);
      root->addChild(myEventCB);

    SoOrthographicCamera* camera = new SoOrthographicCamera();
      root->addChild( camera );

    // Data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      root->addChild(volData);

    // Range of values mapped to colormap (use embedded window center/width if possible)
    SoDataRange* dataRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( dataRange, volData );
      root->addChild( dataRange );

    // Gray scale
    SoTransferFunction* transFunc = new SoTransferFunction();
      transFunc->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( transFunc, volData );
      root->addChild( transFunc );

    // Use full intensity
    SoMaterial* material = new SoMaterial();
      material->diffuseColor.setValue( 1, 1, 1 );
      root->addChild( material );
    
    // Volume shader node to apply gamma correction
    // Our custom shader will generate transparency even if current material parameter(transparency, colorMap, ...) won't.
    // Tell OIV that this shape must be considered as transparent.
    SoVolumeShader *volShader = new SoVolumeShader();
      volShader->generateTransparency = TRUE;
      // Define shader function to be applied
      m_fragmentShader = new SoFragmentShader();
        m_fragmentShader->sourceProgram.setValue( SHADER_FILENAME );
        m_fragmentShader->addShaderParameter1f( "gamma", m_gammaValue ); // Shader parameter
      volShader->shaderObject.set1Value( SoVolumeShader::FRAGMENT_COMPUTE_COLOR, m_fragmentShader );
      root->addChild(volShader);

    SoOrthoSlice *orthoSlice = new SoOrthoSlice();
      orthoSlice->sliceNumber   = volData->data.getSize()[2] / 2;
      orthoSlice->interpolation = SoSlice::MULTISAMPLE_12;
      root->addChild( orthoSlice );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Annotation
    root->addChild( MedicalHelper::buildSliceAnnotation( camera, orthoSlice, &IMAGE_FILENAME ) );

    // Display gamma value and some hints
    SoMaterial* textMat = new SoMaterial();
      textMat->diffuseColor.setValue( 0.5f, 1, 0.5f );
      root->addChild( textMat );
    TextBox* text = new TextBox();
      text->position.setValue( -0.98f, -0.85f, 0 );
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Gamma correction value : 2.2" );
      text->addLine( "Press up/down arrow to modify." );
      root->addChild( text );
      m_annotationBox = text; // For UI updates

    // Set up viewer.
    SoXtExaminerViewer *viewer = new SoXtExaminerViewer(myWindow);
      viewer->setDecoration(FALSE);
      viewer->setViewing( FALSE );  // So that we get the arrow key events instead of the viewer
      viewer->setSize( MedicalHelper::exampleWindowSize() );
      viewer->setSceneGraph(root.ptr());

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::AXIAL, camera, volData );
    viewer->saveHomePosition();

    // Run then cleanup
    viewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
void updateGammaValue()
{
  m_fragmentShader->setShaderParameter1f( "gamma", m_gammaValue);
  SbString str;
  str.sprintf( "Gamma correction value : %.1f", m_gammaValue );
  m_annotationBox->setLine( str );
}

///////////////////////////////////////////////////////////////////////////////
void myKeyPressCB(void* /*usrData*/, SoEventCallback* sender)
{
    SoKeyboardEvent *evt = (SoKeyboardEvent*)sender->getEvent();
    if (SO_KEY_PRESS_EVENT(evt, UP_ARROW))
    {
      m_gammaValue += 0.1f;
      updateGammaValue();
    }
    else if (SO_KEY_PRESS_EVENT(evt, DOWN_ARROW))
    {
      m_gammaValue -= 0.1f;
      if (m_gammaValue < 0) m_gammaValue = 0;
      updateGammaValue();
    }
}
