///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to use a SoVolumeRenderingQuality property node
for lighting a volume.
author : Benjamin Grange
Updaded by Pascal Estrade (Sep 2014)
September 2005
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoRotationXYZ.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

#define VOLUME_FILENAME "$OIVHOME/examples/data/Medical/files/head.ldm"

#define COLORMAP_FILENAME "$OIVHOME/examples/data/Medical/resources/volrenGlow.am"

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
    // Initialize and create window
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator();

    // Camera (the viewer will automatically control this camera)
    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
      root->addChild( camera );

    // The Directional light that will highlight the volume.
    // You can use ivTune to change the position of this light.
    // The viewer will _not_ automatically control this light.
    // So this light rotates with the volume (instead of with the camera).
    SoDirectionalLight* light = new SoDirectionalLight();
      light->direction.setValue( -1, 0, 0 );
      root->addChild( light );

    SoSeparator* volSep = new SoSeparator();
      root->addChild( volSep );

    // Node to hold the volume data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILENAME;
      volSep->addChild( volData );

    // Load the colorMap from a file
    SoTransferFunction* transFunc = new SoTransferFunction;
      transFunc->loadColormap(COLORMAP_FILENAME);
      transFunc->minValue = 30;
      volSep->addChild( transFunc );

    //Set the ambient, diffuse, transparency, specular and shininess of the material
    SoMaterial* volMat = new SoMaterial();
      volMat->ambientColor.setValue( 0.3f, 0.3f, 0.3f );
      volMat->specularColor.setValue( 0.8f, 0.8f, 0.8f );
      volMat->shininess.setValue( 0.5 );
      volSep->addChild( volMat );

    // Change complexity when the dataset moves.
    SoInteractiveComplexity *intercatComplexity = new SoInteractiveComplexity();
      intercatComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
      intercatComplexity->fieldSettings.set1Value(2, "SoVolumeRender interpolation LINEAR CUBIC");
      intercatComplexity->refinementDelay = 0;
      volSep->addChild(intercatComplexity);
      volSep->addChild(new SoComplexity());

    // Property node which allows SoVolumeRender to draw lighted volumes
    SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality;
      volQuality->deferredLighting = TRUE;
      volQuality->preIntegrated = TRUE;
      volSep->addChild( volQuality );

    // Node in charge of drawing the volume
    SoVolumeRender* volRender = new SoVolumeRender;
      volRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
      volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volSep->addChild( volRender );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation
    // Note: The single light in this scene would only ever illuminate one side
    //       of the gnomon.  We could create a separate light, but it's easier
    //       to just turn off lighting for the gnomon.
    SoSeparator* gnomonSep = new SoSeparator();
      root->addChild( gnomonSep );
      SoLightModel* lmodel = new SoLightModel();
        lmodel->model = SoLightModel::BASE_COLOR;
        lmodel->setOverride( TRUE );
      gnomonSep->addChild( lmodel );
      gnomonSep->addChild( new Gnomon() );

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 );
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Note: Light rotates with volume (not with camera)" );
      root->addChild( text );

    // Set up viewer:
    SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myWindow);
      myViewer->setSceneGraph(root.ptr());
      myViewer->setTitle( "Simple lighted volume" );
      myViewer->setDecoration( FALSE );
      myViewer->setSize( MedicalHelper::exampleWindowSize() );
      myViewer->setHeadlight(FALSE); // Turn off the viewer's automatic light!

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, camera, volData );
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

