///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShadowGroup.h>

#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoFenceSlice.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include <cstdlib>

///////////////////////////////////////////////////////////////////////////////
SbString dataFile = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalSimpleShadows/scene.iv";
SbString interfaceName = "$OIVHOME/examples/source/Medical/Rendering/Techniques/medicalSimpleShadows/interface.iv";
#define COLORMAPFILENAME "$OIVHOME/examples/data/Medical/resources/volrenGlow.am"

/*********************************************************************************/
int main(int, char **argv)
{
    SoTopLevelDialog *topLevelDialog;
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow)
        return 0;

    SoVolumeRendering::init();
    SoDialogViz::init();
    InventorMedical::init();

    SoSeparator* scene = MedicalHelper::readFile(dataFile.toLatin1());
    if(!scene)
    {
        new SoMessageDialog( dataFile, "Unable to open:", SoMessageDialog::MD_ERROR );
        //std::cerr << dataFile.toStdString() << "not found" << std::endl;
        return EXIT_FAILURE;
    }

    Widget parent = MedicalHelper::buildInterface(myWindow, interfaceName.toLatin1(), "Viewer", &topLevelDialog);

    SoRef<SoSeparator> root = new SoSeparator;

    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
    root->addChild( camera );

    root->addChild(scene);

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    SoTransferFunction* tf = MedicalHelper::find<SoTransferFunction>(scene);
    tf->loadColormap(COLORMAPFILENAME);

    SoShadowGroup* sg = MedicalHelper::find<SoShadowGroup>(scene);
    if ( !sg )
        return EXIT_FAILURE;

    SoDialogRealSlider *realSlider;
    realSlider = (SoDialogRealSlider *)topLevelDialog->searchForAuditorId("intensity");
    sg->intensity.connectFrom(&realSlider->value);

    SoDialogIntegerSlider *intSlider;
    intSlider = (SoDialogIntegerSlider *)topLevelDialog->searchForAuditorId("smoothness");
    sg->smoothFactor.connectFrom(&intSlider->value);

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Simple Shadows");
    myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    myViewer->setDecoration( FALSE );
    //myViewer->setHeadlight( FALSE );
    myViewer->setSize( MedicalHelper::exampleWindowSize() );

    // Find the viewer's headlight and give it a twist so we can see the shadow better
    SoDirectionalLight* light = MedicalHelper::find<SoDirectionalLight>(myViewer->getSceneManager()->getSceneGraph(),"");
    if (light != NULL)
      light->direction.setValue( 0.391487f, 0.18071f, -0.902265f );

    // Adjust camera
    myViewer->viewAll();
    MedicalHelper::dollyZoom( 1.5, camera );  // viewAll is too conservative, let's get closer.
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    topLevelDialog->close();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}
