#ifndef TIME_SERIES_CACHED_GENERIC_READER
#define TIME_SERIES_CACHED_GENERIC_READER

#include <VolumeViz/readers/SoVRGenericFileReader.h>

#include "TimeSeriesVolumeReader.h"

/**
 * TimeSeries reader that uses SoVRGenericFileReader internally
 * and manages a cache of the buffers returned by getSubSlice()
 */
class TimeSeriesCachedGenericReader : public TimeSeriesVolumeReader
{
public:
  TimeSeriesCachedGenericReader()
  : TimeSeriesVolumeReader()
  , m_size(-1.0f, -1.0f, -1.0f, 1.0f, 1.0f, 1.0f)
  , m_dim(0, 0, 0)
  , m_type(SoVolumeData::UNSIGNED_BYTE)
  , m_headerSize(0)
  , m_sliceCacheMutex(new SbThreadMutex)
  {}

  /**
   * Since the format is raw data, it is necessary to specify
   * the dimension and data size of the volume.
   * You can also specify a header size to add support for your own file format.
   * (copied from SoVRGenericFileReader)
   */
  void setDataChar(const SbBox3f& size, const SoDataSet::DataType& type, const SbVec3i32& dim, int headerSize = 0)
  {
    m_size = size;
    m_type = type;
    m_dim = dim;
    m_headerSize = headerSize;
  }

  /** Redefined to enable caching */
  virtual void getSubSlice(const SbBox2i32& subSlice, int sliceNumber, void* data)
  {
    m_sliceCacheMutex->lock();
    SliceCacheValue& cacheValue = m_sliceCache[SliceCacheKey(getCurrentTimeStep(), sliceNumber, subSlice)];
    if (cacheValue.buffer.ptr() == nullptr)
    {
      const size_t bufferSize = (subSlice.getMax()[0] - subSlice.getMin()[0] + 1) *
        (subSlice.getMax()[1] - subSlice.getMin()[1] + 1) * getNumBytesPerDatum();
      if (bufferSize != 0)
      {
        cacheValue.buffer = new SoCpuBufferObject;
        cacheValue.buffer->setSize(bufferSize);
        TimeSeriesVolumeReader::getSubSlice(subSlice, sliceNumber, cacheValue.buffer->map(SoCpuBufferObject::SET));
        cacheValue.buffer->unmap();
      }
    }
    m_sliceCacheMutex->unlock();

    if (cacheValue.buffer.ptr() != nullptr)
    {
      memcpy(data, cacheValue.buffer->map(SoCpuBufferObject::READ_ONLY), cacheValue.buffer->getSize());
      cacheValue.buffer->unmap();
    }
  }

protected:
  /** Redefined from TimeSeriesVolumeReader to specify the readers to use internally */
  virtual SoVolumeReader* createReader(const SbString& filename)
  {
    SoVRGenericFileReader* reader = new SoVRGenericFileReader;
    reader->setFilename(filename);
    reader->setDataChar(m_size, m_type, m_dim, m_headerSize);
    return reader;
  }

  ~TimeSeriesCachedGenericReader()
  {
    delete m_sliceCacheMutex;
  }

  SbBox3f m_size;
  SbVec3i32 m_dim;
  SoDataSet::DataType m_type;
  int m_headerSize;

private:
  struct SliceCacheKey
  {
    SliceCacheKey(size_t _timeStep, int _sliceNumber, const SbBox2i32& _subSlice)
    : timeStep(_timeStep)
    , sliceNumber(_sliceNumber)
    , subSlice(_subSlice)
    {}

    size_t timeStep;
    int sliceNumber;
    SbBox2i32 subSlice;

    bool operator<(const SliceCacheKey& other) const
    {
      if (timeStep != other.timeStep)
        return timeStep < other.timeStep;

      if (sliceNumber != other.sliceNumber)
        return sliceNumber < other.sliceNumber;

      if (subSlice.getMin()[0] != other.subSlice.getMin()[0])
        return subSlice.getMin()[0] < other.subSlice.getMin()[0];

      if (subSlice.getMin()[1] != other.subSlice.getMin()[1])
        return subSlice.getMin()[1] < other.subSlice.getMin()[1];

      if (subSlice.getMax()[0] != other.subSlice.getMax()[0])
        return subSlice.getMax()[0] < other.subSlice.getMax()[0];

      if (subSlice.getMax()[1] != other.subSlice.getMax()[1])
        return subSlice.getMax()[1] < other.subSlice.getMax()[1];

      return false;
    }
  };

  struct SliceCacheValue
  {
    SoRef<SoCpuBufferObject> buffer;
  };

  typedef std::map<SliceCacheKey, SliceCacheValue> SliceCacheMap;

  /**
   * A map acting as a cache for the subslice buffers.
   * It is limitless because we know that everything fits in memory in this demo,
   * however it would need to have some limits in a more general case.
   */
  SliceCacheMap m_sliceCache;

  /** For theadsafe map access */
  SbThreadMutex* m_sliceCacheMutex;
};

#endif /* TIME_SERIES_CACHED_GENERIC_READER */
