#ifndef DEMO_SCENE_H
#define DEMO_SCENE_H

#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/nodes/SoROI.h>
#include <LDM/SoLDMGlobalResourceParameters.h>
#include <LDM/SoLDM.h>

#include "TimeSeriesCachedGenericReader.h"

#include <iostream>

const std::vector<std::string>& getFilenames()
{
  static std::vector<std::string> filenames;
  if (!filenames.empty())
    return filenames;

  static const char* FILE_DIR = "$OIVHOME/examples/data/Medical/";
  static const char* FILE_PATTERN = "files/4D/bob256x256x32-%03d.raw";
  static const int FILE_FIRST = 0;
  static const int FILE_LAST = 19;

  const SbString directory(FILE_DIR);
  SbString filename;
  for (int i = FILE_FIRST; i <= FILE_LAST; ++i)
  {
    filename.sprintf(FILE_PATTERN, i);
    filenames.push_back((directory + filename).getString());
  }

  return filenames;
}

class TimeSeriesScene
{
public:
  TimeSeriesScene()
  : m_sceneGraph(new SoSeparator())
  {
    // Volume rendering sub-graph
    SoSeparator* volSep = new SoSeparator;
    m_sceneGraph->addChild(volSep);

    m_volumeData = new SoVolumeData;
    TimeSeriesCachedGenericReader* reader = new TimeSeriesCachedGenericReader;
    // These are "raw" data files so the dimensions and header size must be specified.
    reader->setDataChar(SbBox3f(0, 0, 0, 255, 255, 64), SoDataSet::SIGNED_SHORT, SbVec3i32(256, 256, 32), 13);
    reader->setFilenames(getFilenames());
    m_volumeData->setReader(*reader, TRUE);
    m_volumeData->ldmResourceParameters.getValue()->tileDimension = SbVec3i32(256, 256, 256);
    volSep->addChild(m_volumeData);

    // Restrict data range to get a nice image
    SoDataRange* volRange = new SoDataRange;
    volRange->min = -32750;
    volRange->max = -32670;
    volSep->addChild(volRange);

    // Load color map
    SoTransferFunction* volTF = new SoTransferFunction;
    volTF->loadColormap("$OIVHOME/examples/data/Medical/resources/volrenGlow.am");
    volSep->addChild(volTF);

    // Display volume at full intensity
    SoMaterial* volMat = new SoMaterial;
    volMat->diffuseColor.setValue(1.0f, 1.0f, 1.0f);
    volSep->addChild(volMat);

    // Volume rendering parameters
    SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality;
    volQual->preIntegrated = TRUE;
    volQual->lighting = TRUE;
    volQual->lightingModel = SoVolumeRenderingQuality::OPENGL;
    volQual->surfaceScalarExponent = 5;
    volQual->interpolateOnMove = TRUE;
    volQual->deferredLighting = FALSE;
    volSep->addChild(volQual);

    SoVolumeRender* volRend = new SoVolumeRender;
    volRend->numSlicesControl = SoVolumeRender::AUTOMATIC;
    volRend->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volRend->opacityThreshold = 0.1f;
    volSep->addChild(volRend);
  }

  SoSeparator* getSceneGraph() const
  {
    return m_sceneGraph.ptr();
  }

  SoVolumeData* getVolumeData() const
  {
    return m_volumeData;
  }

  void setTimeStep(int step)
  {
    // Change step in reader
    TimeSeriesVolumeReader* reader = static_cast<TimeSeriesVolumeReader*>(getVolumeData()->getReader());
    reader->setTimestep(step);

    // Force an update on all the volume
    const SbBox3i32 volDataBox(SbVec3i32(0, 0, 0), getVolumeData()->data.getSize() - SbVec3i32(1, 1, 1));
    getVolumeData()->updateRegions(&volDataBox, 1);
  }

  int getTimeStep() const
  {
    TimeSeriesVolumeReader* reader = static_cast<TimeSeriesVolumeReader*>(getVolumeData()->getReader());
    return static_cast<int>(reader->getCurrentTimeStep());
  }

  int getNumTimeSteps() const
  {
    TimeSeriesVolumeReader* reader = static_cast<TimeSeriesVolumeReader*>(getVolumeData()->getReader());
    return static_cast<int>(reader->getNumTimeSteps());
  }

private:
  // Root of scene graph
  SoRef<SoSeparator> m_sceneGraph;
  SoVolumeData* m_volumeData;
};

#endif /* DEMO_SCENE_H */
