#ifndef TIME_SERIES_VOLUME_READER
#define TIME_SERIES_VOLUME_READER

#include <VolumeViz/readers/SoVolumeReader.h>
#include <VolumeViz/readers/SoVRLdmFileReader.h>
#include <VolumeViz/nodes/SoVolumeData.h>

/**
 * A volume reader for timeseries that uses multiple readers internally, for each time step.
 */
class TimeSeriesVolumeReader : public SoVolumeReader
{
public:
  TimeSeriesVolumeReader()
  : SoVolumeReader()
  , m_mutex(new SbThreadMutex)
  , m_currentTimestep(0)
  , m_numTimesteps(0)
  {}

  /** Standard reader methods redefined */

  virtual int setFilename(const SbString& filename)
  {
    return m_readers[m_currentTimestep]->setFilename(filename);
  }

  virtual SoVolumeReader::ReadError getDataChar(SbBox3f& size, SoVolumeData::DataType& type, SbVec3i32& dim)
  {
    return m_readers[m_currentTimestep]->getDataChar(size, type, dim);
  }

  virtual SbBool getTileSize(SbVec3i32& size)
  {
    return m_readers[m_currentTimestep]->getTileSize(size);
  }

  virtual SbBool getMinMax(int& min, int& max)
  {
    return m_readers[m_currentTimestep]->getMinMax(min, max);
  }

  virtual SbBool getMinMax(double& min, double& max)
  {
    return m_readers[m_currentTimestep]->getMinMax(min, max);
  }

  virtual SbBool getMinMax(int64_t& min, int64_t& max)
  {
    return m_readers[m_currentTimestep]->getMinMax(min, max);
  }

  virtual SoBufferObject* readTile(int index, const SbBox3i32& tilePosition)
  {
    m_mutex->lock();
    SoBufferObject* bo = m_readers[m_currentTimestep]->readTile(index, tilePosition);
    m_mutex->unlock();
    return bo;
  }

  virtual void getSubSlice(const SbBox2i32& subSlice, int sliceNumber, void* data)
  {
    m_mutex->lock();
    m_readers[m_currentTimestep]->getSubSlice(subSlice, sliceNumber, data);
    m_mutex->unlock();
  }

  virtual SbBool isThreadSafe()
  {
    return m_readers[m_currentTimestep]->isThreadSafe();
  }

  virtual SbBool isDataConverted() const
  {
    return m_readers[m_currentTimestep]->isDataConverted();
  }

  virtual SbVec2d getTileMinMax(int fileId) const
  {
    const_cast<TimeSeriesVolumeReader*>(this)->lockFileMutex();
    return m_readers[m_currentTimestep]->getTileMinMax(fileId);
    const_cast<TimeSeriesVolumeReader*>(this)->unlockFileMutex();
  }

  virtual int getNumBytesPerDatum()
  {
    return m_readers[m_currentTimestep]->getNumBytesPerDatum();
  }


  /** Timestep methods */

  void setTimestep(int timestep)
  {
    m_mutex->lock();

    m_readers[m_currentTimestep]->closeAllHandles();
    m_currentTimestep = timestep % m_numTimesteps;

    SbBox3f size;
    SoVolumeData::DataType type;
    SbVec3i32 dim;
    getDataChar(size, type, dim);

    m_mutex->unlock();
  }

  size_t getCurrentTimeStep() const
  {
    return m_currentTimestep;
  }

  size_t getNumTimeSteps() const
  {
    return m_numTimesteps;
  }

  void setFilenames(const std::vector<std::string>& filenames)
  {
    m_filenames = filenames;
    m_readers.resize(m_filenames.size());
    for (size_t i = 0; i < m_filenames.size(); ++i)
      m_readers[i] = createReader(m_filenames[i]);
    m_numTimesteps = m_filenames.size();
  }

  const std::vector<std::string>& getFilenames() const
  {
    return m_filenames;
  }

protected:
  ~TimeSeriesVolumeReader()
  {
    delete m_mutex;
  }

  /**
   * By default, this reader uses SoVRLdmFileReader instances for each timestep reader.
   * Can be overloaded by derived classes to use a specific reader.
   */
  virtual SoVolumeReader* createReader(const SbString& filename)
  {
    SoVolumeReader* reader = new SoVRLdmFileReader;
    reader->setFilename(filename);
    return reader;
  }

private:
  SbThreadMutex* m_mutex;
  size_t m_currentTimestep;
  size_t m_numTimesteps;
  std::vector<SoVolumeReader*> m_readers;
  std::vector<std::string> m_filenames;
};

#endif /* TIME_SERIES_VOLUME_READER */
