///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author		:  Mike Heck (November 2014)
** Modified		:  Pascal Estrade (June 2015)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Animate volume time series with volumeViz.
Description : Main
-------------------------------------------------------------------------*/
// The default data comes from the SPEC Viewperf 12 benchmark.
// https://www.spec.org/gwpg/gpc.static/vp12info.html


#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoOrthographicCamera.h>

#include <Inventor/sensors/SoTimerSensor.h>

// Medical headers
#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/TextBox.h>

#include <iostream>

#include "TimeSeriesScene.h"

////////////////////////////////////////////////////////////////////////
static SoTimerSensor* g_timerSensor = nullptr;
SoSceneManager* g_sceneManager = nullptr;
TimeSeriesScene* g_timeSeriesScene = nullptr;

////////////////////////////////////////////////////////////////////////
void
timerCallback(void* data, SoSensor*)
{
  TimeSeriesScene* timeSeriesScene = static_cast<TimeSeriesScene*>(data);
  const int animationStep = (timeSeriesScene->getTimeStep() + 1) % timeSeriesScene->getNumTimeSteps();
  timeSeriesScene->setTimeStep(animationStep);
  std::cout << animationStep << "  \r";
}

void
enableAnimation(bool enable)
{
  const bool isAnimating = g_timerSensor->isScheduled();
  if (enable == isAnimating)
    return;

  if (enable)
  {
    // During animation, we need to force interaction
    // or else the scene manager thinks we are in STILL mode
    // and some optimizations won't be applied.
    g_sceneManager->setAutoInteractiveMode(FALSE);
    g_sceneManager->setInteractive(TRUE);
    g_timerSensor->schedule();
  }
  else
  {
    g_sceneManager->setAutoInteractiveMode(TRUE);
    g_sceneManager->setInteractive(FALSE);
    g_timerSensor->unschedule();
  }
}

///////////////////////////////////////////////////////////////////////
void
keyEventCB(void* userData, SoEventCallback* node)
{
  TimeSeriesScene* timeSeriesScene = static_cast<TimeSeriesScene*>(userData);

  const SoEvent* evt = node->getEvent();
  if (SO_KEY_PRESS_EVENT(evt, A))
  {
    // Enable/disable animation
    enableAnimation(!g_timerSensor->isScheduled());
    node->setHandled();
  }
  else if (SO_KEY_PRESS_EVENT(evt, PAGE_UP))
  {
    // Next time step
    const int animationStep = (timeSeriesScene->getTimeStep() + 1) % timeSeriesScene->getNumTimeSteps();
    timeSeriesScene->setTimeStep(animationStep);
    std::cout << animationStep << "  \r";
    node->setHandled();
  }
  else if (SO_KEY_PRESS_EVENT(evt, PAGE_DOWN))
  {
    // Previous time step
    int animationStep = timeSeriesScene->getTimeStep() - 1;
    animationStep = (animationStep < 0) ? timeSeriesScene->getNumTimeSteps() - 1 : animationStep;
    timeSeriesScene->setTimeStep(animationStep);
    std::cout << animationStep << "  \r";
    node->setHandled();
  }
}

SoNode*
buildScenegraph()
{
  SoSeparator* root = new SoSeparator;

  // Camera (must have explicit camera to use TextBox node)
  root->addChild(new SoOrthographicCamera);

  // Time Series sub scenegraph (contains volume rendering nodes)
  g_timeSeriesScene = new TimeSeriesScene;
  root->addChild(g_timeSeriesScene->getSceneGraph());

  // Create a cube with the geometric size of the volume
  SbVec3f boxMin, boxMax;
  g_timeSeriesScene->getVolumeData()->extent.getValue().getBounds(boxMin, boxMax);
  root->addChild(MedicalHelper::createBoundingBox(SbBox3f(boxMin, boxMax)));

  // Define Open Inventor logo
  root->addChild(MedicalHelper::exampleLogoNode());

  // Instructions
  TextBox* textBox = new TextBox;
  textBox->position.setValue(-0.99f, 0.99f, 0);
  textBox->addLine("Press 'A' to start/stop animation.");
  textBox->addLine("Press PageUp/PageDown to step through animation.");
  root->addChild(textBox);

  // Key events
  SoEventCallback* eventCB = new SoEventCallback;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), keyEventCB, g_timeSeriesScene);
  root->addChild(eventCB);

  return root;
}

////////////////////////////////////////////////////////////////////////
int
main(int /*argc*/, char** argv)
{
  std::cout << "Press 'A' to start/stop animation.\n";
  std::cout << "Press PageUp/PageDown to step through animation.\n\n";

  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);
  SoVolumeRendering::init();
  InventorMedical::init();

  SoRef<SoNode> root = buildScenegraph();

  // Create a viewer
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);
  SoXt::setWidgetSize(myWindow, SbVec2s(1024, 768));
  viewer->setTitle("Volume time series");
  viewer->setDecoration(false);
  viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  viewer->setSceneGraph(root.ptr());
  viewer->viewAll();
  viewer->saveHomePosition();
  viewer->show();
  SoXt::show(myWindow);

  // Animation timer
  g_sceneManager = viewer->getSceneManager();
  g_timerSensor = new SoTimerSensor(timerCallback, g_timeSeriesScene);
  g_timerSensor->setInterval(0.02f);
  enableAnimation(TRUE);

  // Loop, then cleanup
  SoXt::mainLoop();

  delete g_timerSensor;
  g_timerSensor = nullptr;
  delete g_timeSeriesScene;
  g_timeSeriesScene = nullptr;
  g_sceneManager = nullptr;

  root = nullptr;
  delete viewer;

  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();

  return 0;
}
