﻿///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (Oct 2015)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how change camera orientation to follow Anatomical
          Orientation.  Shows how to animate camera to new view.
Description : Main
-------------------------------------------------------------------------*/
// Inventor Headers
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/SbTime.h>
#include <Inventor/fields/SoSFTime.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/SoDialogVizAll.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/manips/SoROIManip.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////
// Application state
static SoXtExaminerViewer* m_viewer3D = NULL;
static SoInteractiveComplexity* m_intercatComplexity = NULL;
static SoVolumeData* m_volData = NULL;

static SbVec3f m_oldCamPosition;
static SbVec3f m_oldCamFocalPt;
static SbRotation m_oldCamOrientation;
static float m_oldCamDist;
static SbVec3f m_newCamPosition;
static SbVec3f m_newCamFocalPt;
static float m_newCamDist;

static SbRotation m_newCamOrientation;
static SoFieldSensor *m_seekAnimationSensor;
static SoSFTime *m_viewerRealTime;
static SbTime m_seekStartTime;
static int m_seekAnimTime;

///////////////////////////////////////////////////////////////////////////////
// Data Set
// This file contains the list of dicom we want to load.
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm";

// color map can be edited within Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

///////////////////////////////////////////////////////////////////////////////

static void moveCamera(SoXtExaminerViewer *viewer, const SoCamera* pViewPt);

static void axisView(SoXtExaminerViewer *viewer, int axis);

static void handleEvent(void *pUserData, SoEventCallback *pEventNode);

static void interpolateOrbitAnimation(float t);

static void cameraAnimationSensorCB(void* data, SoSensor* sensor);

///////////////////////////////////////////////////////////////////////////////

int main(int /*argc*/, char **argv)
{
    Widget myWindow = SoXt::init(argv[0]);
    if (! SbFileHelper::isAccessible(VOLUME_FILENAME)) {
      SoDialogViz::init();
      new SoMessageDialog( VOLUME_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    SoVolumeRendering::init();
    InventorMedical::init();

    std::cout << "Press letter key to animate to view.\n";
    std::cout << "  A - Axial view\n";
    std::cout << "  C - Coronal view\n";
    std::cout << "  S - Sagittal view\n";
    std::cout << "Press same letter _again_ for opposite view.\n";

    // Scene graph root
    SoRef<SoSeparator> root = new SoSeparator();

    // Camera
    SoOrthographicCamera* camera = new SoOrthographicCamera();
      root->addChild(camera);
    
    // Handle some events
    SoEventCallback* eventNode = new SoEventCallback();
      eventNode->addEventCallback( SoKeyboardEvent::getClassTypeId(), handleEvent );
      root->addChild(eventNode);
    
    // Create the volume rendering scene graph
    {
        // Volume Rendering Separator =========================================
        SoSeparator *rootVR = new SoSeparator();
          rootVR->setName("root_VR");
          root->addChild(rootVR);

        // Node to hold the volume data
        m_volData = new SoVolumeData();
          m_volData->fileName = VOLUME_FILENAME;
          MedicalHelper::dicomAdjustVolume( m_volData );
          rootVR->addChild(m_volData);

        // Add specular to change rendering quality.
        SoMaterial *volumeMaterial = new SoMaterial();
          volumeMaterial->specularColor.setValue(0.26f, 0.26f, 0.26f);
          rootVR->addChild(volumeMaterial);

        // Load the colorMap
        SoTransferFunction *pVRTransFunc = new SoTransferFunction();
          pVRTransFunc->loadColormap(COLORMAP_FILENAME);
          rootVR->addChild(pVRTransFunc);

        // Set range of values to map into the color map
        SoDataRange *VRRange = new SoDataRange();
          VRRange->min = 176;
          VRRange->max = 476;
          rootVR->addChild(VRRange);

        // Change complexity when the camera moves.
        m_intercatComplexity = new SoInteractiveComplexity();
          // Decrease the number of samples
          m_intercatComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
          m_intercatComplexity->fieldSettings.set1Value(1, "SoVolumeRenderingQuality preIntegrated FALSE TRUE");
          // Decrease interpolation quality.
          m_intercatComplexity->fieldSettings.set1Value(2, "SoVolumeRender interpolation LINEAR CUBIC");
          // Don't wait before returning to full quality rendering.
          m_intercatComplexity->refinementDelay = 0;
          rootVR->addChild(m_intercatComplexity);
          rootVR->addChild(new SoComplexity());

        // Property node which allows SoVolumeRender to draw High Quality volumes.  
        SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality();
          pVRVolQuality->interpolateOnMove = TRUE;
          pVRVolQuality->deferredLighting = TRUE;
          pVRVolQuality->preIntegrated = TRUE;
          pVRVolQuality->ambientOcclusion = TRUE;
          rootVR->addChild(pVRVolQuality);

        // Node in charge of drawing the volume
        SoVolumeRender *pVolRender = new SoVolumeRender();
          pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
          pVolRender->lowScreenResolutionScale = 2;
          pVolRender->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
          pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
          rootVR->addChild(pVolRender);

        // Medical Gnomon.
        Gnomon *gnomon = new Gnomon();
          root->addChild(gnomon);

        // Dicom annotation
        root->addChild( MedicalHelper::exampleDicomAnnotation( IMAGE_FILENAME ) );

        // Define Open Inventor logo
        root->addChild( MedicalHelper::exampleLogoNode() );

        TextBox* text = new TextBox();
          text->position.setValue( -0.98f, -0.6f, 0 );
          text->alignmentH = TextBox::LEFT;
          text->alignmentV = TextBox::BOTTOM;
          text->border     = TRUE;
          text->addLine( "'A' for Axial view" );
          text->addLine( "'C' for Coronal view" );
          text->addLine( "'S' for Sagittal view" );
          root->addChild( text );
    }

    // Create sensor to manage animation
    m_seekAnimationSensor = new SoFieldSensor();
    m_seekAnimationSensor->setFunction(cameraAnimationSensorCB, NULL);
    m_viewerRealTime = (SoSFTime*)SoDB::getGlobalField("realTime");
    m_seekAnimTime = 1;

    // Set up viewer :
    m_viewer3D = new SoXtExaminerViewer(myWindow);
    m_viewer3D->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    m_viewer3D->setViewing(FALSE);
    m_viewer3D->setDecoration(FALSE);
    m_viewer3D->setSize( MedicalHelper::exampleWindowSize() );
    m_viewer3D->setTitle("Anatomical Views");
    m_viewer3D->setSceneGraph(root.ptr());

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::CORONAL, camera, m_volData );
    m_viewer3D->saveHomePosition();

    // Run then cleanup
    m_viewer3D->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete m_viewer3D;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Handle user events
void handleEvent(void* /*pUserData*/, SoEventCallback* pEventNode)
{
    const SoEvent* evt = pEventNode->getEvent();

    // Sagittal view
    // Note the viewer handles key 'S' in viewing mode.
    if (SO_KEY_PRESS_EVENT(evt, S)) {
        axisView( m_viewer3D, MedicalHelper::SAGITTAL );
    }

    // Coronal view
    if (SO_KEY_PRESS_EVENT(evt, C)) {
        axisView( m_viewer3D, MedicalHelper::CORONAL );
    }

    // Axial/Transverse view
    if (SO_KEY_PRESS_EVENT(evt, A) || SO_KEY_PRESS_EVENT(evt, T)) {
        axisView( m_viewer3D, MedicalHelper::AXIAL );
    }
}

///////////////////////////////////////////////////////////////////////////////
// Compute a new viewpoint on the specified axis
void axisView(SoXtExaminerViewer *viewer, int axis)
{
    // Start with a copy of the current camera
    SoCamera* curCamera = viewer->getCamera();
    SoRef<SoCamera> newCamera = (SoCamera*)curCamera->copy();

    // Get the correct orientation for the specified view
    if (axis == MedicalHelper::SAGITTAL) { // Sagittal View = X axis
      MedicalHelper::orientView( MedicalHelper::SAGITTAL, newCamera.ptr(), m_volData );
    }
    else if (axis == MedicalHelper::AXIAL) { // Axial/Transverse View = Z axis.
      MedicalHelper::orientView( MedicalHelper::AXIAL, newCamera.ptr(), m_volData );
    }
    else { // Coronal View = Y axis
      MedicalHelper::orientView( MedicalHelper::CORONAL, newCamera.ptr(), m_volData );
    }

    // If already in the default orientation for this view...
    const SbRotation& newOrientation = newCamera->orientation.getValue();
    if (newOrientation == curCamera->orientation.getValue()) {
      // Flip to opposite direction
      newCamera->orientation = SbRotation( SbVec3f(0,1,0), (float)M_PI ) * newOrientation;
    }

    // Animate to new settings
    moveCamera(viewer, newCamera.ptr());
}

///////////////////////////////////////////////////////////////////////////////
// Animate camera to new position/direction
void moveCamera(SoXtExaminerViewer *viewer, const SoCamera* pViewPt)
{
    SoCamera* pCam = viewer->getCamera();
    pCam->focalDistance = pViewPt->focalDistance.getValue();
    m_oldCamPosition = pCam->position.getValue();
    m_oldCamOrientation = pCam->orientation.getValue();
    m_oldCamDist = pCam->focalDistance.getValue();
    m_newCamPosition = pViewPt->position.getValue();
    m_newCamOrientation = pViewPt->orientation.getValue();
    m_newCamDist = pViewPt->focalDistance.getValue();
    
    // Get the old and new camera focal points
    SbMatrix mx;
    mx = pCam->orientation.getValue();
    SbVec3f oldViewVec(-mx[2][0], -mx[2][1], -mx[2][2]);
    m_oldCamFocalPt = pCam->position.getValue() + pViewPt->focalDistance.getValue() * oldViewVec;

    mx = m_newCamOrientation;
    SbVec3f newViewVec(-mx[2][0], -mx[2][1], -mx[2][2]);
    m_newCamFocalPt = m_newCamPosition + m_newCamDist * newViewVec;

    // Schedule sensor
    if (!m_seekAnimationSensor->getAttachedField()) {
        m_seekAnimationSensor->attach((SoSFTime*)SoDB::getGlobalField("realTime"));
        m_seekAnimationSensor->schedule();
    }
    // Save animation start time
    m_seekStartTime = m_viewerRealTime->getValue();
}

//------------------------------------------------------------------------------
// Usually called from animation timer callback to update camera state.
// The parameter "t" should vary linearly from 0 to 1.
// Animation can be forward or backward.
//
// Parameters are taken from global variables for now.
// Should be re-factored as a class.
//
// Simple interpolation of camera position works for "seek" animation
// in the viewer, but gives very bad results for general camera animation
// because (for example) the camera will swoop in very close to the object
// to get from a front view to a back view.  A better solution is to
// interpolate the focal point and focal distance, then compute each new
// camera position from that data.  Essentially interpolated "orbit".
void interpolateOrbitAnimation(float t)
{
    SoCamera *camera = m_viewer3D->getCamera();

    // If this is the last step of the animation...
    if (t == 1.0f) {
      // Make sure we end up _exactly_ at the final camera position/orientation.
      // This is important because the application may need to compare the camera
      // state with a defined viewpoint.
      camera->orientation   = m_newCamOrientation;
      camera->position      = m_newCamPosition;
      camera->focalDistance = m_newCamDist;
    }
    else {
      // Compute the next step in the animation.
      // Use an ease-in ease-out approach
      float cos_t = (float)(0.5 * (1.0f - cos(t * M_PI)));

      // New rotation
      SbRotation newRot = SbRotation::slerp(m_oldCamOrientation, m_newCamOrientation, cos_t);
      camera->orientation = newRot;

      // New position
      SbVec3f newFocalPt = m_oldCamFocalPt + (m_newCamFocalPt - m_oldCamFocalPt) * cos_t;
      float   newFocalDist = m_oldCamDist + (m_newCamDist - m_oldCamDist) * cos_t;
      SbMatrix mx;
      mx = newRot;
      SbVec3f viewVector(-mx[2][0], -mx[2][1], -mx[2][2]);
      SbVec3f newPos = newFocalPt - newFocalDist * viewVector;
      camera->position = newPos;
    }
}

///////////////////////////////////////////////////////////////////////////////
// Called from the timer sensor that controls a camera animation.
void cameraAnimationSensorCB(void* /*data*/, SoSensor* /*sensor*/)
{
    SbTime time = m_viewerRealTime->getValue();
    float sec = (float)(time - m_seekStartTime).getValue();
  
    if (sec == 0.0)
        sec = 1.0f / 72.0f;  // at least one frame (needed for first call)
    float t = (sec / m_seekAnimTime);

    // Check that values are correctly clipped
    if (t > 1.0)
        t = 1.0;
    else if ((1.0 - t) < 0.0001)
        t = 1.0;    // this will be the last one
    m_intercatComplexity->interactiveMode = SoInteractiveComplexity::FORCE_INTERACTION;
    // Update camera
    interpolateOrbitAnimation(t);

    // Stop animation if this was the last interval
    if (t == 1.0) {
        m_seekAnimationSensor->detach();
        m_seekAnimationSensor->unschedule();
        m_intercatComplexity->interactiveMode = SoInteractiveComplexity::AUTO;
    }

    // Hack to get nicer camera position - this should not be necessary...
    m_viewer3D->adjustClippingPlanes();
    m_viewer3D->viewAll();
}
