///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

#include <Inventor/SoInput.h>
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoEnvironment.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLineSet.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

// Compute lines colors based on tangents
void computeLineSetColorsAndTangents( SoLineSet* lineSet );

int
main( int /*argc*/, char** argv )
{
  Widget window = SoXt::init( argv[0] );
  if ( window == nullptr )
    exit( 1 );
  InventorMedical::init();

  // Open LineSet
  SoRef<SoSeparator> lineSetRoot;
  SoRef<SoLineSet> fileLineSet;
  {
    lineSetRoot = MedicalHelper::readFile( "$OIVHOME/examples/data/Medical/dtiLineSet/dti_10000_lines.iv" );
    lineSetRoot->ref();
    fileLineSet = static_cast<SoLineSet*>( lineSetRoot->getChild( 0 ) );
  }

  // Compute and add in vertexProperty the tangents and colors
  computeLineSetColorsAndTangents( fileLineSet.ptr() );

  // Construct SceneGraph
  SoRef<SoSeparator> root = new SoSeparator;

  // Set the camera
  SoPerspectiveCamera* camera = new SoPerspectiveCamera;
  camera->position.setValue( 180.f, 0.f, 0.f );
  camera->orientation.setValue( SbVec3f( 0.5f, 0.5f, 0.5f ), static_cast<float>( 2 * M_PI / 3 ) );
  root->addChild( camera );

  // Add an environment node to activate the Ambient Occlusion
  SoEnvironment* env = new SoEnvironment;
  env->ambientOcclusion = TRUE;
  root->addChild( env );

  // Add a lightModel with PER_PIXEL_PHONG lighting to activate tangent lighting
  SoLightModel* lightModel = new SoLightModel;
  lightModel->model.setValue( SoLightModel::PER_PIXEL_PHONG );
  root->addChild( lightModel );

  // Add a material to modulate the lighting
  SoMaterial* mat = new SoMaterial;
  mat->ambientColor.setValue( SbVec3f( 0.8f, 0.8f, 0.8f ) );
  mat->diffuseColor.setValue( SbVec3f( 0.7f, 0.7f, 0.7f ) );
  mat->specularColor.setValue( SbVec3f( 0.2f, 0.2f, 0.2f ) );
  mat->shininess.setValue( 1.0f );
  root->addChild( mat );

  SoSeparator* lineSetSeparator = new SoSeparator;
  root->addChild( lineSetSeparator );

  // Align the LineSet to fit medical conventions with the Gnomon
  SoTransform* transform = new SoTransform;
  transform->translation.setValue( -72.f, -76.f, -26.f );
  transform->rotation.setValue( SbVec3f( -1.f, 0.f, 0.f ), 0.5f );
  lineSetSeparator->addChild( transform );

  lineSetSeparator->addChild( fileLineSet.ptr() );

  // Define Open Inventor logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Medical Gnomon.
  Gnomon* gnomon = new Gnomon();
  root->addChild( gnomon );

  // Set-up viewer
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer( window );
  viewer->setDecoration( false );
  viewer->setSize( MedicalHelper::exampleWindowSize() );
  viewer->setTitle( "medicalDTI" );
  viewer->setSceneGraph( root.ptr() );
  viewer->show();
  viewer->viewAll();

  SoXt::show( window );
  SoXt::mainLoop();

  delete viewer;

  root = nullptr;
  lineSetRoot = nullptr;
  fileLineSet = nullptr;

  InventorMedical::finish();
  SoXt::finish();
  return EXIT_SUCCESS;
}

// Compute DTI lines colors based on tangents
void
computeLineSetColorsAndTangents( SoLineSet* lineSet )
{
  SoVertexProperty* vp = static_cast<SoVertexProperty*>( lineSet->vertexProperty.getValue() );

  const SbVec3f* vertices = vp->vertex.getValues( 0 );
  int numLines = lineSet->numVertices.getNum();
  const int32_t* numVertByLine = lineSet->numVertices.getValues( 0 );

  vp->tangentBinding.setValue( SoVertexProperty::PER_VERTEX );
  vp->materialBinding.setValue( SoVertexProperty::PER_VERTEX );
  vp->tangent.setNum( vp->vertex.getNum() );
  vp->orderedRGBA.setNum( vp->vertex.getNum() );
  SbVec3f* tangents = vp->tangent.startEditing();
  uint32_t* colors = vp->orderedRGBA.startEditing();

  int vtxIt = 0;
  for ( int i = 0; i < numLines; i++ )
  {
    const int32_t numVerts = numVertByLine[i];

    if ( numVerts == 0 )
      continue;

    {
      SbVec3f tangent = vertices[vtxIt + 1] - vertices[vtxIt];
      tangent.normalize();
      tangents[vtxIt] = tangent;
      colors[vtxIt++] = SbColor( std::abs( tangent[0] ), std::abs( tangent[1] ), std::abs( tangent[2] ) ).getPackedValue();
    }

    if ( numVerts == 1 )
      continue;

    for ( int j = 1; j < numVerts - 1; j++ )
    {
      SbVec3f prevVec = vertices[vtxIt] - vertices[vtxIt - 1];
      SbVec3f nextVec = vertices[vtxIt + 1] - vertices[vtxIt];

      SbVec3f tangent = nextVec + prevVec;
      tangent.normalize();
      tangents[vtxIt] = tangent;
      colors[vtxIt++] = SbColor( std::abs( tangent[0] ), std::abs( tangent[1] ), std::abs( tangent[2] ) ).getPackedValue();
    }

    {
      SbVec3f tangent = vertices[vtxIt] - vertices[vtxIt - 1];
      tangent.normalize();
      tangents[vtxIt] = tangent;
      colors[vtxIt++] = SbColor( std::abs( tangent[0] ), std::abs( tangent[1] ), std::abs( tangent[2] ) ).getPackedValue();
    }
  }
  vp->tangent.finishEditing();
  vp->orderedRGBA.finishEditing();
}
