///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (Jan 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Superposition of volume rendering dataset and IndexedLineSet (DTI).
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoMatrixTransform.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText3.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////
// Data Set
const SbString DTI_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/DTI/DTI.iv";
const SbString VOL_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/DTI/list.dcm";
const SbString IMG_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/DTI/T1-BrainOnly-0001.dcm";

// color map that can be edited in Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

/*********************************************************************************/
int main(int /*argc*/, char **argv)
{
  Widget myWindow = SoXt::init(argv[0]);
  SoVolumeRendering::init();
  InventorMedical::init();

  // Root
  SoRef<SoSeparator> root = new SoSeparator();

  // Camera
  root->addChild(new SoOrthographicCamera());

  // Load tractography geometry
  root->addChild(MedicalHelper::readFile(DTI_FILENAME.toLatin1()));

  // Volume Rendering Separator =========================================
  SoSeparator* rootVR = new SoSeparator();
    rootVR->setName("root_VR");
    root->addChild(rootVR);

  // Node to hold the volume data
  SoVolumeData *volData = new SoVolumeData();
    volData->fileName = VOL_FILENAME;
    SoMatrixTransform* transform = new SoMatrixTransform;
    MedicalHelper::dicomAdjustVolume( volData, transform );
    rootVR->addChild(volData);

  // Load the colorMap
  SoTransferFunction *transFunc = new SoTransferFunction();
    transFunc->loadColormap(COLORMAP_FILENAME);
    transFunc->maxValue.setValue(87);
    rootVR->addChild(transFunc);

  // Data range (use embedded window center/width if possible)
  SoDataRange *VRRange = new SoDataRange();
    MedicalHelper::dicomAdjustDataRange( VRRange, volData );
    rootVR->addChild(VRRange);

  // Change complexity when the dataset moves.
  SoInteractiveComplexity *intercatComplexity = new SoInteractiveComplexity();
    intercatComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.9");
    intercatComplexity->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    intercatComplexity->refinementDelay = 0;
    rootVR->addChild(intercatComplexity);
    rootVR->addChild(new SoComplexity());

  // Property node which allows SoVolumeRender to draw High Quality volumes.  
  SoVolumeRenderingQuality *volQuality = new SoVolumeRenderingQuality();
    volQuality->interpolateOnMove = true;
    volQuality->preIntegrated = true;
    volQuality->deferredLighting = false;
    volQuality->ambientOcclusion = false;
    rootVR->addChild(volQuality);

  // Increase transparency so line geometry is visible
  SoMaterial *volMat = new SoMaterial();
    volMat->transparency.setValue(0.9f);
    rootVR->addChild(volMat);

  // Node in charge of drawing the volume
  SoVolumeRender *volRender = new SoVolumeRender();
    volRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
    volRender->interpolation = SoVolumeRender::CUBIC;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    rootVR->addChild(volRender);

  // Define Open Inventor logo
  root->addChild(MedicalHelper::exampleLogoNode());

  // Medical Gnomon.
  Gnomon *gnomon = new Gnomon();
  root->addChild(gnomon);

  // DICOM annotation
  root->addChild( MedicalHelper::exampleDicomAnnotation(IMG_FILENAME) );

  // Set up viewer :
  SoXtExaminerViewer *viewer3D = new SoXtExaminerViewer(myWindow);
  viewer3D->setCameraType(SoOrthographicCamera::getClassTypeId());
  viewer3D->setDecoration(false);
  viewer3D->setTransparencyType( SoGLRenderAction::SORTED_PIXEL);
  viewer3D->setAntialiasing(0.3f, SoSceneManager::SMAA);
  viewer3D->setSize(MedicalHelper::exampleWindowSize());
  viewer3D->setTitle("VolumeRendering&IndexedLineSet");
  viewer3D->setSceneGraph(root.ptr());

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::SAGITTAL, viewer3D->getCamera(), volData, 0.8f );
  viewer3D->saveHomePosition();

  // Run then cleanup
  viewer3D->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer3D;
  root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


