///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (JANUARY 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Min, Max and Average Intensity projection example.
Description : Main
-------------------------------------------------------------------------*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoAntiSquish.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoRotation.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoTransformSeparator.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/manips/SoTransformerManip.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/fields/SoSFVec3f.h>
#include <Inventor/sensors/SoNodeSensor.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////
// Application state

static int m_mode = 1;

static SoXtExaminerViewer* m_viewer3D;
static SoXtPlaneViewer*    m_viewer2D;
static SoVolumeRender*     m_volRender;
static SoTransferFunction* m_transFunc;

static TextBox*            m_textBox;               // So we can update it when mode changes.
static SoTransformerManip* m_slabManip;             // So we can reset it in key event handler.
static SoNodeSensor*       m_obliqueRotationSensor; // Should be member var to protect from garbage collector.

// This is the initial rotation of the 2D view camera (front/anterior view).
static SbRotation m_2DCamBaseRotation( SbVec3f(1,0,0), (float)M_PI/2 );
static SbVec3f    m_slabDefaultScale( 1, 0.15f, 1 );

// Data Set
const SbString FILENAME = "$OIVHOME/examples/data/Medical/files/head.ldm";

///////////////////////////////////////////////////////////////////////////////

static void myKeyPressCB(void *usrData, SoEventCallback *sender);   

static void slabRenderingSensorCB( void *userData, SoSensor* s );

///////////////////////////////////////////////////////////////////////////////
int main(int /*argc*/, char **argv)
{
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator();

    // Events to change MIPs.
    SoEventCallback*myEventCB = new SoEventCallback();
      myEventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), myKeyPressCB);
      root->addChild(myEventCB);

    // Camera
    root->addChild(new SoOrthographicCamera());

    // Medical Gnomon.
    Gnomon *gnomon = new Gnomon();
    root->addChild( gnomon );

    // Open Inventor logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 );
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Mode: Maximum Intensity Projection (MIP)" );
      text->addLine( "Click and drag to move/rotate ROI and 2D view direction (red arrow)" );
      text->addLine( "Press keypad '+' to change projection mode." );
      m_textBox = text;
      root->addChild( text );
    
    //-------------------------------------------------------------------------

    SoTransformSeparator *protecManip = new SoTransformSeparator();
      root->addChild( protecManip );

    // Transformer manip allows user to drag/scale/rotate clipping box.
    // Set initial scale so volume is clipped enough that MinIP projection
    // generates a useful image, i.e. remove 0 values around actual data.
    m_slabManip = new SoTransformerManip();
      m_slabManip->scaleFactor = m_slabDefaultScale;
      protecManip->addChild(m_slabManip);

    //-------------------------------------------------------------------------
    // Create volume clipping group
    // "Slab" = Transform + Cube
    SoVolumeClippingGroup *slab = new SoVolumeClippingGroup();
//      root->addChild( slab );

    // This node prevents local transform from "leaking" out of clipping group.
    SoTransformSeparator *localProtection = new SoTransformSeparator();
      slab->addChild( localProtection );

    // This transform controls the position/size/rotation of the cube.
    // Its fields are connected from the manip's fields so it automatically
    // tracks the changes that the user makes to the manip.
    // Conceptually we could just put the manip itself in this group, but
    // that won't work because the manip's geometry would be used for clipping!
    SoTransform *slabTransfo = new SoTransform();
      slabTransfo->setName("Slab_Transfo");
      slabTransfo->translation.connectFrom( &m_slabManip->translation );
      slabTransfo->rotation.connectFrom( &m_slabManip->rotation );
      slabTransfo->scaleFactor.connectFrom( &m_slabManip->scaleFactor );
      slabTransfo->scaleOrientation.connectFrom( &m_slabManip->scaleOrientation );
      slabTransfo->center.connectFrom( &m_slabManip->center );
      localProtection->addChild(slabTransfo);

    // Add actual geometry of the slab
    localProtection->addChild(new SoCube());
    
    //-------------------------------------------------------------------------
    // Used to crop the head dataset. Then something is visible winthin MIN_INTENSITY_PROJECTION.
    // Important point : It is possible to rotate the ROI.
    SoSeparator *cubeRepSep = new SoSeparator();
      root->addChild(cubeRepSep);

      SoTransform *cubeTransfo = new SoTransform();
        cubeTransfo->translation.connectFrom( &m_slabManip->translation );
        cubeTransfo->rotation.connectFrom( &m_slabManip->rotation );
        cubeTransfo->scaleFactor.connectFrom( &m_slabManip->scaleFactor );
        cubeTransfo->scaleOrientation.connectFrom( &m_slabManip->scaleOrientation );
        cubeTransfo->center.connectFrom( &m_slabManip->center );
        cubeRepSep->addChild(cubeTransfo);

      SoAntiSquish* antiSquish = new SoAntiSquish();  // Prevents transform scale from affecting arrowhead (cone)
        cubeRepSep->addChild( antiSquish );

      SoTranslation *arrowTrans = new SoTranslation();
        arrowTrans->translation.setValue(0,-3,0);
        cubeRepSep->addChild(arrowTrans);

      SoMaterial *arrowMat = new SoMaterial();
        arrowMat->diffuseColor.setValue(1,0,0);
        cubeRepSep->addChild(arrowMat);

      SoCylinder *projectionCylinderArrow = new SoCylinder();
        projectionCylinderArrow->radius.setValue(0.02f);
        cubeRepSep->addChild(projectionCylinderArrow);

      SoTranslation *coneArrowTrans = new SoTranslation();
        coneArrowTrans->translation.setValue(0,1,0);
        cubeRepSep->addChild(coneArrowTrans);

      SoCone *projectionConeArrow = new SoCone();
        projectionConeArrow->bottomRadius.setValue(0.08f);
        projectionConeArrow->height.setValue(0.2f);
        cubeRepSep->addChild(projectionConeArrow);

    //-------------------------------------------------------------------------
    // Create 3D (volume) Rendering sub-graph
    SoSeparator *separator3D = new SoSeparator();
      root->addChild( separator3D );

    // Change complexity when the dataset moves.
    //SoInteractiveComplexity *intercatComplexity = new SoInteractiveComplexity();
    //  intercatComplexity->fieldSettings.set1Value(0, "SoComplexity value 0.5 0.9");
    //  intercatComplexity->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    //  intercatComplexity->refinementDelay = 0;
    //  separator3D->addChild(intercatComplexity);
    //  separator3D->addChild(new SoComplexity());
    
    // Node to hold the volume data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = FILENAME;
      separator3D->addChild(volData);

    // Load the colorMap from an Avizo colormap.
    m_transFunc = new SoTransferFunction();
      m_transFunc->predefColorMap = SoTransferFunction::INTENSITY;
      separator3D->addChild(m_transFunc);

    // Volume rendering properties
    SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
      //volQual->preIntegrated = TRUE;
      volQual->interpolateOnMove = TRUE;
      separator3D->addChild( volQual );

    // Volume clipping group defined above
    separator3D->addChild( slab );

    // Node in charge of drawing the volume
    m_volRender = new SoVolumeRender();
      m_volRender->renderMode = SoVolumeRender::MAX_INTENSITY_PROJECTION; // Default projection
      m_mode = 1;
      m_volRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
      m_volRender->samplingAlignment= SoVolumeRender::BOUNDARY_ALIGNED;
      separator3D->addChild( m_volRender );

    // Show the bounds of the volume for reference
    root->addChild( MedicalHelper::createBoundingBox( volData->extent.getValue(), new SbColor(0.8f,0.4f,0) ));

    //////////////////////////////////////////////////////////////////////////////
    // 2D Part
    SoRef<SoSeparator> root2D = new SoSeparator();

    SoTransformSeparator *transSeparator = new SoTransformSeparator();
      root2D->addChild( transSeparator );

    SoTransform *localTrans = new SoTransform();
      localTrans->setName( "transfo_Camera" );
      transSeparator->addChild(localTrans);

    SoOrthographicCamera *root2DCamera = new SoOrthographicCamera();
      root2DCamera->setName( "root2D_Camera" );
      root2DCamera->position.setValue(m_slabManip->translation.getValue());
      root2DCamera->orientation  = m_2DCamBaseRotation * m_slabManip->rotation.getValue();  // Front/anterior view
      root2DCamera->nearDistance = -50.0;
      root2DCamera->farDistance  =  50.0;
      transSeparator->addChild(root2DCamera);

    // Re-instance the volume rendering sub-graph from the 3D view
    root2D->addChild(separator3D);

    // OIV Logo
    root2D->addChild( MedicalHelper::exampleLogoNode() );

    // Monitor the slab manipulator and update the 2D camera when it changes.
    m_obliqueRotationSensor = new SoNodeSensor( slabRenderingSensorCB, root.ptr() );
      m_obliqueRotationSensor->attach(m_slabManip);
      m_obliqueRotationSensor->setPriority(0);
      m_slabManip->touch();

    // Set up 3D viewer
    m_viewer3D = new SoXtExaminerViewer(myWindow);
      m_viewer3D->setCameraType( SoOrthographicCamera::getClassTypeId() );
      m_viewer3D->setDecoration(FALSE);
      m_viewer3D->setViewing( FALSE );
      m_viewer3D->setSceneGraph(root.ptr());
      m_viewer3D->setTransparencyType( SoGLRenderAction::SORTED_PIXEL);
      m_viewer3D->setTitle("3D Maximum Intensity Projection" );

    // Adjust camera
    SoCamera* camera = m_viewer3D->getCamera();
      camera->orientation.setValue( SbVec3f(0.8f,0.38f,0.45f), 1.5278f ); // Rotate for more interesting view
      camera->viewAll( volData->extent.getValue(), m_viewer3D->getViewportRegion() );
      m_viewer3D->saveHomePosition();

    // Adjust window position/size
    m_viewer3D->show();
#ifdef _WIN32
    Widget localWidget = (Widget)m_viewer3D->getNormalWidget();
    MoveWindow( GetAncestor(localWidget, GA_ROOTOWNER), 0, 0, 800, 633, true); 
#endif

    // Set up 2D viewer
    m_viewer2D = new SoXtPlaneViewer(NULL, "" , FALSE);
      m_viewer2D->setTitle("2D Maximum Intensity Projection");
      m_viewer2D->setSceneGraph(root2D.ptr());
      m_viewer2D->setAutoClipping(FALSE);
      m_viewer2D->setDecoration(FALSE);
      m_viewer2D->show();
#ifdef _WIN32
      localWidget = (Widget)m_viewer2D->getNormalWidget();
      MoveWindow( GetAncestor(localWidget, GA_ROOTOWNER), 800, 0, 633, 633, true); 
#endif

    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete m_viewer3D;
    delete m_viewer2D;
    root = NULL;
    root2D = NULL;
    delete m_obliqueRotationSensor;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
void myKeyPressCB(void* /*usrData*/, SoEventCallback* sender)
{
    SoKeyboardEvent *evt = (SoKeyboardEvent*)sender->getEvent();

    if (SO_KEY_PRESS_EVENT(evt, PAD_ADD)) {
      m_mode = (m_mode + 1) % 4;
      switch (m_mode) {
        case 0: // Min
            m_volRender->renderMode = SoVolumeRender::RenderMode::MIN_INTENSITY_PROJECTION;
            m_transFunc->maxValue = 93;
            m_viewer3D->setTitle("3D Minimum Intensity Projection (MinIP)");
            m_viewer2D->setTitle("2D Minimum Intensity Projection (MinIP)");
            m_textBox->setLine( "Minimum Intensity Projection (MinIP)" );
            break;

        case 1: // Max 
            m_volRender->renderMode = SoVolumeRender::RenderMode::MAX_INTENSITY_PROJECTION;
            m_transFunc->maxValue = 255;
            m_viewer3D->setTitle("3D Maximum Intensity Projection (MIP)");
            m_viewer2D->setTitle("2D Maximum Intensity Projection (MIP)");
            m_textBox->setLine("Maximum Intensity Projection (MIP)");
            break;

        case 2: // Average
            m_volRender->renderMode = SoVolumeRender::RenderMode::AVERAGE_INTENSITY_PROJECTION;
            m_transFunc->maxValue = 155;
            m_viewer3D->setTitle("3D Average Intensity Projection (AvgIP)");
            m_viewer2D->setTitle("2D Average Intensity Projection (AvgIP)");
            m_textBox->setLine("Average Intensity Projection (AvgIP)");
            break;
        case 3: // Mida
            m_volRender->renderMode = SoVolumeRender::RenderMode::MAX_INTENSITY_DIFFERENCE_ACCUMULATION;
            m_transFunc->maxValue = 255;
            m_viewer3D->setTitle("3D Max Intensity Difference Accumulation (MIDA)");
            m_viewer2D->setTitle("2D Max Intensity Difference Accumulation (MIDA)");
            m_textBox->setLine("Max Intensity Difference Accumulation (MIDA)");
            break;
        }
    }
    else if (SO_KEY_PRESS_EVENT(evt, R)) {
      // Reset transform (turn off sensor until all fields have been modified)
      m_obliqueRotationSensor->unschedule();
      m_slabManip->rotation = SbRotation::identity();
      m_slabManip->scaleFactor = m_slabDefaultScale;
      m_slabManip->translation.setValue( 0, 0, 0 );
      m_obliqueRotationSensor->schedule();
      m_slabManip->touch();
    }
}

///////////////////////////////////////////////////////////////////////////////
void slabRenderingSensorCB( void* /*userData*/, SoSensor* s )
{
    SoTransform* manipTransfo = static_cast<SoTransform *>(dynamic_cast<SoNodeSensor*>(s)->getAttachedNode());

    SoOrthographicCamera* cam2D = dynamic_cast<SoOrthographicCamera *>(SoNode::getByName("root2D_Camera"));
    
    // Manip moves and rotates the 2D camera, but _relative_ to its initial/base rotation.
    cam2D->position.setValue(manipTransfo->translation.getValue());
    cam2D->orientation.setValue(m_2DCamBaseRotation * manipTransfo->rotation.getValue());
}
