///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (Jul 2009)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Medical example program.
Description : Manage events for this demonstration.
----------------------------------------------------------------------------------------*/

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/events/SoLocation2Event.h> 
#include <Inventor/events/SoMouseButtonEvent.h> 
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/events/SoKeyboardEvent.h>

#include "manageEvents.h"

static SbBool leftMousePressed = FALSE;

///////////////////////////////////////////////////////////////////////////////
void
keyboardPressCB(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();
  SoSeparator *root = (static_cast<dataEvent*>(userData))->sep;
  int cpt = 0;
  // check for the Up and Down arrow keys being pressed
  if (SO_KEY_PRESS_EVENT(event, UP_ARROW))
    cpt = NUM_I * (NUM_J - 1);
  else if (SO_KEY_PRESS_EVENT(event, DOWN_ARROW)) 
    cpt -= NUM_I * (NUM_J - 1);

  for (int j = 0; j < NUM_J; j++)
  {
    for (int i = 0; i < NUM_I; i++) 
    {
      static_cast<dataEvent*>(userData)->OrthoSlice_Z[i][j]->sliceNumber = static_cast<dataEvent*>(userData)->OrthoSlice_Z[i][j]->sliceNumber.getValue() + cpt;
      SoOrthographicCamera *cam = dynamic_cast<SoOrthographicCamera *>(static_cast<dataEvent*>(userData)->view[i][j]->getPart("cameraKit.camera", true));
      cam->viewAll( root, SbViewportRegion(1,1) ) ;
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
void
  mouseMoveEventCB(void *userData, SoEventCallback *eventCB)
{
  const SoEvent *event = eventCB->getEvent();
  SbVec2s myVec2s = event->getPosition();
  SoTransferFunction *localTransfertFunction = (SoTransferFunction*)userData;

  static int horizVal;
  static int vertiVal;

  if( ::leftMousePressed ) {

    int minVal = localTransfertFunction->minValue.getValue();
    if(  myVec2s[0] > horizVal && minVal < 255 ) {
      minVal += 2;
    } 
    if( myVec2s[0] < horizVal && minVal > 0) {
      minVal -= 2;
    }
    localTransfertFunction->minValue.setValue( minVal );
    horizVal = myVec2s[0];

    int  maxVal = localTransfertFunction->maxValue.getValue(); 
    if(  myVec2s[1] > vertiVal && maxVal < 255) {
      maxVal += 2;
    } 
    if( myVec2s[1] < vertiVal && maxVal > 0) {
      maxVal -= 2;
    }
    localTransfertFunction->maxValue.setValue( maxVal );
    vertiVal = myVec2s[1];
  }
}

///////////////////////////////////////////////////////////////////////////////
void
  mouseKeyEventCB(void* /*userData*/, SoEventCallback* eventCB)
{
  const SoEvent *event = eventCB->getEvent();

  // Check for mouse button being pressed
  if (SO_MOUSE_PRESS_EVENT(event, BUTTON1))
  {
    ::leftMousePressed = TRUE;
    eventCB->setHandled();
  }
  else
    ::leftMousePressed = FALSE;
}

void
  motionObliSliceCallback(void *user_data, SoJackDragger *dragger)
{
  SbVec3f planeNormal(0,0,1) ;
  SbVec3f draggerPos = dragger->translation.getValue();

  // rotate the plane's normal by the dragger rotation
  SbRotation rotation = dragger->rotation.getValue();
  rotation.multVec(SbVec3f(0,1,0),planeNormal);

  SoObliqueSlice *localObliqueSlice = (SoObliqueSlice *)user_data;
  // translate cross section and cross contour
  localObliqueSlice->plane.setValue(SbPlane(planeNormal,draggerPos));
}

///////////////////////////////////////////////////////////////////////////////
void    
  obliqueRotationFieldSensorCB( void *userData, SoSensor *s)
{
  SbRotation localDraggerRot = (dynamic_cast<SoSFRotation *>((dynamic_cast<SoFieldSensor*>(s))->getAttachedField()))->getValue();

  // rotate the plane's normal by the dragger rotation
  SbRotation rotation( SbVec3f(0,1,0), SbVec3f(0,0,1) );

  // Rotate the camera when I move the dragger
  rotation *= localDraggerRot;

  SoOrthographicCamera *obliqueViewerCamera = dynamic_cast<SoOrthographicCamera *>(SoNode::getByName("Oblique_Camera"));
  obliqueViewerCamera->orientation.setValue( rotation );
  //
  obliqueViewerCamera->viewAll( static_cast<SoSeparator*>(userData), SbViewportRegion(1,1) );
}
