///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Jul 2009)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Medical example program.
Purpose : High Quality rendering for medical data.
Description : Main
----------------------------------------------------------------------------------------*/

// Inventor includes
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoClipPlane.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/sensors/SoFieldSensor.h>
#include <Inventor/draggers/SoJackDragger.h>
#include <Inventor/events/SoLocation2Event.h> 

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

// All event managements done in a separate file
#include "manageEvents.h"

///////////////////////////////////////////////////////////////////////////////
// Data Set
//#define FILENAME "$OIVHOME/examples/data/Medical/files/Thorax.ldm"
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";

// color map that can be edited with Avizo.
const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize extensions
  SoVolumeRendering::init();
  InventorMedical::init();

  // ROOTs
  SoRef<SoSeparator> rootOrtho = new SoSeparator();
  rootOrtho->setName("rootOrtho");
  
  SoRef<SoSeparator> rootVR = new SoSeparator();
  rootVR->setName("rootVR");
  SoPerspectiveCamera *camera3D = new SoPerspectiveCamera();
  camera3D->setName("camera3D");
  camera3D->position.setValue(SbVec3f(2.0, -1107, -224));
  camera3D->orientation.setValue(SbRotation(SbVec3f(-1, 0, 0), 4.9f));
  camera3D->nearDistance = 792.0f;
  camera3D->farDistance = 1241.0f;
  camera3D->focalDistance = 1042.0f;
  camera3D->heightAngle = 0.78f;
  rootVR->addChild(camera3D);

  SoRef<SoSeparator> rootOblique = new SoSeparator();
    rootOblique->setName("rootOblique");
  
  SoOrthographicCamera *obliqueCamera = new SoOrthographicCamera();
    obliqueCamera->setName("Oblique_Camera");
    rootOblique->addChild( obliqueCamera );

  // 3D Data Base Separator =========================================
  SoGroup *dataGoup = new SoGroup();

  // Directional Light
  SoDirectionalLight *dirLight = new SoDirectionalLight ;
    dirLight->direction.setValue(0.5f, 0.5f, -0.5f) ;
    dataGoup->addChild(dirLight) ;

  // Define Open Inventor logo
  SoAnnotation *bgImgAnnotation = new SoAnnotation();
    bgImgAnnotation->addChild( MedicalHelper::exampleLogoNode() );
    dataGoup->addChild( bgImgAnnotation );

  // Increase the rendering quality when you stop to move the dataset.
  SoInteractiveComplexity* interaCplx = new SoInteractiveComplexity;
    interaCplx->fieldSettings.set1Value(0, "SoComplexity value 0.2 0.5");
    interaCplx->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    interaCplx->refinementDelay.setValue(0);
    dataGoup->addChild( interaCplx ); 

  SoComplexity *cplx = new SoComplexity();
    dataGoup->addChild( cplx ); 

  // Node to hold the volume data
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = VOLUME_FILENAME;
    MedicalHelper::dicomAdjustVolume( volData );
    dataGoup->addChild( volData );

  rootOrtho->addChild( dataGoup );

  // Ortho Separator =========================================
  SoSeparator *orthoSep = new SoSeparator();
    orthoSep->setName("Ortho_Separator");
    rootOrtho->addChild( orthoSep );

  SoLightModel* lmodel = new SoLightModel(); // Turn off lighting
    lmodel->model = SoLightModel::BASE_COLOR;
    orthoSep->addChild( lmodel );

  // remap data range to full range of data
  SoDataRange* orthoRange = new SoDataRange;
    MedicalHelper::dicomAdjustDataRange( orthoRange, volData );
    //orthoRange->min = -1000;
    //orthoRange->max = 2138;
    orthoSep->addChild( orthoRange );

  // Full intensity
  SoMaterial* orthoMatl = new SoMaterial();
    orthoMatl->diffuseColor.setValue( 1, 1, 1 );
    orthoSep->addChild( orthoMatl );

  // Select intensity ramp
  SoTransferFunction* orthoTransFunc = new SoTransferFunction;
    orthoTransFunc->predefColorMap = SoTransferFunction::INTENSITY;
    orthoSep->addChild( orthoTransFunc );

  // Create light box view
  dataEvent *data = new dataEvent;
  int k = 279;
  float vpSizeX = 1.0f / NUM_I;
  float vpSizeY = 1.0f / NUM_J;
  for (int j=0; j < NUM_J; j++)
  {
    for (int i=0; i < NUM_I; i++) 
    {
      data->view[i][j] = new PoView();
      data->view[i][j]->sensitiveOnEvents(TRUE) ;
      data->view[i][j]->viewportOrigin.setValue( 0.f+(i*vpSizeX), 0.f+(j*vpSizeY) ) ;
      data->view[i][j]->viewportSize.setValue( vpSizeX, vpSizeY ) ;
      SoOrthographicCamera* camera = new SoOrthographicCamera() ;
      data->view[i][j]->setPart("cameraKit.camera", camera) ;
      orthoSep->addChild( data->view[i][j] );
      SoSeparator *locSep = new SoSeparator();
      data->OrthoSlice_Z[i][j] = new SoOrthoSlice();
      data->OrthoSlice_Z[i][j]->sliceNumber = k ;
      k++;
      data->OrthoSlice_Z[i][j]->axis.setValue(SoOrthoSlice::Z);
      locSep->addChild( data->OrthoSlice_Z[i][j] );
      orthoSep->addChild( locSep );
      camera->viewAll( rootOrtho.ptr(), SbViewportRegion(1,1) ) ;
    }
  }
  data->sep = rootOrtho.ptr();

  // An event callback node so we can receive key press events
  SoEventCallback *myEventCB = new SoEventCallback;
    myEventCB->addEventCallback( SoKeyboardEvent::getClassTypeId(), keyboardPressCB, data  );
    rootOrtho->addChild(myEventCB);

  // VR Separator =========================================
  SoSeparator *VRSep = new SoSeparator();
    VRSep->setName("VR_Separator");
    rootVR->addChild( dataGoup );

  VRSep->addChild( new Gnomon() );

  // Load the colorMap from an Avizo colormap.
  SoTransferFunction* VRTransFunc = new SoTransferFunction;
    VRTransFunc->loadColormap(COLORMAP_FILENAME);
    VRSep->addChild( VRTransFunc );

  // Map specific range of data for this colormap
  SoDataRange* VRRange = new SoDataRange;
    VRRange->min = 176;
    VRRange->max = 476;
    VRSep->addChild( VRRange );

  rootVR->addChild( VRSep );

  // Property node which allows SoVolumeRender to draw High Quality volumes.  
  SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;      
    pVRVolQuality->interpolateOnMove = TRUE;
    pVRVolQuality->preIntegrated = TRUE;
    pVRVolQuality->ambientOcclusion = TRUE;
    pVRVolQuality->deferredLighting = TRUE;
    VRSep->addChild( pVRVolQuality );

  SbBox3f volSize = volData->extent.getValue();
  SbVec3f obliPos = volSize.getMin()+(volSize.getMax()-volSize.getMin())/2.0f;
  SbVec3f draggerNormal;
  draggerNormal.setValue( 0, 0, 1 );


  // Oblique Groupe =========================================
  rootOblique->addChild( dataGoup );
    SoGroup *obliqueGroup = new SoGroup();
    obliqueGroup->setName("Oblique_Group");

  // Transfert Function for the oblique slice.
  SoTransferFunction* obliqueTransFunc = new SoTransferFunction;
    obliqueTransFunc->predefColorMap = SoTransferFunction::INTENSITY;
    obliqueGroup->addChild( obliqueTransFunc );

  // remap data range to full range of data
  SoDataRange* obliqueRange = new SoDataRange;
    MedicalHelper::dicomAdjustDataRange( obliqueRange, volData );
    obliqueGroup->addChild( obliqueRange );

  SoObliqueSlice *obliqueSlice = new SoObliqueSlice();
    obliqueSlice->interpolation.setValue(SoObliqueSlice::TRILINEAR);
    obliqueSlice->enableBumpMapping = true;
    obliqueSlice->bumpScale = 10.0f;
    obliqueSlice->plane = SbPlane(draggerNormal, obliPos);
    obliqueGroup->addChild( obliqueSlice );

  rootOblique->addChild( obliqueGroup );

  // Clipping Group =========================================
  SoGroup *clippingGroup = new SoGroup();
    clippingGroup->setName("Clipping_Group");

  SoJackDragger *draggerObliSlice = new SoJackDragger();
    draggerObliSlice->setName("Jack_For_Oblique");
    draggerObliSlice->addMotionCallback((SoDraggerCB *)&motionObliSliceCallback, obliqueSlice );
    draggerObliSlice->rotation = SbRotation(SbVec3f(0,1,0),draggerNormal);
    draggerObliSlice->scaleFactor.setValue( 50, 50, 50);
    draggerObliSlice->translation.setValue(obliPos);
    clippingGroup->addChild( draggerObliSlice );

  SoFieldSensor *obliqueRotationSensor = new SoFieldSensor( obliqueRotationFieldSensorCB, rootOblique.ptr() );
    obliqueRotationSensor->attach( &draggerObliSlice->rotation );
    obliqueRotationSensor->setPriority(0);

  SoClipPlane *obliClipPlane = new SoClipPlane;
    obliClipPlane->setName("Oblique_ClipPlane");
    obliClipPlane->on = TRUE;
    obliClipPlane->plane.connectFrom((SoField*)&(obliqueSlice->plane));
    clippingGroup->addChild( obliClipPlane );

  VRSep->addChild( clippingGroup );

  // Node in charge of drawing the volume
  SoVolumeRender* pVolRender = new SoVolumeRender;
    pVolRender->numSlicesControl = SoVolumeRender:: AUTOMATIC;
    pVolRender->lowResMode.setValue(SoVolumeRender::DECREASE_SCREEN_RESOLUTION);
    pVolRender->lowScreenResolutionScale.setValue( 2 );
    pVolRender->subdivideTile.setValue(TRUE);
    pVolRender->samplingAlignment.setValue(SoVolumeRender::BOUNDARY_ALIGNED);
    pVolRender->opacityThreshold.setValue(0.17f);
    VRSep->addChild( pVolRender );

  // Mouse event callback to manage data range.
  SoEventCallback *mouseMoveEvent = new SoEventCallback();
  SoEventCallback *mouseKeyEvent = new SoEventCallback();
  mouseMoveEvent->addEventCallback( SoLocation2Event::getClassTypeId(), mouseMoveEventCB, VRTransFunc);
 // mouseKeyEvent->addEventCallback( SoMouseButtonEvent::getClassTypeId(), keyboardPressCB, VRSep );

  VRSep->addChild( mouseMoveEvent );
  VRSep->addChild( mouseKeyEvent );

  // Set up viewers :
  // 7 x 7 OrthoSlice viewer.
  SoXtRenderArea *orthoViewer = new SoXtRenderArea(myWindow);
  SbString str;
  str.sprintf( "%d x %d Lightbox", NUM_I, NUM_J );
  orthoViewer->setTitle( str );
  orthoViewer->setSceneGraph(rootOrtho.ptr());
#ifdef _WIN32
  MoveWindow( (HWND)(myWindow), 600, 0, 600, 1080, true); 
#endif
  orthoViewer->show();

  // Volume Rendering  + CLipping viewer.
  SoXtExaminerViewer *viewer3D = new SoXtExaminerViewer(NULL, "" , FALSE);
  viewer3D->setTitle("3D Rendering");
  viewer3D->setSceneGraph(rootVR.ptr());

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::CORONAL, viewer3D->getCamera(), volData );
  viewer3D->saveHomePosition();
  viewer3D->show();
#ifdef _WIN32
  Widget localWidget = (Widget)viewer3D->getNormalWidget();
  MoveWindow( GetAncestor(localWidget, GA_ROOTOWNER), 0, 0, 600, 600, true); 
#endif

  // Oblique Slice viewer.
  SoXtPlaneViewer *viewerObliqueSlice = new SoXtPlaneViewer(NULL, "" , FALSE);
  viewerObliqueSlice->setTitle("Oblique Rendering");
  viewerObliqueSlice->setDecoration( FALSE );
  viewerObliqueSlice->setSceneGraph(rootOblique.ptr());
  viewerObliqueSlice->show();
  viewerObliqueSlice->viewAll();

#ifdef _WIN32
  localWidget = (Widget)viewerObliqueSlice->getNormalWidget();
  MoveWindow(GetAncestor(localWidget, GA_ROOTOWNER), 0, 600, 600, 480, true);
#endif

  // Run then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete orthoViewer;
  delete viewer3D;
  delete viewerObliqueSlice;
  rootVR = NULL;
  rootOblique = NULL;
  rootOrtho = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

