///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Sep 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to render ortho Slice with a Dicom File.
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/SoDialogVizAll.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>

#include <VolumeViz/draggers/SoOrthoSliceDragger.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

// Data Set
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";
const SbString IMAGE_FILENAME  = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    // Create the window, initialize OpenIventor and VolumeViz.
    Widget myWindow = SoXt::init(argv[0]);
    if (! SbFileHelper::isAccessible(VOLUME_FILENAME)) {
      new SoMessageDialog( VOLUME_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    SoVolumeRendering::init();	
    InventorMedical::init();

    // Create the scene graph.
    SoRef<SoSeparator> root = new SoSeparator();

    // Camera
    SoOrthographicCamera* camera = new SoOrthographicCamera();
    root->addChild( camera );

    // Volume rendering stuff
    SoSeparator* volSep = new SoSeparator();
      root->addChild( volSep );

    // Data
    SoVolumeData *volData = new SoVolumeData();
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild(volData);

    // Use embedded window center/width if possible
    SoDataRange *dataRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( dataRange, volData );
      volSep->addChild( dataRange );

    // Use full intensities
    SoMaterial* volMat = new SoMaterial();
      volMat->diffuseColor.setValue( 1, 1, 1 );
      volSep->addChild( volMat );

    // Gray scale
    SoTransferFunction *transFunc = new SoTransferFunction();
      transFunc->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( transFunc, volData );
      volSep->addChild( transFunc );

    // Otho slice rendering & manipulation
    SoOrthoSlice* slice = new SoOrthoSlice();
      slice->sliceNumber   = volData->data.getSize()[2] / 2;
      slice->interpolation = SoSlice::MULTISAMPLE_12;
      volSep->addChild( slice );

    //SoPath *path = new SoPath(root.ptr());
    //path->append( orthoSlice );

    //SoOrthoSliceDragger* orthoSliceDragger = new SoOrthoSliceDragger();
    //orthoSliceDragger->orthoSlicePath  = path;
    //root->addChild(orthoSliceDragger);

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // DICOM annotation
    root->addChild( MedicalHelper::buildSliceAnnotation( camera, slice, &IMAGE_FILENAME ));

    // Set up viewer.
    SoXtPlaneViewer *viewer = new SoXtPlaneViewer(myWindow);
    viewer->setTitle( "Simple OrthoSlice" );
    viewer->setDecoration(FALSE);
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    viewer->setSceneGraph(root.ptr());

    // Initialize the camera view
    MedicalHelper::orientView( MedicalHelper::AXIAL, camera, volData );
    viewer->saveHomePosition();
    viewer->show();

    // Run then cleanup
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

