///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Example for OrthoSlice with border. it uses the SoOrthoSlice
          node that can be found in medicalExample library.
Notes:
  - A different color border is used for each slice to identify the axial,
    sagittal and coronal slices.
  - The slices can be moved interactively by click-and-drag on the slice.
  - The slice border changes color when selected for dragging.
-------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/sensors/SoNodeSensor.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <DialogViz/SoDialogVizAll.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeShader.h>

#include <VolumeViz/draggers/SoOrthoSliceDragger.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

#include <iostream>

// Data file
const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";

////////////////////////////////////////////////////////////////////////
// Forward decls

// Called on key press events
void onKeyEvent( void* data, SoEventCallback* node);

// Called when slice node is modified
void sliceSensorCB( void* data, SoSensor* sensor );

// Updates slice number info on screen
void updateSliceTextBox( SoOrthoSlice* node = nullptr );

////////////////////////////////////////////////////////////////////////
// Application state

static SbVec3i32 m_volDim(0,0,0); // Volume dimensions

static TextBox* m_sliceTextBox = NULL;  // Slice info display

static SoVolumeData*     m_volData = NULL; // Volume data
static SoOrthoSlice* m_aSlice  = NULL; // Slice nodes
static SoOrthoSlice* m_cSlice  = NULL;
static SoOrthoSlice* m_sSlice  = NULL;

const int m_aSliceAxis = 2; // Axial slices are on volume Z axis
const int m_cSliceAxis = 1; // Coronal slices are on volume Y axis
const int m_sSliceAxis = 0; // Sagittal slices are on volume Z axis

const int m_aSliceLine = 1; // Which line in text box to update
const int m_cSliceLine = 2;
const int m_sSliceLine = 3;

// Default slice border colors
const SbColor aSliceColor(1    , 0.11f, 0.11f);  // Red luminance 55%
const SbColor cSliceColor(0    , 0.7f , 0    );  // Blue luminance 55%
const SbColor sSliceColor(0.37f, 0.37f, 1    );  // Green luminance 55%

////////////////////////////////////////////////////////////////////////

int
main(int /*argc*/, char **argv)
{
  std::cout << "Use mouse (in selection mode) to drag slice.\n";
  std::cout << "Press:\n";
  std::cout << "  'B' to toggle border on/off.\n";
  std::cout << "  'C' to toggle border color.\n";
  std::cout << "  'O' to toggle orthoSlice on/off.\n";
  std::cout << "  'W' to toggle border width.\n";

  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);
  if (! SbFileHelper::isAccessible(VOLUME_FILENAME)) {
    new SoMessageDialog( VOLUME_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
    return -1;
  }
  SoVolumeRendering::init();
  InventorMedical::init();

  // Root of scene graph
  SoRef<SoSeparator> root = new SoSeparator;

  // Handle some events
  SoEventCallback* eventCB = new SoEventCallback;
  eventCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), onKeyEvent);
  root->addChild(eventCB);

  // Camera (slightly rotated for more interesting view)
  SoPerspectiveCamera* camera = new SoPerspectiveCamera();
  camera->orientation.setValue(SbVec3f(0.758387f, -0.408331f, -0.50805f), 1.65106f);
  root->addChild(camera);

  // Keep volume viz separate from geometry
  SoSeparator* volSep = new SoSeparator();
  root->addChild(volSep);

  // Load volume data.
  // (Adjust tile size for efficient loading of DICOM CT volume.)
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = VOLUME_FILENAME;
    MedicalHelper::dicomAdjustVolume( volData );
    volSep->addChild(volData);
    m_volDim = volData->data.getSize();
    m_volData = volData;

  // Set range of data values to visualize.
  // Not required for 8-bit voxels, critical for larger data types.
  SoDataRange* volRange = new SoDataRange();
    MedicalHelper::dicomAdjustDataRange( volRange, volData );
    volSep->addChild(volRange);

  // Load opaque intensity ramp
  SoTransferFunction* volTF = new SoTransferFunction();
    volTF->predefColorMap = SoTransferFunction::INTENSITY;
    MedicalHelper::dicomCheckMonochrome1( volTF, volData );
    volSep->addChild(volTF);

  // Display slice at full intensity
  SoMaterial* volMat = new SoMaterial();
    volMat->diffuseColor.setValue(1, 1, 1);
    volSep->addChild(volMat);

  // Use volume dimensions to initially position slices at center of volume
  const SbVec3i32& volDim = volData->data.getSize();

  //---------------------------------------------------------------------------
  // Create Axial slice and dragger (Z axis)
  SoOrthoSlice* axialPlane = new SoOrthoSlice();
    axialPlane->axis = SoOrthoSlice::Z;
    axialPlane->sliceNumber = volDim[2] / 2;
    axialPlane->interpolation = SoOrthoSlice::MULTISAMPLE_12;
    axialPlane->enableBorder = true;
    axialPlane->borderColor = aSliceColor; // Red luminance 55%
    axialPlane->borderWidth = 3;
    axialPlane->setName("axialPlane");
    volSep->addChild(axialPlane);

  // Create a dragger to allow moving the slice.
  // The start and finish callbacks are not required to use the dragger.
  // In this case they allow us to temporarily change the border color.
  SoOrthoSliceDragger* axialPlaneDragger = new SoOrthoSliceDragger();
    axialPlaneDragger->orthoSlicePath  = new SoPath(axialPlane);
    volSep->addChild(axialPlaneDragger);

  //---------------------------------------------------------------------------
  // Create Coronal slice and dragger (Y axis)
  SoOrthoSlice* coronalPlane = new SoOrthoSlice();
    coronalPlane->axis = SoOrthoSlice::Y;
    coronalPlane->sliceNumber = volDim[1] / 2;
    coronalPlane->interpolation = SoOrthoSlice::MULTISAMPLE_12;
    coronalPlane->enableBorder = true;
    coronalPlane->borderColor = cSliceColor;      // Green luminance 55%
    coronalPlane->borderWidth = 3;
    coronalPlane->setName("coronalPlane");
    volSep->addChild(coronalPlane);

  SoOrthoSliceDragger* coronalPlaneDragger = new SoOrthoSliceDragger();
    coronalPlaneDragger->orthoSlicePath  = new SoPath(coronalPlane);
    volSep->addChild(coronalPlaneDragger);

  //---------------------------------------------------------------------------
  // Create Sagittal slice and dragger (X axis)
  SoOrthoSlice* sagittalPlane = new SoOrthoSlice();
    sagittalPlane->axis = SoOrthoSlice::X;
    sagittalPlane->sliceNumber = volDim[0] / 2;
    sagittalPlane->interpolation = SoOrthoSlice::MULTISAMPLE_12;
    sagittalPlane->enableBorder = true;
    sagittalPlane->borderColor = sSliceColor; // Blue luminance 55%
    sagittalPlane->borderWidth = 3;
    sagittalPlane->setName("sagittalPlane");
    volSep->addChild(sagittalPlane);

  SoOrthoSliceDragger* sagittalPlaneDragger = new SoOrthoSliceDragger();
    sagittalPlaneDragger->orthoSlicePath  = new SoPath(sagittalPlane);
    volSep->addChild(sagittalPlaneDragger);

  // Show the extent of the volume
  SbColor boxColor(0.8f, 0.8f, 0.8f);
  root->addChild(MedicalHelper::createBoundingBox(volData->extent.getValue(), &boxColor));

  // OIV Logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Medical Gnomon.
  Gnomon *gnomon = new Gnomon();
  root->addChild(gnomon);

  // Status info
  m_sliceTextBox = new TextBox();
    m_sliceTextBox->position.setValue( -0.98f, 0.98f, 0 );
    m_sliceTextBox->fontSize = 15;
    m_sliceTextBox->addLine( "Click and drag to reposition slices" );
    m_sliceTextBox->addLine( "Axial     : " );
    m_sliceTextBox->addLine( "Coronal: " );
    m_sliceTextBox->addLine( "Sagittal : ");
    m_sliceTextBox->addLine( " " );
    m_sliceTextBox->addLine( "Press:" );
    m_sliceTextBox->addLine( " B - toggle border" );
    m_sliceTextBox->addLine( " C - toggle color" );
    m_sliceTextBox->addLine( " O - toggle slice" );
    m_sliceTextBox->addLine( " W - toggle width" );
    root->addChild( m_sliceTextBox );

  // Create sensors to allow us to update the slice number in the text box.
  // In this demo all the slice sensors trigger the same callback method.
  // That method updates the text for all slices.
  m_aSlice = axialPlane;
  m_cSlice = coronalPlane;
  m_sSlice = sagittalPlane;
  SoNodeSensor* aSliceSensor = new SoNodeSensor( sliceSensorCB, (void*)m_aSlice );
  SoNodeSensor* cSliceSensor = new SoNodeSensor( sliceSensorCB, (void*)m_cSlice );
  SoNodeSensor* sSliceSensor = new SoNodeSensor( sliceSensorCB, (void*)m_sSlice );
  aSliceSensor->attach( m_aSlice );
  cSliceSensor->attach( m_cSlice );
  sSliceSensor->attach( m_sSlice );
  updateSliceTextBox();

  // Create a viewer with a specified size.
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(myWindow);
  viewer->setSize( MedicalHelper::exampleWindowSize() );
  viewer->setDecoration(FALSE);
  viewer->setViewing( FALSE );  // Start in selection mode to allow dragging slices
  viewer->setSceneGraph(root.ptr());
  viewer->setTitle("OrthoSliceBorder");
  // We defined our own camera, so make sure initial and reset position are good.
  viewer->viewAll();
  viewer->saveHomePosition();
  // Visualization uses lines, so antialiasing will increase the image quality.
  viewer->setAntialiasing( 0.5f, SoSceneManager::AntialiasingMode::AUTO );

  // Run then cleanup
  viewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;
  delete aSliceSensor;
  delete cSliceSensor;
  delete sSliceSensor;
  root = NULL; // Releases object managed by SoRef
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Handle keybord events to toggle border on/off, border color, etc.
void onKeyEvent(void* /*userData*/, SoEventCallback* node)
{
  if (m_aSlice == NULL || m_cSlice == NULL || m_sSlice == NULL) {
    std::cerr << "Error: One or more orthoSlice nodes not found.\n";
    return;
  }
  const SoEvent* evt = node->getEvent();

  // --------------------------------------------------------------------------
  // Border on/off
  if (SO_KEY_PRESS_EVENT(evt, B)) {
    // Toggle border visibility
    SbBool enable = ! m_aSlice->enableBorder.getValue();
    m_aSlice->enableBorder = enable;
    m_cSlice->enableBorder = enable;
    m_sSlice->enableBorder = enable;
  }
  // --------------------------------------------------------------------------
  // Border color
  else if (SO_KEY_PRESS_EVENT(evt, C)) {
    SbColor colorA = m_aSlice->borderColor.getValue();
    if (colorA[1] != 1) { // Default value for G component is 0.11
      // Switch to highlight color
      m_aSlice->borderColor.setValue(1, 1, 1);
      m_cSlice->borderColor.setValue(1, 1, 1);
      m_sSlice->borderColor.setValue(1, 1, 1);
    }
    else {
      // Switch to default colors
      m_aSlice->borderColor = aSliceColor;
      m_cSlice->borderColor = cSliceColor;
      m_sSlice->borderColor = sSliceColor;
    }
  }
  // --------------------------------------------------------------------------
  // Slice rendering (toggle)
  else if (SO_KEY_PRESS_EVENT(evt, O)) {
    // Toggle border visibility
    SbBool enable = ! m_aSlice->enableImage.getValue();
    m_aSlice->enableImage = enable;
    m_cSlice->enableImage = enable;
    m_sSlice->enableImage = enable;
  }
  // --------------------------------------------------------------------------
  // Border width
  else if (SO_KEY_PRESS_EVENT(evt, W)) {
    float width = (m_aSlice->borderWidth.getValue() == 0) ? 3.0f : 0;
    m_aSlice->borderWidth = width;
    m_cSlice->borderWidth = width;
    m_sSlice->borderWidth = width;
  }
  // --------------------------------------------------------------------------
  // Clipping
  // Note 1: This is just for testing that clipping does not remove the border.
  // Note 2: Clipping only affects nodes that are traversed _after_ the clipping
  //         slice, so the Axial slice affects both other slices, but enabling
  //         clipping on the Coronal slice would only affect the Sagittal slice.
  else if (SO_KEY_PRESS_EVENT(evt, X)) {
    SbBool enable = !m_aSlice->clipping.getValue();
    m_aSlice->clipping = enable;
  }
}

///////////////////////////////////////////////////////////////////////////////
void sliceSensorCB( void* /*data*/, SoSensor* _sensor )
{
  SoNodeSensor* sensor = dynamic_cast<SoNodeSensor*>(_sensor);
  SoOrthoSlice* slice = dynamic_cast<SoOrthoSlice*>(sensor->getAttachedNode());
  updateSliceTextBox( slice );
}

///////////////////////////////////////////////////////////////////////////////
// Update slice position display.
// If node == NULL, update all slice positions.
void updateSliceTextBox( SoOrthoSlice* node )
{
  SbString str;
  SbVec3f voxelPos;
  if (node == NULL || node == m_aSlice) {
    int sliceNum = m_aSlice->sliceNumber.getValue();
    voxelPos[m_aSliceAxis] = (float)sliceNum;
    SbVec3f xyzPos = m_volData->voxelToXYZ(voxelPos);
    str.sprintf( "Axial       (red)      : %d / %d : %g mm",
      sliceNum, m_volDim[m_aSliceAxis], xyzPos[m_aSliceAxis] );
    m_sliceTextBox->setLine( str, m_aSliceLine );
  }
  if (node == NULL || node == m_cSlice) {
    int sliceNum = m_cSlice->sliceNumber.getValue();
    voxelPos[m_cSliceAxis] = (float)sliceNum;
    SbVec3f xyzPos = m_volData->voxelToXYZ(voxelPos);
    str.sprintf( "Coronal (green) : %d / %d : %g mm",
      sliceNum, m_volDim[m_cSliceAxis], xyzPos[m_cSliceAxis]);
    m_sliceTextBox->setLine( str, m_cSliceLine );
  }
  if (node == NULL || node == m_sSlice) {
    int sliceNum = m_sSlice->sliceNumber.getValue();
    voxelPos[m_sSliceAxis] = (float)sliceNum;
    SbVec3f xyzPos = m_volData->voxelToXYZ(voxelPos);
    str.sprintf( "Sagittal  (blue)    : %d / %d : %g mm",
      sliceNum, m_volDim[m_sSliceAxis], xyzPos[m_sSliceAxis]);
    m_sliceTextBox->setLine( str, m_sSliceLine );
  }
}
