///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to render an isosurface with a SoVolumeIsosurface property node
a volumeData and a volumeRender node.
author : Benjamin Grange
September 2005
Updated by Pascal Estrade (Sep 2014)
           Mike Heck (March 2016)

Open Inventor's GPU isosurface algorithm is a powerful tool.  For example:
  - Isosurface value can be modified interactively.
  - Multiple isosurfaces can be extracted and rendered simultaneously.
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString FILENAME = "$OIVHOME/examples/data/Medical/files/medicalFoot.ldm";

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
    // Create the window
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow) return 0;

    // Initialize extensions
    SoVolumeRendering::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator;

    root->addChild( new SoPerspectiveCamera() ); // Will be adjusted later

    SoSeparator* volSep = new SoSeparator();
      root->addChild( volSep );

    // Node to hold the volume data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    // Get the volume data range.
    // For DICOM volumes use the canonical Hounsfield units range.
    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volSep->addChild(volRange);

    // Specify isosurface rendering style and isovalue(s).
    // The GPU surface algorithm allows multiple isosurfaces to be extracted
    // and rendered simultaneously.
    SoVolumeIsosurface* isoValue = new SoVolumeIsosurface;
      isoValue->isovalues.set1Value( 0,  134 ); // Bone
      isoValue->isovalues.set1Value( 1, -400 ); // Skin
      volSep->addChild( isoValue );

    // Define isosurface material properties.
    // We can specify the diffuseColor and transparency of each isosurface
    // using the indices corresponding to the ones we used in SoVolumeIsosurface.
    SoMaterial* isoMatl = new SoMaterial;  
      isoMatl->diffuseColor.set1Value( 0, 0.8f, 0.8f, 0.8f );    // Bpne
      isoMatl->transparency.set1Value( 0, 0 );

      isoMatl->diffuseColor.set1Value( 1, 1, 0.55f, 0.35f );  // Skin
      isoMatl->transparency.set1Value( 1, 0.5f );
      volSep->addChild( isoMatl );

    // Render the volume
    SoVolumeRender* volRender = new SoVolumeRender();
      volRender->numSlicesControl  = SoVolumeRender::AUTOMATIC;
      volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volSep->addChild( volRender );

   // Medical Gnomon.
    Gnomon *gnomon = new Gnomon();
    root->addChild( gnomon );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 );
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "GPU one-pass rendering of two isosurfaces" );
      root->addChild( text );

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
      myViewer->setSceneGraph(root.ptr());    
      myViewer->setTitle("Isosurface rendering (bone and skin)");
      myViewer->setTransparencyType(SoGLRenderAction::SORTED_PIXEL);
      myViewer->setDecoration( FALSE );
      myViewer->setSize( MedicalHelper::exampleWindowSize() );

    // Adjust Camera 
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, myViewer->getCamera(), volData );
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}


