///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : VSG (MMM YYYY)
** Updated by Pascal Estrade (Sep 2014)
**
** Demonstrates single pass combined volume rendering and isosurface rendering.
**
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/manips/SoROIManip.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString DATA_FILE = "$OIVHOME/examples/data/Medical/files/head.ldm";
const SbString GUI_FILE  = "$OIVHOME/examples/source/Medical/Rendering/Visualization/medicalVolumeDataDrawStyle/interface.iv";

SoVolumeDataDrawStyle* m_vds;
SoTopLevelDialog*      m_topLevelDialog;

static SoSeparator* buildSceneGraph();

///////////////////////////////////////////////////////////////////////////////
// Handle events from user interface
class InterfaceAuditor : public SoDialogAuditor
{
public:
    InterfaceAuditor(SoTopLevelDialog* topDialog)
    { m_top = topDialog; }

private:
    SoTopLevelDialog* m_top;

    void dialogCheckBox(SoDialogCheckBox* cpt)
    {
        SbBool state = cpt->state.getValue();

        if(cpt->auditorID.getValue() == "volume")
        {
            int bitmask = m_vds->style.getValue();;
            m_vds->style = state? (bitmask|SoVolumeDataDrawStyle::VOLUME_RENDER):(bitmask&~SoVolumeDataDrawStyle::VOLUME_RENDER);
        } 
        else if(cpt->auditorID.getValue() == "iso")
        {
            int bitmask = m_vds->style.getValue();;
            m_vds->style = state? (bitmask|SoVolumeDataDrawStyle::ISOSURFACE):(bitmask&~SoVolumeDataDrawStyle::ISOSURFACE);
        }
    }
};

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    SoDialogViz::init();
    InventorMedical::init();

    if (! SbFileHelper::isAccessible( DATA_FILE )) {
      new SoMessageDialog( DATA_FILE, "Cannot open file:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    if (! SbFileHelper::isAccessible( GUI_FILE )) {
      new SoMessageDialog( GUI_FILE, "Cannot open file:", SoMessageDialog::MD_ERROR );
      return -1;
    }

    SoLDMGlobalResourceParameters::setViewCulling(FALSE);
    SoLDMGlobalResourceParameters::setViewpointRefinement(FALSE);
    SoLDMGlobalResourceParameters::setScreenResolutionCulling(FALSE);

    // Create scene graph
    SoRef<SoSeparator> root = new SoSeparator;

    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
      root->addChild( camera );

    SoSeparator* scene = buildSceneGraph();
      root->addChild( scene );

    // Find node that the user interface modifies
    m_vds = MedicalHelper::find<SoVolumeDataDrawStyle>(scene);

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation
    root->addChild(new Gnomon());

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue(0, -0.99f, 0); // Normalized device coordinates
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Single-pass combined volume rendering and isosurface rendering." );
      root->addChild( text );

    // Create user interface and attach auditor
    Widget parent = MedicalHelper::buildInterface(myWindow, GUI_FILE.toLatin1(), "Viewer", &m_topLevelDialog);
    m_topLevelDialog->addAuditor(new InterfaceAuditor(m_topLevelDialog));

    // Create viewer as child of user interface
    // Note: Do not set viewer size.  It is already defined in the interface file.
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
    myViewer->setTransparencyType(SoGLRenderAction::NO_SORT);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Volume Draw Style");
    myViewer->setDecoration( FALSE );
    myViewer->setViewing( FALSE );

    // Adjust camera (more interesting view)
    myViewer->getCamera()->orientation.setValue(SbVec3f(-0.85f, -0.34f, -0.4f), 4.729f);
    myViewer->viewAll();
    myViewer->show();

    // Run then cleanup
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Build scene graph
SoSeparator* buildSceneGraph()
{
  SoSeparator* volSep = new SoSeparator();

  // Volume rendering style
  // Request both volume rendering and isosurface rendering.
  // Set isovalue = 40 and iso material as "yellow".
  //
  // NOTE: This node must be added _before_ the volume data.
  SoVolumeDataDrawStyle* volStyle = new SoVolumeDataDrawStyle();
    volStyle->style = SoVolumeDataDrawStyle::VOLUME_RENDER
                    | SoVolumeDataDrawStyle::ISOSURFACE;
    volStyle->isovalues = 40;
    SoMaterial* isoMatl = new SoMaterial();
      isoMatl->diffuseColor.setValue( 1, 1, 0 );
      volStyle->isosurfacesMaterial = isoMatl;
    volSep->addChild( volStyle );

  // Load volume data
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName = DATA_FILE;
    volSep->addChild( volData );

  // Volume rendering options
  SoVolumeRenderingQuality* volQuality = new SoVolumeRenderingQuality();
    volQuality->preIntegrated    = TRUE;
    volQuality->ambientOcclusion = TRUE;
    volQuality->deferredLighting = TRUE;
    volSep->addChild( volQuality );

  // Volume rendering base material
  SoMaterial* volMatl = new SoMaterial();
    volMatl->specularColor.setValue( 0.3f, 0.3f, 0.3f );
    volMatl->transparency = 0.5f;
    volSep->addChild( volMatl );

  // Volume rendering color map
  SoTransferFunction* volTF = new SoTransferFunction();
    volTF->predefColorMap = SoTransferFunction::GLOW;
    volTF->minValue =  37;
    volTF->maxValue = 255;
    volSep->addChild( volTF );

  // Clip the volume so both volume rendering and isosurface are visible.
  SoROIManip* volManip = new SoROIManip();
    volManip->box.setValue( 0, 35, 30, 58, 245, 230 );
    volManip->constrained = TRUE;
    volSep->addChild( volManip );

  // Rendering
  SoVolumeRender* volRender = new SoVolumeRender();
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volSep->addChild( volRender );

  return volSep;
}
