///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Sep 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : How to extract a volume from a medical dataset.
Description : Main
-------------------------------------------------------------------------*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCSGShape.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoShape.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>

#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/elements/SoViewVolumeElement.h>

#include <Inventor/misc/SbExtrusionGenerator.h>
#include <Inventor/drawers/SoLassoScreenDrawer.h>
#include <Inventor/drawers/SoPolygonScreenDrawer.h>
#include <Inventor/drawers/SoEllipseScreenDrawer.h>
#include <Inventor/drawers/SoRectangleScreenDrawer.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/SoDialogVizAll.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

const SbString DEFAULT_SCENE = "$OIVHOME/examples/source/Medical/Segmentation/medicalFreeHandCutting/model.iv";

// root of scene graph.
SoSeparator* g_root;

// Node containing volume data. It will be used to retrieve boundingBox and 
// extrusion planes. Correspond to VOLUME_DATA_ROOT, or g_root if not specified.
SoRef<SoGroup> g_volumeDataRoot;

// Root of CSGTree. Correspond to CSG_ROOT node, or g_root if CSG_ROOT not specified.
SoRef<SoGroup> g_CSGShapeRoot;

// Switch to enable/disable VolumeClippingGroup node
SoSwitch* g_volumeClippingSwitch;

// Radio button to select current csg operation.
SoDialogRadioButtons* g_csgRadio;


//------------------------------------------------------------------------------
// Callback called when their are not anough pass to properly display VolumeClippingGroup.
void notEnoughPassCB( SoVolumeClippingGroup *, void * )
{
  std::cout << "not enough pass" << std::endl;
}


//------------------------------------------------------------------------------
// Auditor of "Show LDM Topology" checkBox
class ShowTopologyAuditor : public SoDialogCheckBoxAuditor 
{
  virtual void dialogCheckBox( SoDialogCheckBox * cpt )
  {
    SoLDMGlobalResourceParameters::setVisualFeedbackParam(SoLDMGlobalResourceParameters::DRAW_TOPOLOGY, cpt->state.getValue());
  }
};

//------------------------------------------------------------------------------
// Add a new SoCSGShape which right child is specified shape. 
// By adding extruded shapes, we will build the following CSGTree:
//               S
//            S__|__
//         S__|__   |
//      R__|__   |  N4
//    __|__   |  N3
//   |     |  N2
//  NULL   N1
//
// Where S are SoCSGShape, N1, N2, ... 
// The first added is N1, the last is N4.
//
// The last added shape must be in top-right of tree to respect non-associativity 
// and non-commutativity of CSG Operations.
// For example, the following operations:
// N1 + N2 - N3, correspond to ((N1 + N2) - N3), which tree is (R represent RIGHT_ONLY):
//
//            -
//         +__|__
//      R__|__   |
//    __|__   |  N3
//   |     |  N2
//  NULL   N1
void
addShape(SoGroup* CSGTreeRoot, SoNode* newShape, SoCSGShape::CsgOperationType operation )
{
  SoCSGShape* firstCSGShape = MedicalHelper::find<SoCSGShape>(CSGTreeRoot);
  if ( firstCSGShape == NULL )
  {
    firstCSGShape = new SoCSGShape;
    firstCSGShape->rightOperand = newShape;
    firstCSGShape->csgOperation = SoCSGShape::RIGHT_ONLY;
    CSGTreeRoot->addChild( firstCSGShape );
  }
  else
  {
    SoCSGShape* newCSGShape = new SoCSGShape;
    newCSGShape->leftOperand = firstCSGShape;
    newCSGShape->csgOperation = operation;
    newCSGShape->rightOperand = newShape;
    CSGTreeRoot->replaceChild( firstCSGShape, newCSGShape );
  }
}

//------------------------------------------------------------------------------
// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line in camera space, current view volume, current model matrix
// and add an extruded shape to g_CSGShapeRoot.
void lineFinalizedCallback ( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less than 1 point, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 1 )
  {
    lineDrawer->clear();
    return;
  }

  // retrieve points of line in cam space
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
    lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

  //------------------------------------------------------------------------------
  // create a new extruded shape :

  // retrieve path of extruded shape root
  SoSearchAction searchAction;
  searchAction.setNode( g_CSGShapeRoot.ptr() );
  searchAction.apply( g_root );
  SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

  // retrieve bounding box of volumeData node.
  SbBox3f bbox = MedicalHelper::getBoundingBox( g_volumeDataRoot.ptr() );

  // create an extruded shape from specified line. Line is extruded along view 
  // direction between bounding box enclosing planes.
  SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam, 
                                                                      pathToExtrudedShapeRoot, 
                                                                      SoViewVolumeElement::get(action->getState()), 
                                                                      bbox );
  if ( extrudedShape == NULL )
  {
    lineDrawer->clear();
    return;
  }

  // Clip extrudedShape with volumeData bbox.
  SoSeparator* bboxSep = MedicalHelper::createCube(bbox);
  SoCSGShape* extrudedShapeClipped = new SoCSGShape;
  extrudedShapeClipped->leftOperand = extrudedShape;
  extrudedShapeClipped->rightOperand = bboxSep;
  extrudedShapeClipped->csgOperation = SoCSGShape::INTERSECTION;


  // if it's the first time a CSGShape is added to g_CSGShapeRoot,
  // Initialize CSGTree with the box representing volumeData.
  if ( MedicalHelper::find<SoCSGShape>(g_CSGShapeRoot.ptr(), SbString(), true) == NULL )
  {
    SoCSGShape* initialBox = new SoCSGShape;
    initialBox->rightOperand = bboxSep;
    initialBox->csgOperation = SoCSGShape::RIGHT_ONLY;
    g_CSGShapeRoot->addChild(initialBox);

    // and enable VolumeClippingGroup
    g_volumeClippingSwitch->whichChild = SO_SWITCH_ALL;
  }

  // Add this clippedShape to current CSGTree
  switch ( g_csgRadio->selectedItem.getValue() )
  {
  case 0:
    addShape( g_CSGShapeRoot.ptr(), extrudedShapeClipped, SoCSGShape::INTERSECTION );
    break;
  case 1:
    addShape( g_CSGShapeRoot.ptr(), extrudedShapeClipped, SoCSGShape::ADD );
    break;
  case 2:
    addShape( g_CSGShapeRoot.ptr(), extrudedShapeClipped, SoCSGShape::SUB );
    break;
  }


  // don't forget to clear line
  lineDrawer->clear();

  action->setHandled();
}

 
//------------------------------------------------------------------------------
class ClearButtonAuditor : public SoDialogPushButtonAuditor
{
  virtual void dialogPushButton( SoDialogPushButton* )
  {
    g_CSGShapeRoot->removeAllChildren();
    g_volumeClippingSwitch->whichChild = SO_SWITCH_NONE;
  }
};

///////////////////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
  SoPreferences::setValue("OIV_BUFFER_REGION_ENABLE", "0");

  Widget myWindow = SoXt::init(argv[0]); 
  SoDialogViz::init();
  SoVolumeRendering::init();
  InventorMedical::init();

  SoCSGShape::setRescueOperation( SoCSGShape::LEFT_ONLY );

  SoRef<SoSeparator> viewerRoot = new SoSeparator();
  
  // Manually create a camera to view the scene:
  // because SoScreenDrawer contains an orthographic camera to draw line, 
  // SoXtExaminerViewer will not insert a camera in top of sceneGraph.
  SoCamera* camera = new SoOrthographicCamera;
  viewerRoot->addChild( camera );

  // Medical Gnomon.
  viewerRoot->addChild( new Gnomon() );
  
  // DICOM annotation
  viewerRoot->addChild( MedicalHelper::exampleDicomAnnotation( "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm" ) );

  // Instructions
  TextBox* text = new TextBox();
    text->position.setValue( -0.98f, 0.f, 0.f ); // Normalized device coords -1..1
    text->alignmentH = TextBox::LEFT;
    text->alignmentV = TextBox::MIDDLE;
    text->addLine( "Click and drag to outline clip region." );
    viewerRoot->addChild( text );
  
  //------------------------------------------------------------------------------
  // load scene

  SbString filename;
  if ( argc == 1 )
     filename = DEFAULT_SCENE;
  else
    filename = SbString(argv[1]);

  SoInput myInput;
  if (! myInput.openFile(filename) ) 
  {
    new SoMessageDialog( DEFAULT_SCENE, "Unable to open:", SoMessageDialog::MD_ERROR );
    exit(1);
  }

  g_root = SoDB::readAll(&myInput);
  if (g_root == NULL) {
    std::cerr << "Cannot create sceneGraph." << std::endl;
    exit(1);
  }
  viewerRoot->addChild( g_root );
  g_root->setName("ROOT");

  // Define Open Inventor logo
  g_root->addChild( MedicalHelper::exampleLogoNode() );

  //------------------------------------------------------------------------------
  // retrieve volume data root to compute its bounding box

  g_volumeDataRoot = MedicalHelper::find<SoGroup>( g_root, "VOLUME_DATA_ROOT" );
  // If cannot find VOLUME_DATA_ROOT, use root.
  if ( g_volumeDataRoot.ptr() == NULL )
    g_volumeDataRoot = g_root;
  
  //------------------------------------------------------------------------------
  // retrieve VolumeClippingGroup switch

  g_volumeClippingSwitch = MedicalHelper::find<SoSwitch>( g_root, "VOLUME_CLIPPING_ROOT" );

  //------------------------------------------------------------------------------
  // retrieve extruded shapes root

  g_CSGShapeRoot = MedicalHelper::find<SoGroup>( g_root, "CSG_ROOT" );
  // If cannot find CSG_ROOT, use root.
  if ( g_CSGShapeRoot.ptr() == NULL )
    g_CSGShapeRoot = g_root;

  //------------------------------------------------------------------------------
  // Add line drawers
  SoSeparator* drawers = new SoSeparator;
  drawers->fastEditing = SoSeparator::CLEAR_ZBUFFER;
  drawers->boundingBoxIgnoring = TRUE;
  drawers->setName("DRAWERS");
  SoSwitch* lineDrawerSwitch = new SoSwitch();
  lineDrawerSwitch->setName("LINE_DRAWER_SWITCH");
  lineDrawerSwitch->whichChild = 0;
  drawers->addChild(lineDrawerSwitch);
  g_root->addChild(drawers);

  // lasso
  SoLassoScreenDrawer* lasso = new SoLassoScreenDrawer();
  lasso->color.setValue(1, 0, 0);
  lasso->setName("LASSO_SCREEN_DRAWER");
  lasso->onFinish.add(lineFinalizedCallback);
  lasso->isClosed = TRUE;
  lineDrawerSwitch->addChild(lasso);

  // polygon
  SoPolygonScreenDrawer* polygon = new SoPolygonScreenDrawer;
  polygon->color.setValue(0, 1, 0);
  polygon->setName("POLYGON_SCREEN_DRAWER");
  polygon->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(polygon);

  // ellipse corner
  SoEllipseScreenDrawer* ellipseCorner = new SoEllipseScreenDrawer;
  ellipseCorner->color.setValue(0, 0, 1);
  ellipseCorner->method = SoEllipseScreenDrawer::CORNER_CORNER;
  ellipseCorner->setName("ELLIPSE_CORNER_SCREEN_DRAWER");
  ellipseCorner->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(ellipseCorner);

  // ellipse center
  SoEllipseScreenDrawer* ellipseCenter = new SoEllipseScreenDrawer;
  ellipseCenter->color.setValue(1, 1, 0);
  ellipseCenter->method = SoEllipseScreenDrawer::CENTER_CORNER;
  ellipseCenter->setName("ELLIPSE_CENTER_SCREEN_DRAWER");
  ellipseCenter->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(ellipseCenter);

  // Rectangle corner
  SoRectangleScreenDrawer* rectangleCorner = new SoRectangleScreenDrawer;
  rectangleCorner->color.setValue(1, 1, 1);
  rectangleCorner->method = SoRectangleScreenDrawer::CORNER_CORNER;
  rectangleCorner->setName("RECTANGLE_CORNER_SCREEN_DRAWER");
  rectangleCorner->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(rectangleCorner);

  // Rectangle center
  SoRectangleScreenDrawer* rectangleCenter = new SoRectangleScreenDrawer;
  rectangleCenter->color.setValue(0.5, 1, 0);
  rectangleCenter->method = SoRectangleScreenDrawer::CENTER_CORNER;
  rectangleCenter->setName("RECTANGLE_CENTER_SCREEN_DRAWER");
  rectangleCenter->onFinish.add(lineFinalizedCallback);
  lineDrawerSwitch->addChild(rectangleCenter);

  //------------------------------------------------------------------------------
  // Build dialog.

  const SbString InterfaceName = "$OIVHOME/examples/source/Medical/Segmentation/medicalFreeHandCutting/interface.iv";
  SoInput myInputInterface;
  if ( !myInputInterface.openFile(InterfaceName) ) 
  {
    std::cerr << "Cannot open " << InterfaceName.getString() << std::endl;
    exit(1);
  }

  SoRef<SoGroup> myGroup = SoDB::readAll( &myInputInterface );
  if ( !myGroup.ptr() )
  {  
    fprintf(stderr, "Cannot create interface.");
    exit(1);
  }
  
  SoRef<SoTopLevelDialog> myTop = MedicalHelper::find<SoTopLevelDialog>( myGroup.ptr() );
  myTop->buildDialog( myWindow, true );

  SoDialogCustom* myCustom = MedicalHelper::find<SoDialogCustom>( myTop.ptr(), "DIALOG_CUSTOM" );


  // combo box to select lineDrawer
  SoDialogComboBox* modeRadio = MedicalHelper::find<SoDialogComboBox>( myTop.ptr(), "MODE_RADIO" );
  lineDrawerSwitch->whichChild.connectFrom(&modeRadio->selectedItem);

  // combo box to select csgOperation
  g_csgRadio = MedicalHelper::find<SoDialogRadioButtons>( myTop.ptr(), "CSG_RADIO" );

  // CheckBox to clip inside/outside of csg group (only if there is a volume clipping group)
  SoDialogCheckBox* clipOutside = MedicalHelper::find<SoDialogCheckBox>( myTop.ptr(), "CLIP_OUTSIDE_CHECK" );
  SoVolumeClippingGroup* volumeClippingGroup = MedicalHelper::find<SoVolumeClippingGroup>(g_root);
  if ( volumeClippingGroup != NULL )
  {
    volumeClippingGroup->setNotEnoughPassCallback(notEnoughPassCB, NULL);
    volumeClippingGroup->clipOutside.connectFrom(&clipOutside->state);
  }
  else
    clipOutside->enable = FALSE;

  // CheckBox to show/hide LDMTopology
  SoDialogCheckBox* showTopologyCheck = MedicalHelper::find<SoDialogCheckBox>( myTop.ptr(), "ShowTopology" );
  ShowTopologyAuditor* showTopologyAuditor = new ShowTopologyAuditor;
  if ( MedicalHelper::find<SoVolumeRender>(g_root) != NULL )
    showTopologyCheck->addAuditor(showTopologyAuditor);
  else
    showTopologyCheck->enable = FALSE;


  // Combobox to select which object to show
  SoSwitch* showSwitch = MedicalHelper::find<SoSwitch>( g_root, "SHOW_SWITCH" );
  SoDialogComboBox* showSwitchCombo = MedicalHelper::find<SoDialogComboBox>( myTop.ptr(), "SHOW_SWITCH_COMBO" );
  if ( showSwitch != NULL )
  {
    for ( int i = 0; i < showSwitch->getNumChildren(); ++i)
    {
      SoNode* child = showSwitch->getChild(i);
      showSwitchCombo->items.set1Value(i, child->getName().getString());
    }
    showSwitch->whichChild.connectFrom( &showSwitchCombo->selectedItem );
  }
  else
  {
    showSwitchCombo->enable = FALSE;
  }

  // Clear button
  SoDialogPushButton* clearButton = MedicalHelper::find<SoDialogPushButton>( myTop.ptr(), "CLEAR_BUTTON" );
  ClearButtonAuditor* clearButtonAuditor = new ClearButtonAuditor();
  clearButton->addAuditor( clearButtonAuditor );

  myTop->buildDialog(myWindow, TRUE);
  myTop->show();

  SoXtExaminerViewer* myViewer = new SoXtExaminerViewer(myCustom->getWidget());
  myViewer->setDecoration(FALSE);
  myViewer->setViewing(FALSE);
  myViewer->setAntialiasing(0.3f,SoSceneManager::FSAA);
  myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  myViewer->setFastEditSavePolicy(SoGLRenderAction::WHEN_NEEDED);
  myViewer->setSceneGraph(viewerRoot.ptr());

  // Adjust camera
  SoVolumeData* volData = MedicalHelper::find<SoVolumeData>( g_root, "VOLUME_DATA" );
  MedicalHelper::orientView( MedicalHelper::CORONAL, camera, volData );
  myViewer->saveHomePosition();

  // Run then cleanup
  myViewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete myViewer;
  delete showTopologyAuditor;
  delete clearButtonAuditor;
  viewerRoot = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoDialogViz::finish();
  SoXt::finish();
  return 0;
}
