///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Simple implementation of classic Marching Cubes (isosurface
          extraction) for VolumeViz.
Description : Main
-------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoSearchAction.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoAnnotation.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>

#include <VolumeViz/nodes/SoObliqueSlice.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <LDM/SoLDMGlobalResourceParameters.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <LDM/readers/SoLDMReader.h>

#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/SbElapsedTime.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoMenuFileSelection.h>

#include <Inventor/STL/iostream>
#include <Inventor/STL/algorithm>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/helpers/VolumeMarchingCubes.h>
#include <Medical/nodes/Gnomon.h>


////////////////////////////////////////////////////////////////////////
// Application constants

const char* VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/files/medicalFoot.ldm";
const char* DIALOG_FILENAME = "$OIVHOME/examples/source/Medical/Segmentation/medicalMarchingCubesSurface/gui_isoSurface.iv";

const float DEFAULT_ISOVALUE = 134.0;

////////////////////////////////////////////////////////////////////////
// User interface utility

class UserInterface {
public:
  UserInterface() : m_topLevelDialog(NULL), m_window(NULL) {}

  Widget buildInterface(Widget); // Returns primary viewer widget
  Widget getViewerWidget();
  Widget getViewer2Widget();
  SoTopLevelDialog* getTopLevelDialog() const;
protected:
  SoTopLevelDialog* m_topLevelDialog;
  Widget            m_window;
};

////////////////////////////////////////////////////////////////////////
// Application state for user interface
//
// Some nodes we need to access in the user interface.

static SoVolumeData*        m_volData = NULL;
static SoDataRange*         m_volRange = NULL;
static SoVolumeIsosurface*  m_isoNode = NULL;
static SoSeparator*         m_bboxGeomSep = NULL;
static SoSeparator*         m_mcGeomSep = NULL;

static VolumeMarchingCubes* m_marchingCubes = NULL;

static SoXtExaminerViewer*  m_viewer = NULL;
static UserInterface*       m_gui = NULL;
static SoMenuFileSelection* m_fileSelection = NULL; // Open file dialog

////////////////////////////////////////////////////////////////////////
// Forward decls

void EventCB(void* userData, SoEventCallback* node);

static const char* dataTypeName(SoDataSet::DataType type);

////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////

int
main(int /*argc*/, char** argv)
{
  // Initialize Inventor
  Widget myWindow = SoXt::init(argv[0]);
  SoVolumeRendering::init();
  SoDialogViz::init();
  InventorMedical::init();

  if (! SbFileHelper::isAccessible( VOLUME_FILENAME )) {
    new SoMessageDialog( VOLUME_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
    return -1;
  }

  std::cout << "Press 'F' to open a new file\n";

  // Root of scene graph
  SoRef<SoSeparator> root = new SoSeparator;

  // Camera (must have explicit camera to use Gnomon node)
  root->addChild( new SoPerspectiveCamera() );

  SoEventCallback* evCB = new SoEventCallback;
  evCB->addEventCallback(SoKeyboardEvent::getClassTypeId(), EventCB, NULL);
  root->addChild(evCB);

  // Keep volume viz separate from geometry
  SoSeparator* volSep = new SoSeparator();
  root->addChild(volSep);

  m_volData = new SoVolumeData();
  SoVolumeData* volData = m_volData;
  volSep->addChild(volData);
  volData->fileName = VOLUME_FILENAME;


  // Get volume properties
  SbVec3i32           volDim = volData->data.getSize();
  SbBox3f             volExt = volData->extent.getValue();
  SbVec3i32           tileDim = volData->getTileDimension();
  int                 voxSize = volData->getDatumSize();
  SoDataSet::DataType voxType = volData->getDataType();
  SoVolumeReader*     reader = volData->getReader();

  // Adjust tile size to hold volume if possible.
  // (Does not apply to LDM format volumes -- tile size cannot be changed in that case.)
  if (!reader->isOfType(SoLDMReader::getClassTypeId())) {
    int maxDim = std::max(volDim[0], std::max(volDim[1], volDim[2]));
    int newTileDim = std::min(maxDim, 512);
    tileDim.setValue(newTileDim, newTileDim, newTileDim);
    volData->ldmResourceParameters.getValue()->tileDimension = tileDim;
  }

  // Get the volume data range.
  // For DICOM volumes use the canonical Hounsfield units range.
  double volMin, volMax;
  if (reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
    volMin = -1000;
    volMax = 3000;
  }
  else {
    volData->getMinMax(volMin, volMax);
    printf("min %f max %f \n", volMin, volMax);
  }

  // Set range of data values to visualize.
  m_volRange = new SoDataRange();
  m_volRange->min = volMin;
  m_volRange->max = volMax;
  volSep->addChild(m_volRange);

  // What kind of volume do we have?
  std::cout << "Volume: " << SbFileHelper::getBaseName(volData->fileName.getValue()) << std::endl;
  std::cout << "   dim: " << volDim << std::endl;
  std::cout << "   ext: " << volExt.getMin() << " : " << volExt.getMax() << std::endl;
  std::cout << "  tile: " << tileDim[0] << std::endl;
  std::cout << " voxel: " << dataTypeName(voxType) << " (" << voxSize << " bytes)" << std::endl;
  std::cout << " range: " << volMin << " : " << volMax << std::endl;

  //// Display voxels at full intensity
  //SoMaterial* volMat = new SoMaterial();
  //  volMat->diffuseColor.setValue( 1, 1, 1 );
  //  volSep->addChild( volMat );

  SoTransferFunction* volTF = new SoTransferFunction();
  volTF->predefColorMap = SoTransferFunction::INTENSITY;
  volSep->addChild(volTF);

  m_isoNode = new SoVolumeIsosurface();
  m_isoNode->isovalues = DEFAULT_ISOVALUE;
  volSep->addChild(m_isoNode);

  // Display GPU isosurface
  SoVolumeRender* volRend = new SoVolumeRender();
  // Let Inventor compute best number of slices
  volRend->numSlicesControl = SoVolumeRender::AUTOMATIC;
  // Internal optimization.
  volRend->subdivideTile = TRUE;
  volSep->addChild(volRend);

  // Display volume bounding box for reference.
  m_bboxGeomSep = MedicalHelper::createBoundingBox(volData->extent.getValue());
  root->addChild(m_bboxGeomSep);

  // Create a simple user interface to move the slice
  m_gui = new UserInterface();
  Widget parent = m_gui->buildInterface(myWindow);

  // Define Open Inventor logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Orientation marker
  root->addChild( new Gnomon() );

  // Create a viewer for the GPU isosurface
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer(parent);
  viewer->setBackgroundColor(SbColor(0, .1f, .1f));
  viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
  viewer->setSceneGraph(root.ptr());
  viewer->setTitle("GPU Isosurface");
  viewer->setSize(SbVec2s(512, 512));
  viewer->setDecoration(FALSE);
  viewer->viewAll();
  viewer->saveHomePosition();
  viewer->show();
  m_viewer = viewer;

  //-----------------------------------------------------------------------
  // Create another scene graph and viewer for the marching cubes geometry

  SoRef<SoSeparator> root2 = new SoSeparator();

  root2->addChild(evCB);

  root2->addChild(viewer->getCamera());

  root2->addChild(m_bboxGeomSep);

  // Specify a consistent ordering for the extracted triangles and also specify
  // a "crease angle" to smooth the edges for less faceted appearance.
  SoShapeHints* hints = new SoShapeHints();
  hints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  hints->creaseAngle = (float)M_PI;
  root2->addChild(hints);

  m_mcGeomSep = new SoSeparator(); // Holds marching cubes geometry
  root2->addChild(m_mcGeomSep);

  // Create utility object for extracting isosurface geometry.
  m_marchingCubes = new VolumeMarchingCubes();

  Widget parent2 = m_gui->getViewer2Widget();

  SoXtExaminerViewer* viewer2 = new SoXtExaminerViewer(parent2);
  viewer2->setBackgroundColor(SbColor(0, 0.1f, 0.1f));
  viewer2->setSceneGraph(root2.ptr());
  viewer2->setTitle("MarchingCubes");
  viewer2->setSize(SbVec2s(512, 512));
  viewer2->setDecoration(FALSE);
  viewer2->show();

  //HACK
  root->setName("Root");
  root2->setName("Root2");

  // Loop, then cleanup
  SoXt::show(myWindow);
  SoXt::mainLoop();
  delete viewer;
  delete viewer2;
  root = nullptr;
  root2 = nullptr;
  delete m_marchingCubes;
  InventorMedical::finish();
  SoDialogViz::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}

/////////////////////////////////////////////////////////////

void EventCB(void* /*userData*/, SoEventCallback* node)
{
  const SoEvent* event = node->getEvent();

  //-------------------------------------------------------------------
  // Pressing 'I' -> Extract isosurface
  if (SoKeyboardEvent::isKeyPressEvent(event, SoKeyboardEvent::I)) {
    node->setHandled();

    // Delete the current geometry (frees memory)
    m_mcGeomSep->removeAllChildren();

    // Extract new geometry
    float isovalue = m_isoNode->isovalues[0];
    std::cout << "Begin isosurface extraction...\n";
    SbElapsedTime timer;
    SoTriangleSet* scene = m_marchingCubes->getIsosurface(*m_volData, isovalue);
    double elapsed = timer.getElapsed();
    size_t ntris = m_marchingCubes->getNumTriangles();
    std::cout << "  Finished. " << ntris << " triangles in " << elapsed << " secs\n";

    // Add to scene graph for inspection
    m_mcGeomSep->addChild(scene);

    SoGetBoundingBoxAction gbba(SbViewportRegion(500, 500));
    gbba.apply(SoNode::getByName("Root"));
    SbBox3f bbox = gbba.getBoundingBox();
    gbba.clearApplyResult();
    gbba.apply(m_mcGeomSep);
    SbBox3f bbox2 = gbba.getBoundingBox();
    std::cout << "  bbox : " << bbox.getMin() << " : " << bbox.getMax() << std::endl;
    std::cout << "  bbox2: " << bbox2.getMin() << " : " << bbox2.getMax() << std::endl;
  }
  //-------------------------------------------------------------------
  // Pressing 'F' -> Open File
  else if (SoKeyboardEvent::isKeyPressEvent(event, SoKeyboardEvent::F)) {
    node->setHandled();
    // Create file selection dialog if necessary.
    if (m_fileSelection == NULL) {
      m_fileSelection = new SoMenuFileSelection();
      m_fileSelection->filter.set1Value(0, "*.*"); // Allow any file (too many extensions to list all)
      m_fileSelection->filter.set1Value(1, "All files");
    }
    // Display file selection dialog
    m_fileSelection->menuEvent(NULL, 0);

    // Get selected filename
    SbString filepath = m_fileSelection->fileDirectory.getValue();
    const SbString& filename = m_fileSelection->filename.getValue();
    filepath += "/";
    filepath += filename;

    if (filepath.isEmpty()) // If no file was selected, return.
      return;

    if (SbFileHelper::isAccessible(filepath)) { // If selected file can be opened...

      // Load new volume
      m_volData->fileName = filepath;

      //SoVRDicomFileReader* dicomReader = new SoVRDicomFileReader();
      //dicomReader->setFilenameList( "C:\\VSGDemo\\Medical960\\data\\dicomSample\\CVH%03d.dcm", 1, 557 );
      //volData->setReader( *dicomReader );

      // Get volume properties
      SbVec3i32           volDim = m_volData->data.getSize();
      SbBox3f             volExt = m_volData->extent.getValue();
      SbVec3i32           tileDim = m_volData->getTileDimension();
      int                 voxSize = m_volData->getDatumSize();
      SoDataSet::DataType voxType = m_volData->getDataType();
      SoVolumeReader*     reader = m_volData->getReader();

      // Adjust tile size to hold volume if possible.
      // (Does not apply to LDM format volumes -- tile size cannot be changed in that case.)
      if (!reader->isOfType(SoLDMReader::getClassTypeId())) {
        int maxDim = std::max(volDim[0], std::max(volDim[1], volDim[2]));
        int newTileDim = std::min(maxDim, 512);
        tileDim.setValue(newTileDim, newTileDim, newTileDim);
        m_volData->ldmResourceParameters.getValue()->tileDimension = tileDim;
      }

      // Get the volume data range.
      // For DICOM volumes use the canonical Hounsfield units range.
      double volMin, volMax;
      float  isoValue;
      if (reader->isOfType(SoVRDicomFileReader::getClassTypeId())) {
        volMin = -1000;
        volMax = 3000;
        isoValue = 1500;
      }
      else {
        m_volData->getMinMax(volMin, volMax);
        if (volMin == volMax) { // Some volumes are known to return bogus values.
          volMin = 0;
          volMax = 255;
        }
        isoValue = 0.5f * (float)(volMax - volMin);
      }

      // Set range of data values to visualize.
      m_volRange->min = volMin;
      m_volRange->max = volMax;
      m_isoNode->isovalues.set1Value(0, isoValue);

      // What kind of volume do we have?
      std::cout << "Volume: " << SbFileHelper::getBaseName(m_volData->fileName.getValue()) << std::endl;
      std::cout << "   dim: " << volDim << std::endl;
      std::cout << "   ext: " << volExt.getMin() << " : " << volExt.getMax() << std::endl;
      std::cout << "  tile: " << tileDim[0] << std::endl;
      std::cout << " voxel: " << dataTypeName(voxType) << " (" << voxSize << " bytes)" << std::endl;
      std::cout << " range: " << volMin << " : " << volMax << std::endl;

      // Replace the bounding box geometry.
      m_bboxGeomSep->removeAllChildren();
      m_bboxGeomSep->addChild(MedicalHelper::createBoundingBox(m_volData->extent.getValue()));

      // Delete the current isosurface geometry.
      m_mcGeomSep->removeAllChildren();

      // Reset to default view orientation for each new volume.
      m_viewer->getCamera()->orientation.setValue(SbVec3f(0, 0, 1), 0);
      m_viewer->viewAll();
      m_viewer->saveHomePosition();

      // Clear info display in user interface
      SoDialogLabel* infoLabel =
        (SoDialogLabel*)m_gui->getTopLevelDialog()->searchForAuditorId(SbString("MCInfo"));
      if (infoLabel != NULL)
        infoLabel->label.setValue("");
      SoDialogLabel* rangeLabel =
        (SoDialogLabel*)m_gui->getTopLevelDialog()->searchForAuditorId(SbString("IsoRange"));
      if (rangeLabel != NULL) {
        SbString str;
        str.sprintf("Range %g : %g", volMin, volMax);
        rangeLabel->label = str;
      }
      SoDialogRealSlider* slider =
        (SoDialogRealSlider *)m_gui->getTopLevelDialog()->searchForAuditorId(SbString("IsoValue"));
      if (slider != NULL) {
        slider->min = (float)volMin;
        slider->max = (float)volMax;
        slider->value = isoValue;
      }
    }
    else {
      std::cerr << "Failed to load '" << filepath.toLatin1() << "'\n";
    }
  }
}

///////////////////////////////////////////////////////////////////////
// Utility to help printing information about the volume

static const char* dataTypeName(SoDataSet::DataType type)
{
  switch (type) {
  default:
  case SoDataSet::UNSIGNED_BYTE:
    return "Unsigned Byte";
  case SoDataSet::UNSIGNED_SHORT:
    return "Unsigned Short";
  case SoDataSet::UNSIGNED_INT32:
    return "Unsigned Int";
  case SoDataSet::SIGNED_BYTE:
    return "Signed Byte";
  case SoDataSet::SIGNED_SHORT:
    return "Signed Short";
  case SoDataSet::SIGNED_INT32:
    return "Signed Int";
  case SoDataSet::FLOAT:
    return "Float";
  }
}

///////////////////////////////////////////////////////////////////////
//
// Create a wireframe box showing bounding box of volume
//

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>


////////////////////////////////////////////////////////////////////////

// DialogViz auditor class to handle user input
class myAuditorClass : public SoDialogAuditor
{
  void dialogPushButton(SoDialogPushButton* cpt);
  void dialogRealSlider(SoDialogRealSlider* cpt);
};

// Auditor method for slider input
void
myAuditorClass::dialogRealSlider(SoDialogRealSlider* cpt)
{
  // Move slice
  if (cpt->auditorID.getValue() == "IsoValue") {
    if (m_isoNode != NULL) {
      float isovalue = cpt->value.getValue();
      m_isoNode->isovalues.set1Value(0, isovalue);
    }
  }
}

// Auditor method for push button
void
myAuditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
  if (cpt->auditorID.getValue() == "Extract") {
    if (m_mcGeomSep != NULL && m_marchingCubes != NULL) {

      // Delete the current geometry (frees memory)
      m_mcGeomSep->removeAllChildren();
      float isovalue = m_isoNode->isovalues[0];

      // Extract new geometry
      std::cout << "Begin isosurface extraction...\n";
      SbElapsedTime timer;
      SoTriangleSet* scene = m_marchingCubes->getIsosurface(*m_volData, isovalue);
      double elapsed = timer.getElapsed();
      size_t ntris = m_marchingCubes->getNumTriangles();
      std::cout << "  Finished. " << ntris << " triangles in " << elapsed << " secs\n";

      // Add to scene graph for inspection
      m_mcGeomSep->addChild(scene);

      SoDialogLabel* infoLabel =
        (SoDialogLabel*)m_gui->getTopLevelDialog()->searchForAuditorId(SbString("MCInfo"));
      if (infoLabel != NULL) {
        SbString str;
        str.sprintf("Isovalue: %g  Tris: %d  (%g sec)", m_marchingCubes->getIsovalue(), ntris, elapsed);
        infoLabel->label = str;
      }
    }
  }
}

////////////////////////////////////////////////////////////////////////
// Build user interface with embedded viewer

Widget
UserInterface::buildInterface(Widget window)
{
  SoInput myInput;
  if (!myInput.openFile(DIALOG_FILENAME)) {
    fprintf(stderr, "ERROR opening dialogviz file '%s'\n", DIALOG_FILENAME);
    return NULL;
  }

  SoGroup* myGroup = SoDB::readAll(&myInput);
  if (myGroup == NULL) {
    fprintf(stderr, "ERROR reading dialogviz file '%s'\n", DIALOG_FILENAME);
    return NULL;
  }

  m_topLevelDialog = (SoTopLevelDialog *)myGroup->getChild(0);

  // Create and register auditor to handle user input
  myAuditorClass *myAuditor = new myAuditorClass;
  m_topLevelDialog->addAuditor(myAuditor);

  // Initialize isovalue slider
  SoDialogRealSlider* slider =
    (SoDialogRealSlider *)m_topLevelDialog->searchForAuditorId(SbString("IsoValue"));
  if (slider != NULL) {
    float minval = 0;
    float maxval = 255;
    float isoval = 95;
    if (m_volRange) {
      minval = (float)m_volRange->min.getValue();
      maxval = (float)m_volRange->max.getValue();
    }
    if (m_isoNode) {
      isoval = m_isoNode->isovalues[0];
    }
    slider->min = minval;
    slider->max = maxval;
    slider->value = isoval;

    // Initialize isovalue range
    SoDialogLabel* rangeLabel =
      (SoDialogLabel*)m_topLevelDialog->searchForAuditorId(SbString("IsoRange"));
    if (rangeLabel != NULL) {
      SbString str;
      str.sprintf("Range %g : %g", minval, maxval);
      rangeLabel->label = str;
    }
  }

  // Initialize triangle count (assumes extraction has already been run).
  SoDialogLabel* infoLabel =
    (SoDialogLabel*)m_topLevelDialog->searchForAuditorId(SbString("MCInfo"));
  if (infoLabel != NULL && m_marchingCubes != NULL) {
    size_t count = m_marchingCubes->getNumTriangles();
    SbString str;
    str.sprintf("%d triangles", count);
    infoLabel->label = str;
  }

  // Build dialog
  SoDialogCustom *customNode = (SoDialogCustom *)m_topLevelDialog->searchForAuditorId(SbString("Viewer"));
  m_topLevelDialog->buildDialog(window, customNode != NULL);
  m_topLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}

Widget
UserInterface::getViewerWidget()
{
  SoDialogCustom* customNode = (SoDialogCustom *)m_topLevelDialog->searchForAuditorId(SbString("Viewer"));
  return customNode ? customNode->getWidget() : m_window;
}

Widget
UserInterface::getViewer2Widget()
{
  SoDialogCustom* customNode = (SoDialogCustom *)m_topLevelDialog->searchForAuditorId(SbString("Viewer2"));
  return customNode ? customNode->getWidget() : m_window;
}

SoTopLevelDialog*
UserInterface::getTopLevelDialog() const
{
  return m_topLevelDialog;
}

