///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volumeRender node.
author : David Beilloin
Updaded by Pascal Estrade (Sep 2014)
August 2002
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSphere.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/files/head.ldm";

const SbString COLORMAP_FILENAME = "$OIVHOME/examples/data/Medical/resources/volrenGlow.am";

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
    // Initialize and create the window
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator();

    root->addChild( new SoPerspectiveCamera() );  // Will be adjusted later

    // Complexity controls the number of vertices in the sphere.
    // This goes in the top level of the scene so it affects both the
    // clipping sphere (in the clipping group) and the visible sphere.
    SoComplexity* sphereComplexity = new SoComplexity();
      sphereComplexity->value = 1.0;
      root->addChild( sphereComplexity );

    // Volume rendering
    SoSeparator* volSep = new SoSeparator();
      volSep->setName( "VolSep" ); // Just for convenience in IvTune
      root->addChild( volSep );

    // Volume data  
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILENAME;
      volSep->addChild( volData );

    // Colormap
    SoTransferFunction* volTF = new SoTransferFunction;
      volTF->loadColormap(COLORMAP_FILENAME);
      volTF->minValue = 30;
      volTF->maxValue = 159;
      volSep->addChild( volTF );

    // Volume rendering options
    SoVolumeRenderingQuality* volrq = new SoVolumeRenderingQuality;
      volrq->ambientOcclusion = TRUE;
      volrq->deferredLighting = TRUE;
      volSep->addChild( volrq );

    // Group to hold the clipping geometry.
    // Remember this is a *group* and state will leak out of it.
    SoVolumeClippingGroup* volClipGroup = new SoVolumeClippingGroup;
      volSep->addChild( volClipGroup );

    // Clip with a sphere
    SoSphere* sphere = new SoSphere;
      sphere->radius = 0.6f;
      volClipGroup->addChild( sphere );

    // Render the volume
    SoVolumeRender* volRender = new SoVolumeRender;
      volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volRender->interpolation = SoVolumeRender::CUBIC;
      volSep->addChild( volRender );

    // To render the clipping object (the sphere), instance again in main scene graph
    SoSeparator* sphereSep = new SoSeparator();
      sphereSep->setName( "SphereSep" );
      root->addChild( sphereSep );

    SoDrawStyle* sphereDS = new SoDrawStyle();
      sphereDS->style = SoDrawStyle::POINTS;
      sphereSep->addChild(sphereDS);
      sphereSep->addChild( sphere );

    // Outline volume extent
    SbColor color(1, 0.5f, 0);
    root->addChild(MedicalHelper::createBoundingBox(volData->extent.getValue(), &color));

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation
    root->addChild( new Gnomon() );

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 ); // Normalized device coords
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Volume clipped by a sphere" );
      root->addChild( text );

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Clipping Group");
    myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    myViewer->setDecoration( FALSE );
    myViewer->setSize( MedicalHelper::exampleWindowSize() );

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, myViewer->getCamera() );
    myViewer->viewAll();
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}


