///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Updated by Pascal Estrade (Sep 2014)
**
** Demonstrates using "mask" volumes to apply different transfer functions
** to different regions of the volume.
**
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoRotationXYZ.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeDataDrawStyle.h>
#include <VolumeViz/nodes/SoVolumeIsosurface.h>
#include <VolumeViz/nodes/SoVolumeMask.h>
#include <VolumeViz/nodes/SoVolumeMaskGroup.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <Inventor/STL/vector>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/files/spine.lda";

/*******************************************************************************/
const size_t NUM_MASKS = 3;

void generateMasks(SoVolumeData* vd, const std::vector<SoVolumeMask*>& vmList, const std::vector<SoVolumeDataDrawStyle*>& vmStyleList);
SoTransferFunction* generateTf(SbVec3f color);

/*******************************************************************************/
int main(int, char **argv)
{
    // Initialize and create the window
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator();

    // Camera (rotate for nicer view)
    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
      camera->orientation.setValue( SbVec3f(1,0,0), (float)M_PI/2 );
      root->addChild( camera );

    // Combining multiple volumes, so must use special Separator
    SoMultiDataSeparator* volSep = new SoMultiDataSeparator();
      root->addChild( volSep );

    // Node to hold the volume data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILENAME;
      volSep->addChild( volData );

    SoDataRange* volRange = new SoDataRange();
      volRange->min = 0;
      volRange->max = 2000;
      volSep->addChild( volRange );

    // Material which defines the isosurface diffuse color, transparency,
    // specular color and shininess
    SoMaterial* matVolRend = new SoMaterial;
      matVolRend->specularColor.setValue(0.3f,0.3f,0.3f);
      matVolRend->shininess.setValue(0.1f);
      matVolRend->transparency.setValue(0.8f);
      volSep->addChild( matVolRend );

    //This draw style will be applied on the whole volume
    SoVolumeDataDrawStyle* vdsGlobal = new SoVolumeDataDrawStyle;
      vdsGlobal->style = SoVolumeDataDrawStyle::VOLUME_RENDER;
      volSep->addChild( vdsGlobal );

    // Add lighting and preintegration
    SoVolumeRenderingQuality* vrq = new SoVolumeRenderingQuality;
      vrq->ambientOcclusion = TRUE;
      vrq->deferredLighting = TRUE;
      volSep->addChild( vrq );

    // Draw style and transfer function for intersecting masks
    SoVolumeDataDrawStyle* vdsIntersection = new SoVolumeDataDrawStyle;
      vdsIntersection->style = SoVolumeDataDrawStyle::VOLUME_RENDER;

    SoTransferFunction* tfIntersection = new SoTransferFunction;
      tfIntersection->predefColorMap = SoTransferFunction::GREY;
      tfIntersection->minValue = 185;
      volSep->addChild( tfIntersection );
    //tfIntersection->transferFunctionId = 1;//SoVolumeMaskGroup::TRANSFERFUNCTION_INTERSECTION_ID;

    // Add masks/styles/transfer funtion in the SoVolumeMaskGroup
    SoVolumeMaskGroup* vmg = new SoVolumeMaskGroup;
    std::vector<SoVolumeMask*> vmList;
    std::vector<SoVolumeDataDrawStyle*> vmStyleList;
    for ( size_t i = 0; i < NUM_MASKS; i++ )
    {
        // Create a mask volume (must have same extent and tile dimension as data volume).
        SoVolumeMask* vm = new SoVolumeMask;
        vm->ldmResourceParameters.getValue()->tileDimension = volData->ldmResourceParameters.getValue()->tileDimension;
        vm->extent = volData->extent;
        vm->dataSetId = volData->dataSetId.getValue()+int(i)+1;
        vmList.push_back(vm);

        vmStyleList.push_back(new SoVolumeDataDrawStyle);

        // Create transfer functions with different colors
        SoTransferFunction* tf;
        if (i == 0)
          tf = generateTf( SbColor(1,0,0) );
        else if (i == 1)
          tf = generateTf( SbColor(0,1,0) );
        else if (i == 2)
          tf = generateTf( SbColor(0,0,1) );
        else {
          SbVec3f cmColor = SbVec3f((float)rand(), (float)rand(), (float)rand())/float(RAND_MAX);
          tf = generateTf(cmColor);
        }
        tf->transferFunctionId.connectFrom(&vm->dataSetId);
        vmg->addChild(tf);
        vmg->addChild(vmStyleList.back());
        vmg->addChild(vm);
    }
    volSep->addChild( vmg );

    generateMasks(volData, vmList, vmStyleList);

    // Node in charge of drawing the volume
    SoVolumeRender* volRender = new SoVolumeRender;
      volRender->subdivideTile = TRUE;
      volRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
      volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volRender->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
      volRender->lowScreenResolutionScale = 2;
      volSep->addChild( volRender );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orienation
    root->addChild( new Gnomon() );

    // Notes
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 );
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "3 volume masks with independent transfer functions." );
      root->addChild( text );

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Volume Mask");
    myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    myViewer->setDecoration(FALSE);
    myViewer->setSize( MedicalHelper::exampleWindowSize() );
    myViewer->viewAll();
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

void generateMasks(SoVolumeData* /*vd*/, const std::vector<SoVolumeMask*>& vmList, const std::vector<SoVolumeDataDrawStyle*>& vmStyleList)
{
    vmStyleList[0]->style = SoVolumeDataDrawStyle::ISOSURFACE;
    vmStyleList[1]->style = SoVolumeDataDrawStyle::VOLUME_RENDER;
    vmStyleList[2]->style = SoVolumeDataDrawStyle::VOLUME_RENDER;
    // Isosurface value for the first vertebra.
    vmStyleList[0]->isovalues.set1Value(0, 1470);

    SoMaterial* mat;
    mat = vmStyleList[0]->isosurfacesMaterial = new SoMaterial;
    mat->diffuseColor.set1Value(0, SbColor(1., 0., 0.));
    mat->transparency.set1Value(0, 0.3f);

    vmList[0]->fileName = "$OIVHOME/examples/source/Medical/Segmentation/medicalSimpleVolumeMask/L1.lda";
    vmList[1]->fileName = "$OIVHOME/examples/source/Medical/Segmentation/medicalSimpleVolumeMask/L2.lda";
    vmList[2]->fileName = "$OIVHOME/examples/source/Medical/Segmentation/medicalSimpleVolumeMask/L3.lda";
}

/*******************************************************************************/
//Add color to all tf entries
SoTransferFunction*
    generateTf(SbVec3f color)
{
    SoTransferFunction* tf = new SoTransferFunction;
    //tf->predefColorMap = SoTransferFunction::STANDARD;
    //tf->minValue = 178;
    tf->predefColorMap = SoTransferFunction::NONE;
    tf->minValue = 180;
    tf->maxValue = 255;

    SbVec4f* rgba = (SbVec4f*)tf->actualColorMap.startEditing();
    for ( size_t i = 0; i < 256; i++)
    {
        //float r = (color[0]+rgba[i][0])*0.5f;
        //float g = (color[1]+rgba[i][1])*0.5f;
        //float b = (color[2]+rgba[i][2])*0.5f;
        //rgba[i] = SbVec4f(r, g, b, rgba[i][3]);
      rgba[i] = SbVec4f(color[0],color[1],color[2], (float)i/255 );
    }
    tf->actualColorMap.finishEditing();
    return tf;
}


