///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : David Beilloin (Dec 2010)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtRenderArea.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/Xt/SoXtDirectionalLightEditor.h>

#include <Inventor/SoInput.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/draggers/SoTransformerDragger.h>
#include <Inventor/events/SoKeyboardEvent.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoUniformGridClipping.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <DialogViz/dialog/SoDialogPushButton.h>
#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoDialogCheckBox.h>
#include <DialogViz/dialog/SoDialogComboBox.h>
#include <DialogViz/dialog/SoDialogRealSlider.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>
#include <DialogViz/dialog/SoDialogCustom.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include "SoOffscreenVolumeRenderCustom.h"

///////////////////////////////////////////////////////////////////////////////

SoVolumeData* g_resultVolData = NULL;
SoNode* g_extractSelectionSep;
SoSceneManager* g_sceneMgr = NULL;
SoOffscreenVolumeRender::Components components = SoOffscreenVolumeRender::RGBA;
SoOffscreenVolumeRenderCustom* offVolRender = NULL;

/*********************************************************************************/
class InterfaceAuditor : public SoDialogAuditor
{
public:
    InterfaceAuditor(SoTopLevelDialog* topDialog){m_top = topDialog;}

private:
    SoTopLevelDialog* m_top;

    virtual void dialogPushButton(SoDialogPushButton* cpt)
    {
        if (cpt->auditorID.getValue() == "extract")
        {
            //Clear the resulting volume data
            SbVec3i32 size;          
            SbDataType dataType;
            unsigned int* data = (unsigned int*)g_resultVolData->data.startEditing(size, dataType);
            memset(data, 0, size[0]*size[1]*size[2]*dataType.getSize());
            g_resultVolData->data.finishEditing();

            // Adjust the volume data extent according to the extraction box
            SoGetBoundingBoxAction extractBboxAction = SoGetBoundingBoxAction(SbViewportRegion());
            extractBboxAction.apply(g_extractSelectionSep);
            SbXfBox3f xfbbox = extractBboxAction.getXfBoundingBox();    
            SbVec3f translation;
            SbRotation rotation;
            SbVec3f scaleFactor;
            SbRotation scaleOrientation;
            xfbbox.getTransform().getTransform(translation,rotation,scaleFactor,scaleOrientation);

            SbVec3f minE;
            SbVec3f maxE;
            xfbbox.getBounds(minE, maxE);
            minE = minE*scaleFactor;
            maxE = maxE*scaleFactor;
            g_resultVolData->extent = SbBox3f(minE, maxE);

#if 1
            // SYNCHRONOUS CASE
            offVolRender->synchronousExtraction(g_sceneMgr);
#else
            // ASYNCHRONOUS CASE
            offVolRender->trigger.setValue(TRUE);
#endif
        }
    }

    virtual void dialogCheckBox(SoDialogCheckBox* cpt)
    {
        if (cpt->auditorID.getValue() == "opacityOnly")
        {   
            SbBool opacityOnly = cpt->state.getValue();
            offVolRender->components = opacityOnly?SoOffscreenVolumeRender::ALPHA:SoOffscreenVolumeRender::RGBA;
            std::vector<unsigned int> data(offVolRender->boxSize.getValue()[0]*offVolRender->boxSize.getValue()[1]*offVolRender->boxSize.getValue()[2]);
            SbDataType dataType = opacityOnly ? SbDataType::UNSIGNED_BYTE:SbDataType::UNSIGNED_INT32;
            g_resultVolData->data.setValue( offVolRender->boxSize.getValue(), dataType, dataType.getNumBits(), &data[0], SoSFArray::COPY );
            g_resultVolData->dataRGBA = !opacityOnly;
            offVolRender->trigger.setValue(TRUE);
        }
    }

    void dialogComboBox(SoDialogComboBox* cpt)
    {
        int selectedItem = cpt->selectedItem.getValue();
        switch ( selectedItem )
        {
        case 0:
            offVolRender->dataSpaceMatrix = SbMatrix::identity();
            break;
        case 1:
            offVolRender->dataSpaceMatrix = SbMatrix(
                -1, 0, 0, 0,
                0, 1, 0, 0,
                0, 0, 1, 0,
                1, 0, 0, 1);
            break;
        case 2:
            offVolRender->dataSpaceMatrix = SbMatrix(
                1, 0, 0, 0,
                0,-1, 0, 0,
                0, 0, 1, 0,
                0, 1, 0, 1);
            break;
        case 3:
            offVolRender->dataSpaceMatrix = SbMatrix(
                1, 0, 0, 0,
                0, 1, 0, 0,
                0, 0, -1, 0,
                0, 0, 1, 1);
            break;
        }
    }

    virtual void dialogRealSlider(SoDialogRealSlider* cpt)
    {
        if (cpt->auditorID.getValue() == "opacityThreshold")
        {
            offVolRender->opacityThreshold.setValue(cpt->value.getValue());
        }
    }
};

///////////////////////////////////////////////////////////////////////////////
int
main(int argc, char **argv)
{
    Widget mainWindow = SoXt::init(argv[0]); // pass the app name
    SoDialogViz::init();
    SoVolumeRendering::init();
    InventorMedical::init();
    SoOffscreenVolumeRenderCustom::initClass();

    // read Volume render scene graph    
    SbString sceneToExtract = "$OIVHOME/examples/source/Medical/Segmentation/medicalVolumeExtract/simpleVolumeRender.iv";
    if ( argc == 2 )
        sceneToExtract = argv[1];
    SoRef<SoSeparator> volumeRenderSep = MedicalHelper::readFile(sceneToExtract.toLatin1());

    SoTopLevelDialog *myTopLevelDialog;
    Widget parent = MedicalHelper::buildInterface(mainWindow, "$OIVHOME/examples/source/Medical/Segmentation/medicalVolumeExtract/interface.iv", "Viewer", &myTopLevelDialog);
    InterfaceAuditor* myInterfaceAuditor = new InterfaceAuditor(myTopLevelDialog);
    myTopLevelDialog->addAuditor(myInterfaceAuditor);

    //
    // Create Viewer and scene graph in charge of Volume clipping rendering
    //
    SoSwitch* volClippingRoot=new SoSwitch();
    volClippingRoot->whichChild = -3;

    SoDirectionalLight* light = new SoDirectionalLight();
    volClippingRoot->addChild(light);

    SoPerspectiveCamera* cam = new SoPerspectiveCamera();
    volClippingRoot->addChild(cam);

    // cube selection
    SoSwitch* cubeSelectionSwitch = new SoSwitch();
    cubeSelectionSwitch->whichChild = -3;
    cubeSelectionSwitch->addChild( MedicalHelper::readFile("$OIVHOME/examples/source/Medical/Segmentation/medicalVolumeExtract/simpleCubeSelection.iv") );
    volClippingRoot->addChild(cubeSelectionSwitch);

    // fix default dragger position based on current VolumeData extent
  SoVolumeData* volData = MedicalHelper::find<SoVolumeData>(volumeRenderSep.ptr());
  SoTransformerDragger* dragTrans = MedicalHelper::find<SoTransformerDragger>(cubeSelectionSwitch);
    if ( volData && dragTrans )
    {
        SbBox3f extent = volData->extent.getValue();
        // translate to center
        dragTrans->translation.setValue((extent.getMax()+extent.getMin())/2.0f);
        // scale to same size
        dragTrans->scaleFactor.setValue((extent.getMax()-extent.getMin())/2.0f);
    }

    // volume Rendering
    SoSwitch* volumeRenderSwitch = new SoSwitch();
    volumeRenderSwitch->whichChild = SO_SWITCH_ALL;
    volumeRenderSwitch->addChild(volumeRenderSep.ptr());
    volClippingRoot->addChild(volumeRenderSwitch);

    // OIV Logo
    volClippingRoot->addChild( MedicalHelper::exampleLogoNode() );
 
    SoXtExaminerViewer* volClippingViewer = new SoXtExaminerViewer(parent);
    volClippingViewer->setSceneGraph(volClippingRoot);
    volClippingViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    volClippingViewer->setDecoration(FALSE);
    volClippingViewer->setSize( MedicalHelper::exampleWindowSize() );
    volClippingViewer->show();
    volClippingViewer->viewAll();

    // setup the scenemanager to be used by extraction
    g_sceneMgr = volClippingViewer->getSceneManager();

    // create the object in charge of OffscreenRendering
    offVolRender = new SoOffscreenVolumeRenderCustom();
    offVolRender->volumerenderSceneGraph.setValue(volumeRenderSep.ptr());
    g_extractSelectionSep = MedicalHelper::find<SoNode>(volClippingRoot,"extractSelectionSep");
    offVolRender->bboxSceneGraph.setValue(g_extractSelectionSep);
    offVolRender->boxSize.setValue(SbVec3i32(256,256,109));
    offVolRender->boxSubdivision.setValue(SbVec3i32(1,1,0));
    offVolRender->components = components;
    volClippingRoot->addChild(offVolRender);

    // read Volume render scene graph. Define how the extracted data will be rendered.
    SoRef<SoSeparator> resultRenderSep = MedicalHelper::readFile( "$OIVHOME/examples/source/Medical/Segmentation/medicalVolumeExtract/simpleResultRender.iv" );

    g_resultVolData = MedicalHelper::find<SoVolumeData>(resultRenderSep.ptr());
    offVolRender->setResultVolumeData(g_resultVolData);
    g_resultVolData->extent = volData->extent.getValue();
    std::vector<unsigned int> data(offVolRender->boxSize.getValue()[0]*offVolRender->boxSize.getValue()[1]*offVolRender->boxSize.getValue()[2]);
    SbDataType dataType = (components == SoOffscreenVolumeRender::RGBA) ? SbDataType::UNSIGNED_INT32 : SbDataType::UNSIGNED_BYTE;
    g_resultVolData->data.setValue( offVolRender->boxSize.getValue(), dataType, dataType.getNumBits(), &data[0], SoSFArray::COPY );
    g_resultVolData->dataRGBA = TRUE;

    resultRenderSep->addChild( MedicalHelper::exampleLogoNode() );

    SoXtExaminerViewer* resultViewer = new SoXtExaminerViewer(mainWindow,"Result rendering",FALSE);
    resultViewer->setTitle("Result of the extraction");
    resultViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    resultViewer->setSceneGraph(resultRenderSep.ptr());
    resultViewer->setDecoration(FALSE);
    resultViewer->setSize( MedicalHelper::exampleWindowSize() );
    resultViewer->show();

    // Main Inventor event loop
    SoXt::show(mainWindow);
    SoXt::mainLoop();

    // release objects
    delete resultViewer;
    delete volClippingViewer;
    delete myInterfaceAuditor;

    resultRenderSep = NULL;

    // release modules
    SoOffscreenVolumeRenderCustom::exitClass();
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();

    return (0);
}


