///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volume geometry.
A standard manipulator allows the geometry to be moved through the volume.
author : Jerome Hummel
August 2002

Updated by Pascal Estrade (Sep 2014)
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/manips/SoTransformerManip.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeTriangleStripSet.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

///////////////////////////////////////////////////////////////////////////////

// This geometry is taken from the Open Inventor Mentor example
// 05.3.TriangleStripSet.cxx, showing that the data for a
// VolumeTriangleStripSet is the same as for a TriangleStripSet.

// Positions of all of the vertices:
static const float 
    vertexPositions[40][3] =
{
    { 0.0f,12.0f, 0.0f }, { 0.0f,15.0f, 0.0f},
    { 2.1f,12.1f,-0.2f }, { 2.1f,14.6f,-0.2f},
    { 4.0f,12.5f,-0.7f }, { 4.0f,14.5f,-0.7f},
    { 4.5f,12.6f,-0.8f }, { 4.5f,14.4f,-0.8f},
    { 5.0f,12.7f,-1.0f }, { 5.0f,14.4f,-1.0f},
    { 4.5f,12.8f,-1.4f }, { 4.5f,14.6f,-1.4f},
    { 4.0f,12.9f,-1.6f }, { 4.0f,14.8f,-1.6f},
    { 3.3f,12.9f,-1.8f }, { 3.3f,14.9f,-1.8f},
    { 3.0f,13.0f,-2.0f }, { 3.0f,14.9f,-2.0f}, 
    { 3.3f,13.1f,-2.2f }, { 3.3f,15.0f,-2.2f},
    { 4.0f,13.2f,-2.5f }, { 4.0f,15.0f,-2.5f},
    { 6.0f,13.5f,-2.2f }, { 6.0f,14.8f,-2.2f},
    { 8.0f,13.4f,-2.0f }, { 8.0f,14.6f,-2.0f},
    {10.0f,13.7f,-1.8f }, {10.0f,14.4f,-1.8f},
    {12.0f,14.0f,-1.3f }, {12.0f,14.5f,-1.3f},
    {15.0f,14.9f,-1.2f }, {15.0f,15.0f,-1.2f},

    {-0.5f,15.0f, 0.0f }, {-0.5f, 0.0f, 0.0f},   // the flagpole
    { 0.0f,15.0f, 0.5f }, { 0.0f, 0.0f, 0.5f},
    { 0.0f,15.0f,-0.5f }, { 0.0f, 0.0f,-0.5f},
    {-0.5f,15.0f, 0.0f }, {-0.5f, 0.0f, 0.0f}
};

// Number of vertices in each strip.
static int32_t 
    numVertices[2] =
{
    32, // flag
    8   // pole
};

///////////////////////////////////////////////////////////////////////
//
// Create a wireframe box showing geometric bounds of volume

SoNode *makeBox( SbBox3f &volSize )
{
    // Just draw the wireframe of the box
    SoDrawStyle *pStyle = new SoDrawStyle;
    pStyle->style = SoDrawStyle::LINES;

    // The box will be easier to see without lighting
    SoLightModel *pLModel = new SoLightModel;
    pLModel->model = SoLightModel::BASE_COLOR;

    // The box should be unpickable so manip can be used inside it
    SoPickStyle *pPickable = new SoPickStyle;
    pPickable->style = SoPickStyle::UNPICKABLE;

    // Translate cube (nominally at 0,0,0) to geometric center of volume
    SoTranslation *pTranslate = new SoTranslation;
    pTranslate->translation = volSize.getCenter();

    // Create a cube with the geometric size of the volume
    SoCube *pCube = new SoCube;
    float xsize, ysize, zsize;
    volSize.getSize( xsize, ysize, zsize );
    pCube->width  = xsize;
    pCube->height = ysize;
    pCube->depth  = zsize;

    // Assemble scene graph
    SoSeparator *pBoxSep = new SoSeparator;
    pBoxSep->addChild( pStyle );
    pBoxSep->addChild( pLModel );
    pBoxSep->addChild( pPickable );
    pBoxSep->addChild( pTranslate );
    pBoxSep->addChild( pCube );
    return pBoxSep;
}

///////////////////////////////////////////////////////////////////////
//
// Main function

int main(int, char **argv)
{
    // Create the window
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow) return 0;

    // Initialize extensions
    SoVolumeRendering::init();
    InventorMedical::init();

    // Node to hold the volume data
    // In the first call we specify the voxel dimensions and the actual data.
    // In the second call we specify the 3D extent of the volume in modeling coords.
    // Note that the geometric size and position of the volume does not have to be
    // the same as the dimensions of the volume data.
    SoVolumeData* pVolData = new SoVolumeData();
    pVolData->fileName = "$OIVHOME/examples/data/Medical/files/spine.lda";
    SbBox3f volSize;
    volSize = pVolData->extent.getValue();
    pVolData->extent = volSize;

    SoDataRange *pDataRange = new SoDataRange();
    pDataRange->min = 1332;
    pDataRange->max = 1775;

    // Use a predefined colorMap with the SoTransferFunction
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->predefColorMap = SoTransferFunction::STANDARD;

    // Define coordinates and base color for volume geometry
    SoVertexProperty *myVertexProperty = new SoVertexProperty;
    myVertexProperty->orderedRGBA.set1Value(0, 0xFFFFFFFF);
    myVertexProperty->vertex.setValues(0, 40, vertexPositions);

    // Volume geometry node
    SoVolumeTriangleStripSet *pVolGeom = new SoVolumeTriangleStripSet;
    pVolGeom->numVertices.setValues(0, 2, numVertices);
    pVolGeom->vertexProperty.setValue( myVertexProperty );

    // This ShapeHints will enable two-sided lighting for the geometry
    SoShapeHints *pHints = new SoShapeHints;
    pHints->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;

    // Assemble the scene graph
    // Note: Volume geometry must appear after the SoVolumeData node.
    SoRef<SoSeparator> root = new SoSeparator;

    root->addChild( makeBox( volSize ) );     // Draws volume bounds
    root->addChild( pVolData );               // Volume data
    root->addChild( pDataRange );			  // Data Range
    root->addChild( pTransFunc );             // Transfer function
    root->addChild( pHints );                 // Shape hints

    SoTransformerManip *localManip = new SoTransformerManip();
    localManip->translation.setValue(SbVec3f(-106.482f, -112.217f, -113.069f));
    root->addChild( localManip ); // Manip to move geometry
    root->addChild( pVolGeom );               // Volume geometry
    
    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
    myViewer->setSceneGraph(root.ptr());
    myViewer->setTitle("Volume Geometry");
    myViewer->setDecoration(FALSE);
    myViewer->setSize( MedicalHelper::exampleWindowSize() );
    myViewer->show();

    SoXt::show(myWindow);
    SoXt::mainLoop();

    delete myViewer;

    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();

    return 0;
}


