///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Roberto Calabrese (Mar 2011)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

/*******************************************************************************
* This example shows how to use a VolumeClippingGroup to clip a VolumeData with a CircularExtrusion
* than can be selected by the user from a set.
* Some SoCircularExtrusions are created on the fly inside the volume of the chosen VolumeData .ldm file,
* the VolumeData is sketched in the scene by mean of its bounding box.
* The fields ( radius and activesection start and end points ) of the seelcted SoCircularExtrusion
* can be modified interactively by the user.
* The SoPipeShas owns one SoComplexity node as to see how its value impacts on the Pipe representation.
* The scene graph contains a VolumeClippingGroup node that is used to clip the VolumeData.
* When a CircularExtrusion is selected the selection callback assigns the selected CircularExtrusion as 
* child of this VolumeClippingGroup node.
*******************************************************************************/

#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Inventor/nodes/SoCone.h>

#include <Inventor/nodes/SoCircularExtrusion.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSelection.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRender.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILE    = "$OIVHOME/examples/data/Medical/files/spine.lda";
const SbString EXTRUSION_FILE = "$OIVHOME/examples/source/Medical/Segmentation/medicalVolumePipeClipping/pipescenegraph.iv";
const SbString INTERFACE_FILE = "$OIVHOME/examples/source/Medical/Segmentation/medicalVolumePipeClipping/interface.iv";

static SoRef<SoSeparator>   m_root;

static SoCircularExtrusion* m_spineClipping;
static SoComplexity*        m_complexity;

static void BuildSceneGraph();

static Widget BuildInterface(Widget window);

///////////////////////////////////////////////////////////////////////////////
int
main( int , char **argv )
{
    // Lower epsilon value to avoid "twisting" artifacts
    SoPreferences::setFloat("OIV_EXTRUSION_EPSILON", 0.999995f);

    Widget mainWindow = SoXt::init( argv[0] );
    SoVolumeRendering::init();
    InventorMedical::init();
    SoDialogViz::init();

    if (! SbFileHelper::isAccessible(VOLUME_FILE)) {
      new SoMessageDialog( VOLUME_FILE, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    if (! SbFileHelper::isAccessible(INTERFACE_FILE)) {
      new SoMessageDialog( INTERFACE_FILE, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    if (! SbFileHelper::isAccessible(EXTRUSION_FILE)) {
      new SoMessageDialog( EXTRUSION_FILE, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }

    // Build scene graph
    BuildSceneGraph();

    // Create user interface objects
    Widget parent = BuildInterface( mainWindow );

    // Viewer setup (Do not call setSize() here. Size is set in the UI file.)
    SoXtExaminerViewer* viewer = new SoXtExaminerViewer( parent );
    viewer->setTitle( "Volume clipping with CircularExtrusion" );
    viewer->setSceneGraph( m_root.ptr() );
    viewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );	
    viewer->setDecoration(FALSE);

    viewer->show();        
    viewer->viewAll();
    viewer->saveHomePosition();

    // Main Inventor event loop
    SoXt::show( mainWindow );
    SoXt::mainLoop();
    delete viewer;
    m_root = NULL;
    SoDialogViz::finish();
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Build scene graph
void BuildSceneGraph()
{
    // Root separator
    m_root = new SoSeparator();

    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
    camera->orientation = SbRotation( SbVec3f(1, 0, 0), (float)M_PI / 2)
                        * SbRotation( SbVec3f(0, 0, 1), (float)M_PI / 2);
    m_root->addChild(camera);

    // Volume Data
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILE;
      m_root->addChild(volData);

    SoDataRange* dataRange = new SoDataRange();
      double minval, maxval;
      volData->getMinMax( minval, maxval);
      // Value chosen to have a "nice" rendering allowing to see the extrusion path.
      dataRange->min = 1480;
      dataRange->max = maxval;
      m_root->addChild(dataRange);

    SoTransferFunction* transferFunction = new SoTransferFunction();
      transferFunction->predefColorMap = SoTransferFunction::STANDARD;
      transferFunction->minValue       = 1; // Entry 0 will be completely transparent
      m_root->addChild(transferFunction);

    // Extrusion path in 3D (profile)
    SoSeparator* spineProfile = new SoSeparator();
    m_root->addChild(spineProfile);

    SoMaterial* materialProfile = new SoMaterial();
    materialProfile->diffuseColor.setValue(0, 1, 0);
    spineProfile->addChild(materialProfile);

    SoShapeHints* shapeHint = new SoShapeHints();
    shapeHint->creaseAngle = 0.785f;
    spineProfile->addChild(shapeHint);

    // Load extrusion path from file.
    SoSeparator* extrusionSep = MedicalHelper::readFile(EXTRUSION_FILE.toLatin1());
    SoCircularExtrusion* profileExtrusion = MedicalHelper::find<SoCircularExtrusion>(extrusionSep);
    profileExtrusion->radius = 0.1f;
    spineProfile->addChild(extrusionSep);

    // Use the spine profile with a larger radius to clip the volume
    SoVolumeClippingGroup* clippingGroup = new SoVolumeClippingGroup();
      m_root->addChild(clippingGroup);

      m_complexity = new SoComplexity();
        m_complexity->value = 0.5f;
        clippingGroup->addChild(m_complexity);

      m_spineClipping = new SoCircularExtrusion();
        m_spineClipping = (SoCircularExtrusion*)profileExtrusion->copy();
        m_spineClipping->radius = 4.2f;
        m_spineClipping->activeSection.setValue(0.343f, 1);
        clippingGroup->addChild(m_spineClipping);

    // VolumeRendering
    SoSeparator* volSep = new SoSeparator();
      m_root->addChild(volSep);

    SoMaterial* volumeTransparency = new SoMaterial();
      volumeTransparency->transparency = 0.5f;
      volSep->addChild(volumeTransparency);

    SoVolumeRenderingQuality* renderingQuality = new SoVolumeRenderingQuality();
      renderingQuality->interpolateOnMove = TRUE;
      renderingQuality->preIntegrated     = TRUE;
      renderingQuality->deferredLighting  = TRUE;
      volSep->addChild(renderingQuality);

    SoVolumeRender* volumeRender = new SoVolumeRender();
      volumeRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volSep->addChild(volumeRender);

    // Show volume orientation
    m_root->addChild(new Gnomon());

    // OIV Logo
    m_root->addChild(MedicalHelper::exampleLogoNode());

    // Note
    TextBox* text = new TextBox();
      text->position.setValue(0, -0.98f, 0); // Normalized device coordinates
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine("Volume is clipped by a cylindrical extrusion along a path (shown in green).");
      m_root->addChild(text);
}


///////////////////////////////////////////////////////////////////////////////
/** pipe sliders auditor  */
class MyAuditor : public SoDialogAuditor
{
public:
    void dialogRealSlider( SoDialogRealSlider *cpt );
};

///////////////////////////////////////////////////////////////////////////////
// Sliders 
void MyAuditor::dialogRealSlider( SoDialogRealSlider *cpt )
{
  // Spine radius
  if (cpt->auditorID.getValue() == "CHANGE_RADIUS") {
    m_spineClipping->radius = cpt->value.getValue();
  }

  // Spine start
  else if (cpt->auditorID.getValue() == "CHANGE_START") {
    SbVec2f section = m_spineClipping->activeSection.getValue();
    section[0] = cpt->value.getValue();
    m_spineClipping->activeSection = section;
  }

  // Spine end
  else if (cpt->auditorID.getValue() == "CHANGE_END") {
    SbVec2f section = m_spineClipping->activeSection.getValue();
    section[1] = cpt->value.getValue();
    m_spineClipping->activeSection = section;
  }

  // Spine complexity
  else if (cpt->auditorID.getValue() == "CHANGE_COMPLEXITY") {
    m_complexity->value = cpt->value.getValue();
  }
}


////////////////////////////////////////////////////////////////////////
// Build user interface with embedded viewer

Widget
BuildInterface(Widget window)
{
  SoInput myInput;
  myInput.openFile( INTERFACE_FILE );
  SoGroup *myGroup = SoDB::readAll( &myInput );
  myInput.closeFile();

  SoTopLevelDialog* myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

  // Create and register auditor to handle user input
  MyAuditor* auditor = new MyAuditor;
  myTopLevelDialog->addAuditor( auditor );

  // Build dialog
  SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));
  myTopLevelDialog->buildDialog( window, customNode != NULL );
  myTopLevelDialog->show();

  return customNode ? customNode->getWidget() : window;
}
