///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (Jul 2009)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Medical example program.
Purpose : High Quality rendering for medical data.
Description : Main
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCube.h> 
#include <Inventor/nodes/SoFont.h> 
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoPerspectiveCamera.h> 
#include <Inventor/nodes/SoRotationXYZ.h> 
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoShadowGroup.h> 
#include <Inventor/nodes/SoText2.h> 
#include <Inventor/nodes/SoTranslation.h> 

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/TextBox.h>

#include "manageEvents.h"


///////////////////////////////////////////////////////////////////////////////
// Data Set
#define FILENAME "$OIVHOME/examples/data/Medical/files/medicalFoot.ldm"

// color map that can be edited in Avizo.
#define COLORMAPFILENAME "$OIVHOME/examples/data/Medical/resources/volrenGlow.am"

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
    // Create the window
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow) return 0;
    SoVolumeRendering::init();
    InventorMedical::init();

    // ROOT
    SoRef<SoSeparator> root = new SoSeparator;

    // Separator for data.
    SoSeparator *dataSep = new SoSeparator();

    // The camera for data.
    SoPerspectiveCamera *localCam = new SoPerspectiveCamera();

    // Increase the rendering quality when you stop to move the dataset.
    SoInteractiveComplexity* cplx = new SoInteractiveComplexity;
    cplx->setName("InteractiveComplexity");

    // All data under this group will be shadowed.
    SoShadowGroup *data = new SoShadowGroup();
    data->method.setValue(SoShadowGroup::VARIANCE_SHADOW_MAP);
    data->intensity.setValue(0.7f);
    data->smoothFactor.setValue(4);
    data->isActive = true;

    // Separator for the foot support.
    SoSeparator *suppportSep = new SoSeparator();
    SoTranslation *suppTrans = new SoTranslation();
    suppTrans->translation.setValue(10, 0.0, 0.0);

    // The support.
    SoCube *support = new SoCube();
    support->depth.setValue(50.0f);
    support->width.setValue(50.0f);
    support->height.setValue(0.05f);

    // Node to hold the volume data
    SoVolumeData* pVolData = new SoVolumeData();
    pVolData->texturePrecision = 16;  // Store actual values on GPU.
    pVolData->fileName = FILENAME;

    // Load the colorMap from an Avizo colormap.
    SoTransferFunction* pTransFunc = new SoTransferFunction;
    pTransFunc->loadColormap(COLORMAPFILENAME);

    // Set data default range.
    SoDataRange *localDataRange = new SoDataRange;
    localDataRange->min = 176;
    localDataRange->max = 476;

    // Property node which allows SoVolumeRender to draw High Quality volumes.  
    SoVolumeRenderingQuality *pVRVolQuality = new SoVolumeRenderingQuality;
    pVRVolQuality->preIntegrated = TRUE;
    pVRVolQuality->ambientOcclusion = TRUE;
    pVRVolQuality->deferredLighting = TRUE;

    // Node in charge of drawing the volume
    SoVolumeRender* pVolRender = new SoVolumeRender;
    pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
    pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;

    //Set the ambient, diffuse, transparency, specular and shininess of the material
    SoMaterial *material = new SoMaterial;
    material->ambientColor.setValue(.0f, .0f, .0f);
    material->diffuseColor.setValue(1, 1, 1);
    material->transparency.setValue(0.0f);
    material->specularColor.setValue(1, 1, 1);
    material->shininess.setValue(0.5);

    SoRotationXYZ *rotX = new SoRotationXYZ();
    rotX->axis = SoRotationXYZ::X;
    rotX->angle.setValue(-3.14f / 2);
    SoRotationXYZ *rotZ = new SoRotationXYZ();
    rotZ->axis = SoRotationXYZ::Z;
    rotZ->angle.setValue(-3.14f / 2);

    // Mouse event callback to manage data range.
    SoEventCallback *mouseMoveEvent = new SoEventCallback();
    SoEventCallback *mouseKeyEvent = new SoEventCallback();

    // Add Children
    root->addChild(dataSep);
    dataSep->addChild(localCam);

    // Change complexity when the dataset move.
    cplx->fieldSettings.set1Value(0, "SoComplexity value 0.3 0.7");
    cplx->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
    cplx->refinementDelay = 0;
    dataSep->addChild(cplx);
    dataSep->addChild(new SoComplexity());

    dataSep->addChild(material);
    dataSep->addChild( MedicalHelper::exampleLogoNode() );
    dataSep->addChild(data);

    data->addChild(suppportSep);
    suppportSep->addChild(suppTrans);
    suppportSep->addChild(support);

    data->addChild(rotX);
    data->addChild(rotZ);
    data->addChild(pVolData);
    data->addChild(localDataRange);
    data->addChild(pTransFunc);
    data->addChild(pVRVolQuality);
    data->addChild(pVolRender);
    data->addChild(mouseMoveEvent);
    data->addChild(mouseKeyEvent);

    // Instructions
    TextBox* textBox = new TextBox();
    textBox->position.setValue(0, 0.99f, 0);  // Normalized Device Coordinates (-1..1)
    textBox->alignmentV = TextBox::TOP;
    textBox->alignmentH = TextBox::CENTER;
    textBox->addLine("In selection mode :");
    textBox->addLine("  -> Drag horizontally with left mouse button pressed to change the min data range.");
    textBox->addLine("  -> Drag vertically with left mouse button pressed to change the max data range.");
    root->addChild(textBox);

    textBox = new TextBox();
    textBox->position.setValue(0, -0.99f, 0);  // Normalized Device Coordinates (-1..1)
    textBox->alignmentV = TextBox::BOTTOM;
    textBox->alignmentH = TextBox::CENTER;
    textBox->addLine("Volume rendering with shadow casting");
    root->addChild(textBox);

    // get mouse events to change the colormap ranges.
    mouseMoveEvent->addEventCallback(SoLocation2Event::getClassTypeId(), mouseMoveEventCB, localDataRange);
    mouseKeyEvent->addEventCallback(SoMouseButtonEvent::getClassTypeId(), mouseKeyEventCB, data);

    // Set up viewer:
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
    myViewer->setSize( MedicalHelper::exampleWindowSize() );
    myViewer->setSceneGraph(root.ptr());
    myViewer->setCamera(localCam);
    myViewer->setDecoration(FALSE);
    myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    myViewer->setTitle("Bones & Muscles.");
    myViewer->show();
    myViewer->viewAll();
    localCam->viewAll(data, myViewer->getViewportRegion());
    myViewer->saveHomePosition();
    myViewer->getSceneManager()->setAutoInteractiveMode(TRUE); // Improves perf changing data range

    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;

    root = NULL;

    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}
