///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/* 
 * Demonstrates converting a DICOM data set to LDM format.
 *
 * Open Inventor can load DICOM data files directly. But it can be useful to
 * convert the data to LDM format. LDM is Open Inventor's native file format
 * for multi-resolution tiled image and volume data.
 *
 * Open Inventor automatically manages image/volume data as tiles (blocks of
 * data). This allows your application to display and interact with volume
 * data even when there is not enough CPU or GPU memory to hold the complete
 * full resolution volume.  Other libraries may down-sample the data to make
 * it fit in memory, but then always display the down sampled version of the
 * data.  Open Inventor creates a multi-resolution hierarchy of tiles.  Full
 * resolution data is always displayed if there is sufficient memory.  But if
 * the available memory is too small, Open Inventor will always display the
 * highest resolution data possible and eventually will display the full
 * resolution data when the user zooms in on a portion of the volume.  (Tiles
 * outside the view volume do not need to be retained in memory.)  Open Inventor
 * can automatically switch to lower resolution rendering while the user is
 * interacting (for example moving the camera) in order to maintain interactive
 * frame rate.
 *
 * When DICOM (or other format) data is loaded directly, Open Inventor creates
 * tiles "on the fly".  One advantage of using the LDM file format is that all
 * the tiles (full resolution and lower resolution) have been pre-created and
 * can be loaded very quickly from the LDM data file.  Another advantage of using
 * the LDM file format is that, no matter how large the total data set is, a low
 * resolution image can be displayed almost immediately.  This gives the user of
 * the application a much better experience because the application is always
 * responsive instead of displaying a "loading" message. Another advantage of using
 * the LDM file format is that the LDM header (an XML format file) stores useful
 * information about the data set, for example the min/max values and the histogram,
 * that is time consuming to recompute from the raw data.
 *
 * This program is similar to the LDM converter examples in the standard Open
 * Inventor SDK but customized for the DICOM file format.  Differences include:
 *
 *   - A DICOM file reader (SoVRDicomFileReader) is explicitly created.
 *     This means that the the DICOM file or list file does not need to have a
 *     recognized DICOM file extension (DICOM format is assumed).
 *
 * Notes:
 *
 *   1. "List" file is required.
 *      All the DICOM files that should be loaded as part of the volume must be
 *      named in a list file.  The list file should contain one path name per
 *      line. (This is not necessary in the non-typical case where a single DICOM
 *      file contains all the images in the stack.)

Usage:
 
  Parameters:
    - source volume filename
    - tile dimension in voxels
      Must be a power of 2. Default is 64 meaning each tile will be 64 x 64 x 64 voxels.
      This is quite small and we recommend using a larger tile size up to 512.
    - destination filename. Default is the source filename with extension .ldm.
    - amount of main memory to not exceed. Default is 128 Mb.
    - option to indicate to output the xml header only.

syntax : DICOM2LDM mydcmlist.lst -t <tilesize> -c <compressiontype> -l <compressionlevel> -s <averagesize> -o <outputldmfilename>

Example: DICOM2LDM.exe %OIVHOME%\examples\source\Medical\data\dicomSample\listOfDicomFiles.dcm -s 8 -c gzip -l 5 -m 4096 -t 128 -o testLDM_128_gzip.ldm

Execute this program without any parameters to show the full help text.
Also see the documentation page for class SoConverter for more explanation.
 *
 */

#include <LDM/converters/SoConverterParameters.h>
#include <VolumeViz/converters/SoVolumeConverter.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <Inventor/helpers/SbFileHelper.h>

////////////////////////////////////////////////////////////////////////
class DicomConverter : public SoVolumeConverter
{
public:
	/** Constructor */
	DicomConverter() : m_theDicomReader(NULL)
	{
	}

	/** Destructor */
	virtual ~DicomConverter()
	{
		if (m_theDicomReader != NULL)
		{
			m_theDicomReader->unref();
		};
	}

	/** Set the specific reader to use.
     *  Calling this method with null clears the previously specified reader.
     */
	void setReader(SoVRDicomFileReader* aDicomReader)
	{
		if (m_theDicomReader != NULL)
			m_theDicomReader->unref();

		m_theDicomReader = aDicomReader;

		if (m_theDicomReader != NULL)
			m_theDicomReader->ref();
	};

protected:
	// redefined from SoVolumeConverter
	virtual SoVolumeReader* getReader(const SbString& /*filename*/, const SbString& /*fileExt*/)
	{
		return m_theDicomReader;
	};

	// Progress callback
	SoVolumeConverter::Abort
		progress(int /*numTilesGenerated*/, int /*numTilesToGenerate*/)
	{
		return CVT_CONTINUE;
	};

private:
	// hold the reader to use for this converter
	SoVRDicomFileReader* m_theDicomReader;
};

///////////////////////////////////////////////////////////////////////////////
int
main(int argc, char ** argv)
{
	// Initialize VolumeViz (core Open Inventor is automatically initialized)
	SoVolumeRendering::init();

	// Create the DICOM reader
	SoVRDicomFileReader *theDicomReader = new SoVRDicomFileReader();

	// Create converter parameters from the input command line
	// All parameters are optional
	// Example : DICOM2LDM mydcmlist.lst -t <tilesize> -c <compressiontype> -l <compressionlevel> -s <averagesize> -o <outputldmfilename>
	SoConverterParameters* myConverterParameters = SoConverterParameters::create(argc, argv);

	DicomConverter myDicomConverter;
	myDicomConverter.setReader(theDicomReader);
	int ret = myDicomConverter.convert(myConverterParameters);
	delete myConverterParameters;

	SoVolumeRendering::finish();
	return ret;
}
