///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Nov 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to create a Magic Glass that display enhanced 
information.
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/nodes/SoRenderToTextureProperty.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/draggers/SoTranslate2Dragger.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

#include <iostream>

///////////////////////////////////////////////////////////////////////////////

// Data Set
#define FILENAME "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles.dcm"
const SbString IMAGE_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    // Create the window, initialize OpenIventor and VolumeViz.
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();	
    InventorMedical::init();

    if (! SbFileHelper::isAccessible(IMAGE_FILENAME)) {
      std::cerr << "Unable to open: " << (char*)IMAGE_FILENAME.toLatin1() << std::endl;
      return -1;
    }

    // Create the scene graph.
    SoRef<SoSeparator> root = new SoSeparator();

    // Camera
    SoOrthographicCamera* camera = new SoOrthographicCamera();
      root->addChild( camera );

    //===================================================================================================================
    // Scene Graph for Ortho Slice.
    SoSeparator* volSep = new SoSeparator();
      volSep->setName( "Volume" );
      root->addChild( volSep );

    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    SoMaterial* volMatl = new SoMaterial();
      volMatl->diffuseColor.setValue( 1, 1, 1 );
      volSep->addChild( volMatl );

    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volSep->addChild( volRange );

    SoTransferFunction* volTF = new SoTransferFunction();
      volTF->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( volTF, volData );
      volSep->addChild( volTF );

    SoOrthoSlice* slice = new SoOrthoSlice();
      slice->axis = MedicalHelper::AXIAL;
      slice->sliceNumber = volData->data.getSize()[2] / 2;
      slice->interpolation = SoSlice::MULTISAMPLE_12;
      volSep->addChild( slice );

    //===================================================================================================================
    // Scene graph for the Magic Glass rendering.
    //===================================================================================================================
    // Here we customize a dragger to create a magic glass rendering effect.
    SoTranslate2Dragger *magicGlassDragger = new SoTranslate2Dragger();
    SoSeparator *magicGlassSep = new SoSeparator();

    SoMaterial *magicGlassCubeMaterial = new SoMaterial();
      magicGlassCubeMaterial->diffuseColor.setValue(SbColor( 0, 0.6f,0));
      magicGlassSep->addChild( magicGlassCubeMaterial );

    // Border of the magicGlass.
    SoCube *magicGlassBorder = new SoCube();
      magicGlassBorder->width  = 41;
      magicGlassBorder->height = 41;
      magicGlassBorder->depth  = 1.9f;
      magicGlassSep->addChild(magicGlassBorder);

    // The texture that will be mapped on the SoCube face.
    SoTexture2 *magicGlassTexture = new SoTexture2();
      magicGlassTexture->model = SoTexture2::REPLACE;
      magicGlassSep->addChild(magicGlassTexture);

    // We know the default view is Axial so we need to look at the other side of the cube.
    SoTransform* magicGlassRotation = new SoTransform();
      magicGlassRotation->rotation.connectFrom( &camera->orientation );
      magicGlassSep->addChild( magicGlassRotation );

    // The center of this cube (square) is the center of the "magicGlassCam" camera.
    SoCube *magicGlassSupport = new SoCube();
      magicGlassSupport->width = 40;
      magicGlassSupport->height = 40;
      magicGlassSep->addChild( magicGlassSupport );

    // Here we replace parts to customize the dragger.
    magicGlassDragger->setPart("translatorActive", magicGlassSep);
    magicGlassDragger->setPart("translator", magicGlassSep);
    magicGlassDragger->setPart("xAxisFeedback", new SoSeparator() );
    magicGlassDragger->setPart("yAxisFeedback", new SoSeparator() );
    root->addChild(magicGlassDragger);

    // Scenegraph used by magicGlassRenderToTexture to create a new texture for each new position of the dragger (magicGlassDragger).
    // Then, the texture is mapped on "magicGlassSupport" shape.
    SoRef<SoSeparator> magicGlassSGSep = new SoSeparator();
      magicGlassSGSep->addChild(new SoDirectionalLight());

    // For an Axial view the main camera will be rotated - connect to get the same rotation.
    SoOrthographicCamera *magicGlassCam = new SoOrthographicCamera();
      magicGlassCam->position.connectFrom( &magicGlassDragger->translation );
      magicGlassCam->orientation.connectFrom( &camera->orientation );
      magicGlassCam->viewAll( magicGlassSGSep.ptr(), SbViewportRegion(100, 100) );
      magicGlassCam->height = 40;
      magicGlassCam->farDistance = 2000;
      magicGlassSGSep->addChild(magicGlassCam);

    magicGlassSGSep->addChild( volData );
    magicGlassSGSep->addChild( volMatl );
    magicGlassSGSep->addChild( volRange );

    // TransferFunction used by the magic Glass rendering.
    SoTransferFunction *magicGlassTransFunc = new SoTransferFunction();
      magicGlassTransFunc->predefColorMap = SoTransferFunction::PHYSICS;
      magicGlassTransFunc->minValue = 121;
      magicGlassTransFunc->maxValue = 255;
      magicGlassSGSep->addChild( magicGlassTransFunc );

    // Ortho slice rendering & manipulation
    SoOrthoSlice *magicGlassOrthoSlice = new SoOrthoSlice();
      magicGlassOrthoSlice->interpolation = SoSlice::MULTISAMPLE_12;
      magicGlassOrthoSlice->sliceNumber.connectFrom( &slice->sliceNumber ); // connect from main slice node
      magicGlassSGSep->addChild( magicGlassOrthoSlice );

    SoRenderToTextureProperty *magicGlassRenderToTexture = new SoRenderToTextureProperty();
      magicGlassRenderToTexture->node = magicGlassSGSep.ptr();
      magicGlassRenderToTexture->size.setValue(512,512);
      magicGlassTexture->renderToTextureProperty = magicGlassRenderToTexture;

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Add some typical image viewer annotation
    root->addChild( MedicalHelper::buildSliceAnnotation( camera, slice, &IMAGE_FILENAME) );

    // Add some instructions at the bottom of the window.
    SoMaterial* textMat = new SoMaterial();
      textMat->diffuseColor.setValue(0.5f, 1, 0.5f);
      root->addChild(textMat);
    TextBox* text = new TextBox();
      text->alignmentV = TextBox::BOTTOM;
      text->position.setValue( -0.98f, -0.85f, 0); // Normalized device coords -1..1
      text->addLine( "Click and drag to move Magic Glass." );
      root->addChild( text );

    //===================================================================================================================
    // Set up viewer.
    SoXtPlaneViewer* viewer = new SoXtPlaneViewer(myWindow);
      viewer->setTitle("Magic Glass Demonstration");
      viewer->setDecoration(FALSE);
      viewer->setViewing(FALSE);
      viewer->setSize( MedicalHelper::exampleWindowSize() );
      viewer->setSceneGraph(root.ptr());

    // Adjust camera
    //viewer->viewAll();
    //mainCam->height = 175;
    MedicalHelper::orientView( MedicalHelper::AXIAL, camera, volData );
    viewer->saveHomePosition();
    magicGlassDragger->translation.setValue( 0, 0, volData->extent.getValue().getMin()[2] );

    // Run then cleanup
    viewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    root = NULL;
    magicGlassSGSep = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

