///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Nov 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to create a magnifier.
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/dialog/SoMessageDialog.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/Magnifier.h>

///////////////////////////////////////////////////////////////////////////////

// Data Set
//const SbString IMAGE_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles512.dcm";  //CVH256.dcm";
const SbString IMAGE_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    // Initialize
    Widget myWindow = SoXt::init(argv[0]);
    SoDialogViz::init();

    if (! SbFileHelper::isAccessible(IMAGE_FILENAME)) {
      new SoMessageDialog( IMAGE_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }
    SoVolumeRendering::init();	
    InventorMedical::init();

    // Create the scene graph.
    SoRef<SoSeparator> root = new SoSeparator();

    // Camera
    SoOrthographicCamera* camera = new SoOrthographicCamera();
    //mainCam->height = 175;
      root->addChild ( camera );

    //===================================================================================================================
    // Scene shared between OrthoSlice & Magnifier.
    SoSeparator* volSep = new SoSeparator();
      volSep->setName( "Volume" );
      root->addChild( volSep );

    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    SoMaterial* volMatl = new SoMaterial();
      volMatl->diffuseColor.setValue( 1, 1, 1 );
      volSep->addChild( volMatl );

    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volSep->addChild( volRange );

    SoTransferFunction* volTF = new SoTransferFunction();
      volTF->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( volTF, volData );
      volSep->addChild( volTF );

    SoOrthoSlice* slice = new SoOrthoSlice();
      slice->axis = MedicalHelper::AXIAL;
      slice->sliceNumber = volData->data.getSize()[2] / 2;
      slice->interpolation = SoSlice::MULTISAMPLE_12;
      volSep->addChild( slice );

    // Magnifier
    Magnifier* myMagnifier = new Magnifier();
      // Magnifier initial position is based on volume extent.
      // Move it to the front and offset left so it's easier to see.
      const SbBox3f& volExt = volData->extent.getValue();
      SbVec3f magPos = volExt.getCenter();
      magPos[0] = magPos[0] - (volExt.getSize()[0] / 4); // Offset left
      magPos[2] = volExt.getMax()[2] + 1;                    // In front of the volume
      myMagnifier->translation = magPos;
      myMagnifier->sceneToMagnify = volSep; // Magnifier has its own camera
      myMagnifier->magnifierColor.setValue(SbColor(0, 0.6f,0));
      root->addChild(myMagnifier);

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Annotation
    root->addChild( MedicalHelper::buildSliceAnnotation( camera, slice, &IMAGE_FILENAME ) );

    // Give the user a hint about what to do...
    SoMaterial* textMat = new SoMaterial();
      textMat->diffuseColor.setValue(0.5f, 1, 0.5f);
      root->addChild(textMat);
    TextBox* hints = new TextBox();
      hints->alignmentV = TextBox::BOTTOM;
      hints->position.setValue(-0.99f, -0.85f, 0); // Normalized device coords -1..1
      hints->addLine("Click and drag to move magnifier");
      root->addChild(hints);

    //===================================================================================================================
    // Set up viewer.
    SoXtPlaneViewer* viewer = new SoXtPlaneViewer(myWindow);
    viewer->setTitle("Magnifier Demonstration");
    viewer->setDecoration(FALSE);
    viewer->setViewing(FALSE);
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    viewer->setSceneGraph(root.ptr());

    // Adjust camera.
    // Note: The Magnifier node is very limited (for now) and really only works
    //       with default camera orientation (looking toward -Z).
    //       So the image orientation is not really correct in this demo. :-(
    //MedicalHelper::orientView( MedicalHelper::AXIAL, camera, volData );
    viewer->viewAll();
    camera->height = volExt.getSize()[1];
    viewer->saveHomePosition();

    // Run then cleanup
    viewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}

