///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (MMM 2002)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

/*----------------------------------------------------------------------------------------
Purpose : Demonstrate how to create and use a SoROIManip node.

Note: The tab color changes depending on whether we are manipulating the
ROI box or the subvolume box.  This color change is implemented in
the example program, not in the SoROIManip node.
----------------------------------------------------------------------------------------*/

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/draggers/SoTabBoxDragger.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <LDM/manips/SoROIManip.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/files/head.ldm";

Widget buildInterface(Widget);

SoRef<SoSeparator> m_root;
SoROI       * m_ROI;
SoROIManip  * m_ROIManip;
SoSwitch    * m_renderSwitch;

///////////////////////////////////////////////////////////////////////
//
// This callback is called at the end of a user interaction with the
// viewer (basically when the mouse button is released).

void
    viewerFinishCB( void* /*userData*/, SoXtViewer* /*viewer*/ )
{
    // When using a TabBoxDragger, the scale tab size is computed based on
    // screen size for the current orientation.  If the orientation has
    // changed because of user interaction, then generally the scale tab
    // size will need to be recomputed.

    ((SoTabBoxDragger*)m_ROIManip->getDragger())->adjustScaleTabSize();
}

///////////////////////////////////////////////////////////////////////

int
    main(int , char**)
{
    Widget myWindow = SoXt::init("");
    SoVolumeRendering::init();
    SoDialogViz::init();
    InventorMedical::init();

    // Build Scene Graph
    // Top separator
    m_root = new SoSeparator;

    // Use orthographic camera by default.
    m_root->addChild( new SoOrthographicCamera );

    // Volume rendering stuff
    SoSeparator* volSep = new SoSeparator();
      m_root->addChild( volSep );

    // Volume Data
    SoVolumeData *volData = new SoVolumeData();
      volData->fileName = VOLUME_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    // Get the voxel dimensions of the volume
    SbVec3i32 dimensions = volData->data.getSize();

    // Data range
    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volSep->addChild( volRange );

    // Transfer function
    SoTransferFunction* transferFunction = new SoTransferFunction();
      transferFunction->predefColorMap = SoTransferFunction::STANDARD;
      transferFunction->minValue = 69;
      volSep->addChild( transferFunction );

    SoVolumeRenderingQuality* volumeQuality = new SoVolumeRenderingQuality();
      volumeQuality->ambientOcclusion = TRUE;
      volumeQuality->deferredLighting = TRUE;
      volSep->addChild( volumeQuality );

    // ROI / ROIManip
    // Initialize both ROI box and subvolume to be the entire volume.
    // Constrain the ROIManip to stay inside the volume.
    m_ROI = new SoROI();
    m_ROIManip = new SoROIManip();

    SbVec3i32 volMin = SbVec3i32(0,0,0);
    SbVec3i32 volMax = dimensions-SbVec3i32(1,1,1);

    SbVec3i32 subvolMin = volMin;
    SbVec3i32 subvolMax = volMax;
    subvolMin = (volMax-volMin)/4;
    subvolMax = (volMax-volMin)*3/4;
    volMax[2] = (volMax[2]-volMin[2])/2;

    m_ROIManip->box.setValue( subvolMin, subvolMax );
    //m_ROIManip->subVolume.setValue( volMin,volMax );
    m_ROIManip->constrained = TRUE;
    volSep->addChild( m_ROIManip );

    // PickStyle
    // So that VolumeRender is not pickable -- remember that the volume
    // is pickable even where the voxels are completely transparent.
    // If we don't do this the manip will not be useable in some cases
    // where the user should expect it to be useable.
    SoPickStyle *pickStyle = new SoPickStyle;
    pickStyle->style = SoPickStyle::UNPICKABLE;
    volSep->addChild( pickStyle );

    m_renderSwitch = new SoSwitch;
    m_renderSwitch->whichChild =0;
    volSep->addChild( m_renderSwitch );

    // VolumeRender
    SoVolumeRender *m_volRender = new SoVolumeRender();
      m_volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      m_volRender->fixedNumSlicesInRoi.setValue(TRUE);
      m_renderSwitch->addChild( m_volRender );

    // VolumeSkin
    SoVolumeSkin *m_volSkin = new SoVolumeSkin();
      m_renderSwitch->addChild( m_volSkin );

    // OIV Logo
    m_root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation
    m_root->addChild( new Gnomon() );

    // Actual bounds of volume
    m_root->addChild( MedicalHelper::createBoundingBox( volData->extent.getValue() ));

    // Instructions
    TextBox* text = new TextBox();
      text->position.setValue( 0, -0.98f, 0 );
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine( "Click and drag to move/resize region of interest" );
      m_root->addChild( text );

    // Set up viewer and dialog
    // Note: Do NOT call setSize(). Viewer is determined by the UI file.
    Widget parent = buildInterface(myWindow);

    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(parent);
      myViewer->addFinishCallback( viewerFinishCB, NULL );
      myViewer->setSceneGraph( m_root.ptr() );
      myViewer->setTitle( "ROI Manipulator" );
      myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
      myViewer->setDecoration( FALSE );
      //myViewer->setSize( MedicalHelper::exampleWindowSize() );
      myViewer->setViewing( FALSE );

    // Adjust camera
    myViewer->getCamera()->orientation.setValue( SbVec3f(-0.85f, -0.34f, -0.4f), 4.729f); // More interesting view
    myViewer->viewAll();
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show( myWindow );
    SoXt::mainLoop();
    delete myViewer;
    m_root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}


/****************************************************************************
**
** Dialog section
**
****************************************************************************/

class myAuditorClass : public SoDialogAuditor
{
public:
    myAuditorClass();

private:
    void dialogCheckBox(SoDialogCheckBox* cpt);
    void dialogComboBox(SoDialogComboBox* cpt);
    SbBool m_ROIManipNode;
};

myAuditorClass::myAuditorClass()
{
    m_ROIManipNode = TRUE;
}

void
    myAuditorClass::dialogCheckBox(SoDialogCheckBox* cpt)
{
    SbBool state = cpt->state.getValue();

    SoROI *roi = m_ROIManipNode ? m_ROIManip : m_ROI;

    if (cpt->auditorID.getValue() == "roiNodes") {
        // Search for the current SoROI / SoROIManip and replace it
        roi->ref(); // In order it is not deleted when replaced
        SoSearchAction searchAction;
        searchAction.setNode( roi );
        searchAction.apply( m_root.ptr() );
        SoPath *path = searchAction.getPath();
        if ( !m_ROIManipNode ) {
            m_ROIManip->replaceNode( path );
            ((SoTabBoxDragger*)m_ROIManip->getDragger())->adjustScaleTabSize();
        }
        else
            m_ROIManip->replaceManip( path, m_ROI );
        m_ROIManipNode = !m_ROIManipNode;
        roi->unrefNoDelete(); // Balance the ref() above
    }
    if (cpt->auditorID.getValue() == "renderNodes") {
        if ( state)
            m_renderSwitch->whichChild =0;
        else
            m_renderSwitch->whichChild =1;
    }
    else if (cpt->auditorID.getValue() == "constrained")
        m_ROIManip->constrained = state;

    else if (cpt->auditorID.getValue() == "controlling") {
        m_ROIManip->boxOn = state;
        // Different color according to which box is dragged.
        SoMaterial *material = (SoMaterial*)(m_ROIManip->getDragger()->getPart("tabPlane1.scaleTabMaterial",0));
        if ( state ) {
            material->diffuseColor.setValue(0,1,0);
            material->emissiveColor.setValue(0,1,0);
        } else {
            material->diffuseColor.setValue(1,0,0);
            material->emissiveColor.setValue(1,0,0);
        }
    }

    else if (cpt->auditorID.getValue() == "relative")
        roi->relative = state;

}

void myAuditorClass::dialogComboBox(SoDialogComboBox* cpt)
{
    int selectedItem = cpt->selectedItem.getValue();

    SoROI *roi = m_ROIManipNode ? m_ROIManip : m_ROI;

    int flags[] = { SoROI::SUB_VOLUME, SoROI::EXCLUSION_BOX, SoROI::CROSS, SoROI::CROSS_INVERT, SoROI::FENCE, SoROI::FENCE_INVERT };
    roi->flags = flags[selectedItem];
}

Widget
    buildInterface(Widget window)
{
    SoInput myInput;
    if (! myInput.openFile( "$OIVHOME/examples/source/Medical/Tools/medicalRoiManip/roiManipDialog.iv" ))
        return NULL;
    SoGroup *myGroup = SoDB::readAll( &myInput );
    if (! myGroup)
        return NULL;
    SoTopLevelDialog *myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild( 0 );

    myAuditorClass *myAuditor = new myAuditorClass;
    myTopLevelDialog->addAuditor(myAuditor);

    SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));

    myTopLevelDialog->buildDialog( window, customNode != NULL );
    myTopLevelDialog->show();

    return customNode ? customNode->getWidget() : window;
}


