///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------------------
Purpose : Demonstrate how to rotate a "region of interest".
----------------------------------------------------------------------------------------*/


// Usual header file for volume rendering
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>

#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoInteractiveComplexity.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTransform.h>

#include <Inventor/draggers/SoTransformerDragger.h>

#include <VolumeViz/nodes/SoVolumeClippingGroup.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoTransferFunction.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

#include <iostream>

////////////////////////////////////////////////////////////////////////
// Global
#define VOL_FILE "$OIVHOME/examples/data/Medical/files/3DHEAD.ldm"

SoDragger* m_dragger = NULL;
SbBox3f    m_roiBox;

////////////////////////////////////////////////////////////////////////
// Forward decl

Widget buildInterface(Widget);

////////////////////////////////////////////////////////////////////////

int
main(int /*argc*/, char **argv)
{
    std::cout << "Click       a 'green ball' to rotate the ROI.\n";
    std::cout << "Shift-click a 'white cube' to scale along 1 axis.\n";
    std::cout << "Shift-click a face of the ROI to drag along 1 axis.\n";
    std::cout << "\n";

    // Enable dynamic highlighting of the SoTransformerDragger
    SoPreferences::setValue("IV_ALLOW_LOCATE_HIGHLIGHT", "1");

    // Initialize Inventor
    Widget myWindow = SoXt::init(argv[0]);
    SoVolumeRendering::init();
    SoDialogViz::init();
    InventorMedical::init();

    // Root of scene graph
    SoRef<SoSeparator> root = new SoSeparator;

    // Create our own camera so we set a better initial view of the volume.
    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
    camera->orientation.setValue(SbVec3f(-0.5f, 0.8f, 0.2f), 0.75f);
    root->addChild(camera);

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    SoSeparator* scene = new SoSeparator();
    root->addChild(scene);

    //--------------------------------------------------------------------
    // Volume rendering
    {
        // Keep volume viz separate from geometry
        SoSeparator* volSep = new SoSeparator();
        scene->addChild(volSep);

        // Decrease the quality while moving to have better interactivity
        SoInteractiveComplexity* interact = new SoInteractiveComplexity();
        // Decrease the "number of samples"
        interact->fieldSettings.set1Value(0, "SoComplexity value 0.25 0.5");
        // Decrease interpolation quality. 
        interact->fieldSettings.set1Value(1, "SoVolumeRender interpolation LINEAR CUBIC");
        // Don't wait before returning to full quality rendering. 
        interact->refinementDelay = 0;
        //      volSep->addChild( interact );

        // Complexity node for the interact node to control.
        SoComplexity* volComp = new SoComplexity();
        volSep->addChild(volComp);

        // Load volume data
        SoVolumeData* volData = new SoVolumeData();
        volData->setName("VolData");
        volData->fileName = VOL_FILE;
        volSep->addChild(volData);

        // Choose the initial ROI.
        // For demo purposes we'll define a "slab" in the center of the volume.
        SbVec3i32 volDim = volData->data.getSize();
        SbVec3i32 volMin = SbVec3i32(0, 0, 0);
        SbVec3i32 volMax = volDim - SbVec3i32(1, 1, 1);
        SbBox3f   volExt = volData->extent.getValue();  // Don't need this, but useful for debugging

        int roiWidth = volDim[0] / 5; // Slab width
        int offset = (volDim[0] - roiWidth) / 2;
        SbVec3i32 roiMin = volMin + SbVec3i32(offset, 0, 0);
        SbVec3i32 roiMax = volMax - SbVec3i32(offset, 0, 0);
        SbBox3i32 roi = SbBox3i32(roiMin, roiMax);

        // Convert ROI in IJK voxel coords to box in XYZ 3D coordinates.
        // Note the conversion method requires voxel coordinates as float, not int.
        SbVec3f roiMinF = SbVec3f((float)roiMin[0], (float)roiMin[1], (float)roiMin[2]);
        SbVec3f roiMaxF = SbVec3f((float)roiMax[0], (float)roiMax[1], (float)roiMax[2]);
        SbBox3f roiF = SbBox3f(roiMinF, roiMaxF);
        SbBox3f box = volData->voxelToXYZ(roiF);

        // Compute scale and translate values to make an XYZ box fit the ROI.
        // SoCube and dragger are centered at 0,0,0.
        // So we just need to translate to center of ROI.
        // SoCube and dragger extent is -1..1.
        // So we scale by the box size divided by 2.
        SbVec3f center = box.getCenter();
        SbVec3f scale = box.getSize() / 2;

        std::cout << "Vol dim: " << volDim << std::endl;
        std::cout << "    ext: " << volExt.getMin() << " : " << volExt.getMax() << std::endl;
        std::cout << "ROI ijk: " << roiMin << " : " << roiMax << std::endl;
        std::cout << "    xyz: " << box.getMin() << " - " << box.getMax() << std::endl;
        std::cout << " center: " << center << "\n scale: " << scale << std::endl;

        // Group for rotated ROI.
        // Note this must be a group so the clipping group can affect subsequent volume rendering node.
        SoGroup* rotatedRoiGroup = new SoGroup();
        rotatedRoiGroup->setName("Rotated_ROI");
        volSep->addChild(rotatedRoiGroup);

        // Group for ROI dragger (cannot be part of the volume clipping group)
        SoSeparator* manipGroup = new SoSeparator();
        manipGroup->setName("ROI_Dragger");
        rotatedRoiGroup->addChild(manipGroup);

        SoTransformerDragger* dragger = new SoTransformerDragger();
        dragger->translation = center;
        dragger->scaleFactor = scale;
        manipGroup->addChild(dragger);

        // Group for clipping geometry.
        SoVolumeClippingGroup* clipGroup = new SoVolumeClippingGroup();
        clipGroup->setName("ROI_Geometry");
        rotatedRoiGroup->addChild(clipGroup);

        // Controls size, position and rotation of clipping geometry.
        // Fields will be automatically updated when the user moves the dragger.
        SoTransform* clipTrans = new SoTransform();
        clipTrans->scaleFactor.connectFrom(&(dragger->scaleFactor));
        clipTrans->rotation.connectFrom(&(dragger->rotation));
        clipTrans->translation.connectFrom(&(dragger->translation));
        clipGroup->addChild(clipTrans);

        // Clipping geometry.
        SoCube* clipBox = new SoCube();
        clipGroup->addChild(clipBox);

        // Set range of data values to visualize.
        // Not required for 8-bit voxels, critical for larger data types.
        // The getMinMax() call may be expensive for non-LDM file formats.
        SoDataRange* volRange = new SoDataRange();
        if (volData->getDatumSize() > 1) {
            double minVal, maxVal;
            volData->getMinMax(minVal, maxVal);
            volRange->min = minVal;
            volRange->max = maxVal;
        }
        volSep->addChild(volRange);

        // Color map
        SoTransferFunction* volTF = new SoTransferFunction();
        volTF->predefColorMap = SoTransferFunction::STANDARD;
        volTF->minValue = 69; // Make low intensity voxels transparent
        volSep->addChild(volTF);

        // Display volume at full intensity
        SoMaterial* volMat = new SoMaterial();
        //volMat->diffuseColor.setValue( 1, 1, 1 );
        //volMat->transparency = 0.95f;
        volSep->addChild(volMat);

        // Volume rendering settings
        SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
        // Remove tile boundary artifacts while moving. 
        volQual->interpolateOnMove = TRUE;
        // Higher quality rendering
        volQual->preIntegrated = TRUE;
        // Optional: Enable screen space lighting
        volQual->deferredLighting = TRUE;
        // Optional: If using gradient lighting, increase quality
        volQual->lightingModel = SoVolumeRenderingQuality::OPENGL;
        volQual->surfaceScalarExponent = 5;
        volSep->addChild(volQual);

        // Display volume rendering
        SoVolumeRender* volRend = new SoVolumeRender();
        // Let Inventor compute best number of slices
        volRend->numSlicesControl = SoVolumeRender::AUTOMATIC;
        // Optional: Use lower screen resolution while moving.
        //volRend->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
        volRend->lowScreenResolutionScale = 2;
        // Internal optimization.
        volRend->subdivideTile = TRUE;
        // Remove "slicing" artifacts
        volRend->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
        // Ignore low visibility voxels (OIV 9.3)
#if SO_INVENTOR_VERSION >= 9300
        volRend->opacityThreshold = 0.1f;
#endif
        volSep->addChild(volRend);

        // Set application state for interaction
        m_dragger = dragger;
        m_roiBox = box;

    } // End of VolumeViz section

    // Decoration for testing
    if (1) {
        SoVolumeData* volData = (SoVolumeData*)SoNode::getByName("VolData");
        scene->addChild( MedicalHelper::createBoundingBox(volData->extent.getValue()) );
    }

    // Create a viewer.
    // Note we should do an explicit viewAll() because we created our own camera.
    Widget parent = buildInterface(myWindow);
    SoXtExaminerViewer* viewer = new SoXtExaminerViewer(parent);
    viewer->setBackgroundColor(SbColor(0, .1f, .1f));
    viewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
    viewer->setSize( MedicalHelper::exampleWindowSize() );
    viewer->setSceneGraph(root.ptr());
    viewer->setTitle("Rotated ROI");
    viewer->viewAll();
    viewer->setDecoration(FALSE);
    viewer->saveHomePosition();
    viewer->setViewing(FALSE);
    viewer->show();

    // Loop, then cleanup
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete viewer;
    root = NULL;
    InventorMedical::finish();
    SoDialogViz::finish();
    SoVolumeRendering::finish();
    SoXt::finish();
    return 0;
}

////////////////////////////////////////////////////////////////////////
class myAuditorClass : public SoDialogAuditor
{
protected:
    void dialogPushButton(SoDialogPushButton* cpt);
};

void
myAuditorClass::dialogPushButton(SoDialogPushButton* cpt)
{
    if (cpt->auditorID.getValue() == "resetROI") {
        SbVec3f center = m_roiBox.getCenter();
        SbVec3f scale = m_roiBox.getSize() / 2;

        SoTransformerDragger* dragger = (SoTransformerDragger*)m_dragger;
        dragger->scaleFactor = scale;
        dragger->rotation = SbRotation::identity();
        dragger->translation = center;
    }
}


////////////////////////////////////////////////////////////////////////
Widget buildInterface(Widget window)
{
    SoInput myInput;
    if (!myInput.openFile("$OIVHOME/examples/source/Medical/Tools/medicalRotateROI/rotateRoiDialog.iv"))
        return NULL;
    SoGroup *myGroup = SoDB::readAll(&myInput);
    if (!myGroup)
        return NULL;
    SoTopLevelDialog *myTopLevelDialog = (SoTopLevelDialog *)myGroup->getChild(0);

    myAuditorClass *myAuditor = new myAuditorClass;
    myTopLevelDialog->addAuditor(myAuditor);

    SoDialogCustom *customNode = (SoDialogCustom *)myTopLevelDialog->searchForAuditorId(SbString("Viewer"));

    myTopLevelDialog->buildDialog(window, customNode != NULL);
    myTopLevelDialog->show();

    return customNode ? customNode->getWidget() : window;
}

///////////////////////////////////////////////////////////////////////
//
// Create a wireframe box showing bounding box of volume
//

#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoIndexedLineSet.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoVertexProperty.h>

SoSeparator *makeVolBBox(SbBox3f volSize)
{
    // The box will be easier to see without lighting and with wide lines
    SoLightModel *pLModel = new SoLightModel;
    pLModel->model = SoLightModel::BASE_COLOR;

    SoDrawStyle *pStyle = new SoDrawStyle;
    pStyle->lineWidth = 1; //2;

    // The box should be unpickable so manip can be used inside it
    SoPickStyle *pPickable = new SoPickStyle;
    pPickable->style = SoPickStyle::UNPICKABLE;

    // Create a cube with the geometric size of the volume
    float xmin, xmax, ymin, ymax, zmin, zmax;
    volSize.getBounds(xmin, ymin, zmin, xmax, ymax, zmax);
    SoVertexProperty *pProp = new SoVertexProperty;
    pProp->vertex.set1Value(0, SbVec3f(xmin, ymin, zmin));
    pProp->vertex.set1Value(1, SbVec3f(xmax, ymin, zmin));
    pProp->vertex.set1Value(2, SbVec3f(xmax, ymax, zmin));
    pProp->vertex.set1Value(3, SbVec3f(xmin, ymax, zmin));
    pProp->vertex.set1Value(4, SbVec3f(xmin, ymin, zmax));
    pProp->vertex.set1Value(5, SbVec3f(xmax, ymin, zmax));
    pProp->vertex.set1Value(6, SbVec3f(xmax, ymax, zmax));
    pProp->vertex.set1Value(7, SbVec3f(xmin, ymax, zmax));
    pProp->orderedRGBA.set1Value(0, 0xFF0000FF);

    // Draw it with a line set
    int coordIndices[] = { 0, 1, 2, 3, 0, -1, 4, 5, 6, 7, 4, -1,
        0, 4, -1, 1, 5, -1, 2, 6, -1, 3, 7 };
    int numCoordIndices = sizeof(coordIndices) / sizeof(int);
    SoIndexedLineSet *pLines = new SoIndexedLineSet;
    pLines->vertexProperty = pProp;
    pLines->coordIndex.setValues(0, numCoordIndices, coordIndices);

    // Assemble scene graph
    SoSeparator *pBoxSep = new SoSeparator;
    pBoxSep->addChild(pLModel);
    pBoxSep->addChild(pPickable);
    pBoxSep->addChild(pStyle);
    pBoxSep->addChild(pLines);
    return pBoxSep;
}
