///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** Author      : Pascal Estrade (JANUARY 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.

Demonstrates a simple "ruler" tool for SoOrthoSlice nodes.

-------------------------------------------------------------------------*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>

#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoPolygonOffset.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoCone.h>
#include <Inventor/nodes/SoLineSet.h>

#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/details/SoOrthoSliceDetail.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>

#include <DialogViz/dialog/SoMessageDialog.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Ruler.h>
#include <Medical/nodes/Gnomon.h>

#include <sstream>

///////////////////////////////////////////////////////////////////////////////

const SbString IMAGE_FILENAME = "$OIVHOME/examples/data/Medical/dicomSample/CVH256.dcm";

///////////////////////////////////////////////////////////////////////////////

static SoNode* g_sceneGraph;

static SbBool m_mousePressed = false;
static SbBool m_firstClick   = false;

/** Color of picked voxel */
std::vector<SoMaterial*> g_materialVoxel;

static TextBox* m_infoText = NULL;

///////////////////////////////////////////////////////////////////////////////

static void myMouseMoveCB(void* data, SoEventCallback* eventCB);

static void myMousePressCB(void* data, SoEventCallback* eventCB);

///////////////////////////////////////////////////////////////////////////////

int main(int , char **argv)
{
    Widget myWindow = SoXt::init(argv[0]);
    SoDialogViz::init();
    SoVolumeRendering::init();
    InventorMedical::init();

    if (! SbFileHelper::isAccessible(IMAGE_FILENAME)) {
      new SoMessageDialog( IMAGE_FILENAME, "Unable to open:", SoMessageDialog::MD_ERROR );
      return -1;
    }

    SoRef<SoSeparator> root = new SoSeparator();

    // Camera
    SoOrthographicCamera* camera = new SoOrthographicCamera();
      root->addChild( camera );

    // Volume rendering sub-graph
    SoSeparator* volSep = new SoSeparator();
      root->addChild( volSep );

    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = IMAGE_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    SoMaterial* volMatl = new SoMaterial();
      volMatl->diffuseColor.setValue( 1, 1, 1 );
      volSep->addChild( volMatl );

    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volSep->addChild( volRange );

    SoTransferFunction* volTF = new SoTransferFunction();
      volTF->predefColorMap = SoTransferFunction::INTENSITY;
      MedicalHelper::dicomCheckMonochrome1( volTF, volData );
      volSep->addChild( volTF );

    SoOrthoSlice* slice = new SoOrthoSlice();
      slice->axis = MedicalHelper::AXIAL;
      slice->sliceNumber = volData->data.getSize()[2] / 2;
      slice->interpolation = SoSlice::MULTISAMPLE_12;
      volSep->addChild( slice );

    // Compute a factor to scale Ruler in function of data size.
    float globalFactor = 1.0;
    if (volData ) {
      float x, y, z;
      volData->extent.getValue().getSize(x, y, z);
      globalFactor = (x+y+z)/3/100;
    }

    // Create ruler object
    SoAnnotation* rulerSep = new SoAnnotation();
      root->addChild( rulerSep );

    Ruler *ruler = new Ruler;
      ruler->globalFactor  = globalFactor;
      ruler->getFontNode()->name = "Arial : Bold";
      rulerSep->addChild( ruler );

    // Handle some events
    SoEventCallback* eventNode = new SoEventCallback;
      eventNode->addEventCallback(SoMouseButtonEvent::getClassTypeId(), myMousePressCB, (void*) ruler );
      eventNode->addEventCallback(SoLocation2Event::getClassTypeId(), myMouseMoveCB, (void*) ruler );
      root->addChild( eventNode );

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Annotation
    root->addChild( MedicalHelper::buildSliceAnnotation( camera, slice, &IMAGE_FILENAME ) );

    // Status display
    m_infoText = new TextBox();
      m_infoText->fontSize = 18;
      m_infoText->position.setValue( -0.98f, -0.85f, 0 );
      m_infoText->alignmentH = TextBox::LEFT;
      m_infoText->alignmentV = TextBox::BOTTOM;
      m_infoText->addLine( "Click and drag on the image to measure." );
      root->addChild( m_infoText );

    // Set up viewer
    SoXtPlaneViewer* myViewer = new SoXtPlaneViewer(myWindow);
      myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
      myViewer->setSize( MedicalHelper::exampleWindowSize() );
      myViewer->setDecoration( FALSE );
      myViewer->setViewing(FALSE);
      myViewer->setSceneGraph(root.ptr());
      myViewer->setTitle("Ruler");

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::AXIAL, camera, volData );
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    g_sceneGraph = myViewer->getSceneManager()->getSceneGraph();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}

///////////////////////////////////////////////////////////////////////////////
void myMousePressCB(void* data, SoEventCallback* eventCB)
{
    const SoEvent *event = eventCB->getEvent();

    if (SO_MOUSE_PRESS_EVENT(event, ANY))
    {
        m_mousePressed = true;
        if (! m_firstClick)
            m_firstClick = true;
        myMouseMoveCB(data, eventCB);
    }
    else if( SO_MOUSE_RELEASE_EVENT(event, ANY))
    {
        m_mousePressed = false;
        m_firstClick   = false;
    }
    eventCB->setHandled();
}

///////////////////////////////////////////////////////////////////////////////
void myMouseMoveCB(void* data, SoEventCallback* eventCB)
{
    const SoEvent *event = eventCB->getEvent();
    if (m_mousePressed)
    {
      SoHandleEventAction* action = eventCB->getAction();

      const SbViewportRegion &myRegion = eventCB->getAction()->getViewportRegion();
      SoRayPickAction pickaction = SoRayPickAction(myRegion);
      pickaction.setPoint(event->getPosition());
      pickaction.setSceneManager(eventCB->getAction()->getSceneManager());
      pickaction.apply( eventCB->getAction()->getPickRoot() );

      // Retrieve picked point (if any)
      const SoPickedPoint* p = action->getPickedPoint();   //pickaction.getPickedPoint();
      if (p == NULL)
        return;

      // Get slice detail to pass to the ruler.  Pass if we picked something different.
      const SoOrthoSliceDetail* detail = dynamic_cast<const SoOrthoSliceDetail*>(p->getDetail());
      if (detail == NULL)
        return;

        // Let ruler manage the mouse.
        Ruler* ruler = (Ruler*)data;
        m_firstClick = ruler->manageMouseMove( m_firstClick, detail );

        // Display position and value
        //g_infoText->string.deleteValues(0);
        std::ostringstream buffer;
        buffer << "Position IJK: " << detail->getValueDataPos();
        //g_infoText->string.set1Value(0, SbString(buffer.str().c_str()));
        m_infoText->setLine( buffer.str().c_str(), 0 );
        buffer.str("");
        buffer << "Position XYZ: " << detail->getValueObjectPos();
        //g_infoText->string.set1Value(1, SbString(buffer.str().c_str()));
        m_infoText->setLine( buffer.str().c_str(), 1 );
        buffer.str("");
        buffer << "Voxel value: " << detail->getValue();
        //g_infoText->string.set1Value(2, SbString(buffer.str().c_str()));
        m_infoText->setLine( buffer.str().c_str(), 2 );
        buffer.str("");
    }
    eventCB->setHandled();
}
