///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a simple volumeData and volumeRender node.
author : Jerome Hummel
August 2002
Updaded by Pascal Estrade (Sep 2014)
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoAnnoText3Property.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>

#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>
#include <VolumeViz/nodes/SoVolumeSkin.h>

#include <Inventor/SoPreferences.h>

#include <DialogViz/SoDialogVizAll.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <MeshViz/graph/PoGroup6Axis3.h>
#include <MeshViz/graph/PoGroup3Axis3.h>
#include <MeshViz/graph/PoLinearAxis.h>

#include <MeshViz/nodes/PoDomain.h>
#include <MeshViz/nodes/PoEllipticProfile.h>
#include <MeshViz/nodes/PoIrregularMesh1D.h>
#include <MeshViz/nodes/PoMesh1DHints.h>
#include <MeshViz/nodes/PoMiscTextAttr.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////
#define FILENAME "$OIVHOME/examples/source/Medical/Tools/medicalSimpleVolumeAxis/medicalSimpleVolumeAxis.iv"

// Defining interactive interface file definition
#define INTERFACE_FILE "$OIVHOME/examples/source/Medical/Tools/medicalSimpleVolumeAxis/simpleVolumeAxisInterface.iv"


SoRef<SoGroup> g_myGroup = NULL;

///////////////////////////////////////////////////////////////////////////////
// Define interaction interface listener
class AxisTypeAuditor : public SoDialogAuditor
{
public:
  AxisTypeAuditor(SoTopLevelDialog* topDialog, SoSwitch *axis_switch)
  {
    m_top = topDialog;
    m_axis_switch = axis_switch;

    SoDialogViz* elt = topDialog->searchForAuditorId("axisType");

    SoDialogComboBox* cpt = (SoDialogComboBox*)(elt);
    dialogComboBox(cpt);
  }

  ~AxisTypeAuditor()
  {
    m_top = NULL;
    m_axis_switch = NULL;
  }

private:
  SoTopLevelDialog  *m_top;
  SoSwitch          *m_axis_switch;

  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "axisType")
    {
      int selectedItem = cpt->selectedItem.getValue();
      if (m_axis_switch != NULL)
      {
          if (selectedItem) {
              m_axis_switch->whichChild = selectedItem-1; // Because of DialogViz (First elt == 1).
          }
          else {
              m_axis_switch->whichChild = SO_SWITCH_NONE;
          }
      }
    }
  }
};

///////////////////////////////////////////////////////////////////////////////
// Define interaction interface listener
class AnnoTextTypeAuditor : public SoDialogAuditor
{
public:
  AnnoTextTypeAuditor(SoTopLevelDialog* topDialog, SoAnnoText3Property* axisProperty)
  {
    m_top = topDialog;
    m_axisProperty = axisProperty;

    SoDialogViz* elt = topDialog->searchForAuditorId("annotationType");

    SoDialogComboBox* cpt = (SoDialogComboBox*)(elt);
    dialogComboBox(cpt);
  }

  ~AnnoTextTypeAuditor()
  {
    m_top = NULL;
    m_axisProperty = NULL;
  }

private:
  SoTopLevelDialog  *m_top;
  SoAnnoText3Property *m_axisProperty;

  void dialogComboBox(SoDialogComboBox* cpt)
  {
    if (cpt->auditorID.getValue() == "annotationType")
    {
      int selectedItem = cpt->selectedItem.getValue();

      if (m_axisProperty != NULL)
      {
        m_axisProperty->renderPrintType = selectedItem;
      }
    }
  }
};

///////////////////////////////////////////////////////////////////////////////
// Load a scene graph from an OIV file
SoSeparator *
loadSceneGraph(const char *filename)
{
  // Open the input file
  SoInput mySceneInput;
  if (!mySceneInput.openFile(filename))
  {
    new SoMessageDialog( filename, "Cannot open file ", SoMessageDialog::MD_ERROR );
    return NULL;
  }

  // Read the whole file into the database
  SoSeparator *myGraph = SoDB::readAll(&mySceneInput);
  if (myGraph == NULL)
  {
    new SoMessageDialog( filename, "Problem reading file", SoMessageDialog::MD_ERROR );
    return NULL;
  }

  mySceneInput.closeFile();
  return myGraph;
}

///////////////////////////////////////////////////////////////////////////////
// Build the interface from the file
Widget
buildInterface(Widget window, const char* filename, const char* viewer,
SoTopLevelDialog** myTopLevelDialog)
{
  SoInput myInput;
  if (!myInput.openFile(filename))
    return NULL;

  g_myGroup = SoDB::readAll(&myInput);
  if (!g_myGroup.ptr())
    return NULL;

  *myTopLevelDialog = (SoTopLevelDialog *)g_myGroup->getChild(0);

  SoDialogCustom *customNode = (SoDialogCustom *)(*myTopLevelDialog)->searchForAuditorId(SbString(viewer));

  (*myTopLevelDialog)->buildDialog(window, customNode != NULL);
  (*myTopLevelDialog)->show();

  return customNode ? customNode->getWidget() : window;
}

///////////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init(argv[0]);
  if (!myWindow) return 0;

  // Initialize
  PoBase::init();
  SoVolumeRendering::init();
  SoDialogViz::init();
  InventorMedical::init();

  // Assemble the scene graph
  SoRef<SoSeparator> root = new SoSeparator();
  
  // Camera
  SoPerspectiveCamera* camera = new SoPerspectiveCamera();
  root->addChild( camera );

  SoSeparator* scene = loadSceneGraph(FILENAME);
  if (scene != NULL)
    root->addChild( scene );

  // OIV Logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Compass
  root->addChild( new Gnomon() );

  // Adding a box to select axis type
  SoTopLevelDialog* myTopLevelDialog = NULL;
  Widget myParent = buildInterface(myWindow, INTERFACE_FILE, "Viewer", &myTopLevelDialog);

  // Connect the switch to the combobox
  SoSwitch *axisSwitch = (SoSwitch*)root->getByName("AxisSwitch");
  AxisTypeAuditor* myAxisTypeAuditor = new AxisTypeAuditor(myTopLevelDialog, axisSwitch);
  myTopLevelDialog->addAuditor(myAxisTypeAuditor);

  // Connect the field to the combobox
  SoAnnoText3Property *axisProperty = (SoAnnoText3Property*)root->getByName("AxisProperty");
  AnnoTextTypeAuditor* myAxisTextTypeAuditor = new AnnoTextTypeAuditor(myTopLevelDialog, axisProperty);
  myTopLevelDialog->addAuditor(myAxisTextTypeAuditor);

  // Set up viewer:
  SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myParent);
  myViewer->setSceneGraph(root.ptr());
  myViewer->setTitle("Volume rendering & Axis");
  myViewer->setDecoration(false);
  myViewer->setSize( MedicalHelper::exampleWindowSize() );

  // Adjust camera (more interesting view)
  myViewer->getCamera()->orientation.setValue( SbVec3f(0.81f, 0.366f, 0.455f), 1.542f );
  myViewer->viewAll();
  MedicalHelper::dollyZoom( 2.0f, myViewer->getCamera() ); // Fill viewport
  myViewer->saveHomePosition();

  // Run then cleanup
  myViewer->show();
  SoXt::show(myWindow);
  SoXt::mainLoop();
  root = NULL;
  g_myGroup = NULL;
  delete myViewer;
  delete myAxisTextTypeAuditor;
  delete myAxisTypeAuditor;
  InventorMedical::finish();
  SoDialogViz::finish();
  SoVolumeRendering::finish();
  PoBase::finish();
  SoXt::finish();
  return 0;
}
