///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : VSG (MMM YYYY)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoCallback.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoEventCallback.h>
#include <Inventor/nodes/SoFontStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoText2.h>
#include <Inventor/nodes/SoTranslation.h>

#include <Inventor/SoPickedPoint.h>
#include <Inventor/actions/SoRayPickAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/events/SoLocation2Event.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/helpers/SbFileHelper.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/details/SoVolumeRenderDetail.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

#include <DialogViz/dialog/SoDialogViz.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILENAME = "$OIVHOME/examples/data/Medical/files/head.ldm";

const SbString ANNOTATION_FILENAME = "$OIVHOME/examples/source/Medical/Tools/medicalVolumePickingGpu/info.iv";

static SoNode*       g_sceneGraph  = NULL;
static SoText2*      g_infoText    = NULL;
static bool g_mousePressed = false;

/** Color of picked voxel */
std::vector<SoMaterial*> g_materialVoxel;

///////////////////////////////////////////////////////////////////////////////

static SoSeparator* displayInfo();

static void myMouseMoveCB(void* /*data*/, SoEventCallback* eventCB);

static void myMousePressCB(void* data, SoEventCallback* eventCB);

///////////////////////////////////////////////////////////////////////////////
int main(int /*argc*/, char **argv)
{
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow)
        return 0;

    SoVolumeRendering::init();
    SoDialogViz::init();
    InventorMedical::init();

    SoRef<SoSeparator> root = new SoSeparator;

    // Handle events
    SoEventCallback *eventCB = new SoEventCallback;
      eventCB->addEventCallback(SoMouseButtonEvent::getClassTypeId(), myMousePressCB);
      eventCB->addEventCallback(SoLocation2Event::getClassTypeId(), myMouseMoveCB);
      root->addChild( eventCB );

    // Camera
    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
      root->addChild( camera );

    // Volume rendering sub-graph
    SoSeparator* volSep = new SoSeparator();
      root->addChild( volSep );

    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILENAME;
      MedicalHelper::dicomAdjustVolume( volData );
      volSep->addChild( volData );

    SoDataRange* volRange = new SoDataRange();
      MedicalHelper::dicomAdjustDataRange( volRange, volData );
      volRange->min = 35;  // Remove air and noise voxels
      volSep->addChild( volRange );

    SoTransferFunction* volTF = new SoTransferFunction();
      volTF->predefColorMap = SoTransferFunction::STANDARD;
      volTF->minValue = 1;  // Index 0 will be transparent
      volSep->addChild( volTF );

    SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
      volQual->voxelizedRendering = TRUE; // Specialized rendering mode for precise selection
      volQual->voxelOutline       = TRUE;
      volQual->voxelOutlineThreshold = 2;
      volQual->ambientOcclusion   = TRUE;
      volQual->deferredLighting   = TRUE;
      volQual->interpolateOnMove  = TRUE;
      volSep->addChild( volQual );

    SoVolumeRender* volRender = new SoVolumeRender();
	  volRender->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
      volRender->lowScreenResolutionScale = 2;
      volSep->addChild( volRender );

    // Add status display
    root->addChild(displayInfo());

    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation
    root->addChild( new Gnomon() );

    // Note
    TextBox* text = new TextBox();
      text->position.setValue(0, -0.98f, 0); // Normalized device coordinates
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine("\"Voxelized rendering\" displays each voxel as a cube for accurate selection.");
      text->addLine("Click or Press-and-drag to interactively display voxel properties.");
      root->addChild(text);

    // Set up viewer
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer(myWindow);
      myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
      myViewer->setSceneGraph(root.ptr());
      myViewer->setTitle("Volume Picking Gpu");
      myViewer->setDecoration(FALSE);
      myViewer->setViewing(FALSE);
      myViewer->setSize( MedicalHelper::exampleWindowSize() );

    // Adjust camera for close-up view
    camera->position.setValue( 0.615137f, -0.483181f, -0.0412909f );
    camera->orientation.setValue( SbVec3f(0.674393f, 0.494346f, 0.548466f), 1.86629f );
    camera->focalDistance = 1;
    myViewer->adjustClippingPlanes();
    myViewer->saveHomePosition();

    g_sceneGraph = myViewer->getSceneManager()->getSceneGraph();

    // Run then cleanup
    myViewer->show();
    SoXt::show(myWindow);
    SoXt::mainLoop();
    delete myViewer;
    root = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}


///////////////////////////////////////////////////////////////////////////////
// Create annotation sub-graph
SoSeparator* displayInfo()
{
    SoSeparator* infoSep = MedicalHelper::readFile(ANNOTATION_FILENAME.toLatin1());
    infoSep->ref();
    SoMaterial* voxel0Mat = MedicalHelper::find<SoMaterial>(infoSep, "VOXELMAT");
    g_materialVoxel.push_back(voxel0Mat);
    g_infoText = MedicalHelper::find<SoText2>(infoSep);
    infoSep->unrefNoDelete();

    return infoSep;
}

///////////////////////////////////////////////////////////////////////////////
// Called on mouse move events
void myMouseMoveCB(void* /*data*/, SoEventCallback* eventCB)
{
    const SoEvent *event = eventCB->getEvent();
    if ( g_mousePressed )
    {
        const SbViewportRegion &myRegion = eventCB->getAction()->getViewportRegion();
        SoRayPickAction pickaction = SoRayPickAction(myRegion);
        pickaction.setPoint(event->getPosition());
        pickaction.setSceneManager(eventCB->getAction()->getSceneManager());
        pickaction.apply(g_sceneGraph);

        SoPickedPoint* p = pickaction.getPickedPoint();
        if ( !p )
            return;

        const SoVolumeRenderDetail* detail = dynamic_cast<const SoVolumeRenderDetail*>(p->getDetail());
        if ( !detail )
            return;

        std::vector<SbVec4ub> rgba;
        std::vector<SbVec3f> objPos;
        std::vector<SbVec3i32> dataPos;
        std::vector<SoLDMTileID> tileIds;
        //Ignore fully transparent voxels
        detail->getRgbaValues(rgba, objPos, dataPos, tileIds, 0);

        if ( !rgba.empty() )
        {
            g_materialVoxel[0]->diffuseColor.set1Value(0, rgba[0][0]/255.f, rgba[0][1]/255.f, rgba[0][2]/255.f);
            g_materialVoxel[0]->transparency.set1Value(0, 1.f-rgba[0][3]/255.f);

            g_infoText->string.deleteValues(0);
            std::ostringstream buffer;
            buffer << "Position IJK: " << dataPos[0];
            g_infoText->string.set1Value(0, SbString(buffer.str().c_str()));
            buffer.str("");
            buffer << "Position XYZ: " << objPos[0];
            g_infoText->string.set1Value(1, SbString(buffer.str().c_str()));
            buffer.str("");
            buffer << "RGBA: " << rgba[0];
            g_infoText->string.set1Value(2, SbString(buffer.str().c_str()));
        }
    }
    eventCB->setHandled();
}

///////////////////////////////////////////////////////////////////////////////
// Called on mouse press events
void myMousePressCB(void* data, SoEventCallback* eventCB)
{
    const SoEvent *event = eventCB->getEvent();

    if (SO_MOUSE_PRESS_EVENT(event, ANY))
    {
        g_mousePressed = true;
        myMouseMoveCB(data, eventCB);
    }
    else if( SO_MOUSE_RELEASE_EVENT(event, ANY))
    {
        g_mousePressed = false;
    }
    eventCB->setHandled();
}

