///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Pascal Estrade (Nov 2014)
**=======================================================================*/

/*-----------------------------------------------------------------------
Medical example program.
Purpose : Demonstrate how to create zoom mechanism.
Description : Main
-------------------------------------------------------------------------*/
//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtPlaneViewer.h>
#include <Inventor/nodes/SoSeparator.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoDataRange.h>
#include <Inventor/nodes/SoText3.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoAnnotation.h>
#include <Inventor/nodes/SoOrthographicCamera.h> 
#include <Inventor/Gui/view/PoSceneView.h>
#include <VolumeViz/readers/SoVRDicomFileReader.h>
#include <VolumeViz/nodes/SoOrthoSlice.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/draggers/SoTranslate2Dragger.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/sensors/SoNodeSensor.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/DicomInfo.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

// Data Set
#define FILENAME "$OIVHOME/examples/data/Medical/dicomSample/listOfDicomFiles.dcm"

///////////////////////////////////////////////////////////////////////////////
int main(int, char **argv)
{
    // Create the window, initialize OpenIventor and VolumeViz.
    Widget myWindow = SoXt::init(argv[0]);
    if (!myWindow)
        return 0;

    SoVolumeRendering::init();	
    InventorMedical::init();

    // Create the scene graph.
    SoRef<SoSeparator> root = new SoSeparator();

    // Main camera (we will adjust the view volume after creating the viewer).
    SoOrthographicCamera *mainCam = new SoOrthographicCamera();
    root->addChild ( mainCam );

    //===================================================================================================================
    // Here we create the rectangle used to select the zone we will zoom.
    // We use for that a customized translate2Dragger.
    // The translation field of this dragger is connected to the camera position of the zoom viewer (zoomCam).
    SoTranslate2Dragger *zoomDragger = new SoTranslate2Dragger();
    
      // Create the replacement geometry the dragger will display.
      SoSeparator *zoomCubeSep = new SoSeparator();
    
      SoLightModel *zoomLightModel = new SoLightModel();
      zoomLightModel->model = SoLightModel::BASE_COLOR;           // Lighting off
      zoomCubeSep->addChild( zoomLightModel );
    
      SoMaterial *zoomCubeMaterial = new SoMaterial();
      zoomCubeMaterial->diffuseColor.setValue(SbColor( 1, 0,0));  // Red
      zoomCubeSep->addChild( zoomCubeMaterial );
    
      SoDrawStyle *zoomCubeDrawStyle = new SoDrawStyle();
      zoomCubeDrawStyle->style = SoDrawStyle::LINES;              // Cube will be drawn as lines
      zoomCubeSep->addChild( zoomCubeDrawStyle ); 
    
      // The center of this cube (square) is the center of the zoom camera (zoomCam).
      // Initially the zoom region will be 20 mm (assuming the data is DICOM).
      SoCube *zoomCube = new SoCube();
      zoomCube->height = 20;
      zoomCube->depth  = 1;
      zoomCube->width  = 20;
      zoomCubeSep->addChild( zoomCube );

      // Replace the dragger's default geometry.
      zoomDragger->setPart("translatorActive", zoomCubeSep);
      zoomDragger->setPart("translator"      , zoomCubeSep);
      zoomDragger->setPart("xAxisFeedback"   , new SoSeparator() );
      zoomDragger->setPart("yAxisFeedback"   , new SoSeparator() );
    root->addChild(zoomDragger);
    //===================================================================================================================

    // The image/volume rendering nodes...
    SoSeparator *scene = new SoSeparator();

    // Data
    SoVolumeData *volData = new SoVolumeData();
    {
        // Data set
        volData->fileName = FILENAME;
          MedicalHelper::dicomAdjustVolume( volData );
          scene->addChild(volData);

        // Data range: Use window center/width from the DICOM data if possible.
        SoDataRange* dataRange = new SoDataRange();
          MedicalHelper::dicomAdjustDataRange( dataRange, volData );
          scene->addChild( dataRange );

        // Colormap: Use grayscale (intensity ramp)
        SoTransferFunction *transFunc = new SoTransferFunction();
          transFunc->predefColorMap = SoTransferFunction::INTENSITY;
          scene->addChild( transFunc );
    }

    // Otho slice rendering & manipulation
    SoOrthoSlice *orthoSlice = new SoOrthoSlice();
    {
        orthoSlice->sliceNumber = 256;
        orthoSlice->interpolation = SoOrthoSlice::MULTISAMPLE_12;
        scene->addChild( orthoSlice );
    }
    root->addChild( scene );
    
    // OIV Logo
    root->addChild( MedicalHelper::exampleLogoNode() );

    // Medical Gnomon.
    Gnomon *gnomon = new Gnomon();
    root->addChild( gnomon );

    // DICOM annotation
    root->addChild( MedicalHelper::exampleDicomAnnotation( "$OIVHOME/examples/data/Medical/dicomSample/CVH001.dcm" ));

    //===================================================================================================================
    // Scene graph for the zoom.
    // This scene graph has its own camera and will be displayed in a separate window.
    SoRef<SoSeparator> zoomSep = new SoSeparator();
      zoomSep->addChild(new SoDirectionalLight());
    
      // The zoom camera will be "slaved"
      // - To the main camera for its orientation, and
      // - To the dragger for its position.
      SoOrthographicCamera *zoomCam = new SoOrthographicCamera();
      zoomCam->orientation.connectFrom( &(mainCam->orientation) );
      zoomCam->position.connectFrom( &(zoomDragger->translation) );
      zoomSep->addChild(zoomCam);

      // This camera will view the same scene as the main camera (just with a different view volume).
      zoomSep->addChild(scene);
      zoomCam->viewAll( scene, SbViewportRegion(100, 100) );
    
      // Zoom factor.
      zoomCam->height.connectFrom( &(zoomCube->height) );
    //===================================================================================================================

    // setup viewer for zoom result.
    SoXtRenderArea *zoomViewer = new SoXtRenderArea(myWindow);
    zoomViewer->setTitle("Zoom");
    zoomViewer->setSceneGraph(zoomSep.ptr());
    zoomViewer->setSize( MedicalHelper::exampleWindowSize()  );
#ifdef _WIN32
    MoveWindow( (HWND)(myWindow), 1024, 0, 633, 633, true); 
#endif
    zoomViewer->show();

    // Set up viewer.
    SoXtPlaneViewer *mainViewer = new SoXtPlaneViewer(NULL, "Zoom Demonstration" , FALSE);
    mainViewer->setDecoration(FALSE);
    mainViewer->setViewing(FALSE);
    mainViewer->setSize( MedicalHelper::exampleWindowSize() );
    mainViewer->setSceneGraph(root.ptr());
    mainViewer->viewAll();
    mainCam->height = 175;
    mainViewer->show();

    // Orient the camera for the desired axis.
    // Note: If we use an axis other than AXIAL we need to rotate the dragger.
    //       But zoomCam is slaved to this one, so we don't need to adjust it.
    MedicalHelper::orientView( MedicalHelper::AXIAL, mainCam, volData );

    // Make sure the dragger is positioned in front of the volume (so we can see it).
    // We can use the view volume to get a point exactly at the front of the view volume.
    SbViewVolume viewVol = mainCam->getViewVolume( mainViewer->getViewportRegion().getViewportAspectRatio() );
    SbVec3f      dragPos = viewVol.getSightPoint( viewVol.getNearDist() + 1 );
    zoomDragger->translation = dragPos;

    // Run
    SoXt::show(myWindow);
    SoXt::mainLoop();

    // Clean up.
    delete zoomViewer;
    delete mainViewer;

    root = NULL;
    zoomSep = NULL;

    InventorMedical::finish();
    SoVolumeRendering::finish();

    SoXt::finish();
    return 0;
}

