///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*=======================================================================
** 2DTransferFunction
** Author      : Julien Sallanne (Jun 2011)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

#include "SoTransferFunction2D.h"

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/SoXtColorWheel.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/manips/SoTabBoxManip.h>

#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoImageBackground.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoRenderToTextureProperty.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodes/SoTransform.h>
#include <Inventor/nodes/SoVertexProperty.h>

#include <Inventor/nodes/SoLightModel.h>

#include <LDM/nodes/SoDataRange.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>

#include <Inventor/drawers/SoLassoScreenDrawer.h>
#include <Inventor/drawers/SoPolygonScreenDrawer.h>

#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/engines/SoCalculator.h>

#include <DialogViz/auditors/SoDialogAuditor.h>
#include <DialogViz/dialog/SoColumnDialog.h>
#include <DialogViz/dialog/SoDialogCheckBox.h>
#include <DialogViz/dialog/SoDialogComboBox.h>
#include <DialogViz/dialog/SoDialogCustom.h>
#include <DialogViz/dialog/SoDialogRealSlider.h>
#include <DialogViz/dialog/SoTopLevelDialog.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>

static SoTopLevelDialog* pTopLevelDialog0 = NULL;
static SoMaterial* pColor = NULL;
static SoRef<SoSeparator> pRoot = NULL;
static SoRef<SoSeparator> pToRender = NULL;
static SoRef<SoSeparator> pRenderRoot = NULL;

// Comment out the following line in order to use a second dataset for 2DTransferFunction.
// #define USE_DATA2

// Comment out the following line in order to use a dataset using short data.
// #define SHORT_DATA

SoShape*
createFrom2DPoints( const std::vector<SbVec2f>& pointsInCam )
{
  std::vector<SbVec3f> polygon( pointsInCam.size() );
  for ( unsigned int i = 0; i < pointsInCam.size(); ++i )
  {
    polygon[i].setValue( pointsInCam[i][0], pointsInCam[i][1], 0.5f );
  }

  SoVertexProperty* polygonVertices = new SoVertexProperty();
  polygonVertices->vertex.setValues( 0, (int) polygon.size(), &polygon[0] );

  SoFaceSet* polygonFaceSet = new SoFaceSet();
  polygonFaceSet->vertexProperty = polygonVertices;
  return polygonFaceSet;
}

void
colFinalizedCallback( void*, const float hsv[3] )
{
  SbColor color;
  color.setHSVValue( hsv );
  pColor->diffuseColor = color;
}

// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line, in camera space, and add a shape thanks to this points.
void
lineFinalizedCallback( SoPolyLineScreenDrawer::EventArg& eventArg )
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

  // If less that 2 points, shape cannot be generated.
  if ( lineDrawer->point.getNum() < 3 )
  {
    lineDrawer->clear();
    return;
  }

  // Retrieve points of line in camera space.
  std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
  for ( unsigned int i = 0; i < lineInCam.size(); ++i )
  {
    lineInCam[i][0] = lineDrawer->point[i][0];
    lineInCam[i][1] = lineDrawer->point[i][1];
  }

  // Shape is put in a switch to easily enable/disable it thanks to a check box.
  SoSwitch* switchNode = new SoSwitch;

  SoSeparator* shapesRoot = MedicalHelper::find<SoSeparator>( pRoot.ptr(), "shapesRoot" );

  SoSeparator* manipSeparator = new SoSeparator;

  shapesRoot->addChild( manipSeparator );
  manipSeparator->addChild( switchNode );

  // Add colorEditor values into Material.
  SoMaterial* material = new SoMaterial;
  material->diffuseColor = pColor->diffuseColor;
  material->transparency = pColor->transparency;

  // Add a manipulator.
  SoTabBoxManip* manip = new SoTabBoxManip;

  // Create checkbox.
  SoDialogCheckBox* checkBox = new SoDialogCheckBox;
  checkBox->label = shapesRoot->getNumChildren() - 1;
  checkBox->state = TRUE;
  checkBox->fixedWidth = TRUE;
  checkBox->fixedHeight = TRUE;
  SoColumnDialog* columnDialog = (SoColumnDialog*) pTopLevelDialog0->searchForAuditorId( "columnDialogShape" );
  columnDialog->addChild( checkBox );

  // Link checkbox to switch:
  // We use an engine to convert from TRUE/FALSE to SO_SWITCH_NONE/SO_SWITCH_ALL
  // TRUE/FALSE are respectively 0/1 and SO_SWITCH_NONE/SO_SWITCH_ALL are respectively -3/-1.
  // Using operation "oa = -2*a - 1", if a = 0, oa = -1 and if a = 1, oa = -3.
  // Use a calculator avoid defining a custom engine.
  SoCalculator* boolToInt = new SoCalculator;
  boolToInt->a.connectFrom( &checkBox->state );
  boolToInt->expression = "oa = -2*a - 1";
  switchNode->whichChild.connectFrom( &boolToInt->oa );

  switchNode->addChild( manip );
  switchNode->addChild( material );

  // Add shape.
  SoShape* shape = createFrom2DPoints( lineInCam );
  switchNode->addChild( shape );

  SoSeparator* shapesRoot_RenderToTex = MedicalHelper::find<SoSeparator>( pToRender.ptr(), "shapesRoot_RenderToTex" );

  SoSwitch* switchNode2 = new SoSwitch;

  SoSeparator* transformSeparator = new SoSeparator;

  shapesRoot_RenderToTex->addChild( transformSeparator );
  transformSeparator->addChild( switchNode2 );

  // Link checkbox to switch.
  switchNode2->whichChild.connectFrom( &switchNode->whichChild );

  SoTransform* transform = new SoTransform;
  transform->translation.connectFrom( &manip->translation );
  transform->rotation.connectFrom( &manip->rotation );
  transform->scaleFactor.connectFrom( &manip->scaleFactor );
  transform->scaleOrientation.connectFrom( &manip->scaleOrientation );
  transform->center.connectFrom( &manip->center );

  switchNode2->addChild( material );
  switchNode2->addChild( transform );
  switchNode2->addChild( shape );

  // Don't forget to clear line.
  lineDrawer->clear();
  action->setHandled();
}

class auditorClass : public SoDialogAuditor
{
public:
private:
  void
  dialogRealSlider( SoDialogRealSlider* cpt )
  {
    float value = cpt->value.getValue();

    if ( cpt->auditorID.getValue() == "transparency" )
    {
      pColor->transparency.set1Value( 0, value );
    }
  }
};

SoVolumeData::LDMDataAccess::DataInfoBox
getData( SoVolumeData* volData, SoCpuBufferObject* cpuBufferObject )
{
  SbVec3i32 dim = volData->data.getSize();
  SbBox3f volSize = volData->extent.getValue();
  SbBox3i32 box( SbVec3i32( 0, 0, 0 ), SbVec3i32( dim[0] - 1, dim[1] - 1, dim[2] - 1 ) );

  // Call with NULL buffer (default 3rd parameter) to get number of bytes required to hold data.
  SoVolumeData::LDMDataAccess::DataInfoBox info;
  SoBufferObject* dummyPtr = NULL;
  info = volData->getLdmDataAccess().getData( 0, box, dummyPtr );

  // Fetch the data contained in the ROI in the original volume.
  if ( info.bufferSize )
  {
    cpuBufferObject->setSize( info.bufferSize );
    info = volData->getLdmDataAccess().getData( 0, box, cpuBufferObject );
    return info;
  }
  return info;
}

/**
 * Remap the value 'value' from its input range [ originalRangeMin, originalRangeMax ] to a new range
 * [ targetedRangeMin, targetedRangeMax ].
 */

unsigned short
map( unsigned short value, unsigned short originalRangeMin, unsigned short originalRangeMax, unsigned short targetedRangeMin, unsigned short targetedRangeMax )
{
  return targetedRangeMin + ( value - originalRangeMin ) * ( targetedRangeMax - targetedRangeMin ) / ( originalRangeMax - originalRangeMin );
}

/**
 * This function creates the histogram view. The X axis represents the data value and
 * the Y axis represents the gradient value. The input data can either be a byte data or
 * unsigned short data. In the case of a BYTE data, HISTOGRAM_RESOLUTION must be set to 256
 * as data value are in the interval [ 0,255 ].
 * The gradient is computed using the centralDiff algorithm: We do the difference in data value
 * on each axis and devide by sqrt(3).
 * The histogram represent the number of voxels in the input data with the X value equal to the
 * given data value and the Y value equal to the gradient value. To increase a bin of the histogram,
 * we need to correctly map both voxel value and gradient from their input range to the histogram range:
 * Voxel: [ dataRangeMin, dataRangeMax ] -> [ 0, HISTOGRAM_RESOLUTION-1 ].
 * Gradient: [ 0, maxGradient ] -> [ 0, HISTOGRAM_RESOLUTION-1 ].
 *
 * The histogram is built with an image and each pixel of the image can only be assigned with a byte
 * value. In our case, we decided to clamp all values superior to the 255 to 255.
 *
 * Finally, the scene graph needs to have a SoDataRange node so the values from this range are sent
 * to the GPU and a correct and similar computation can be done in the shader.
 *
 * Important note: To avoid more complexity, we did not define a template function but the type of data
 * in the buffer slicePtr1 needs to be changed accordingly to the input data type.
 */

void
send2DHistogram( SoImageBackground* texHistogram )
{
#ifndef SHORT_DATA
#define HISTOGRAM_RESOLUTION 256
#else
#define HISTOGRAM_RESOLUTION 512
#endif

  // Compute 2D Histogram.
  SoRef<SoCpuBufferObject> cpuBufferObject1 = new SoCpuBufferObject;
  SoVolumeData* data = MedicalHelper::find<SoVolumeData>( pRenderRoot.ptr(), "volData1" );
  SoVolumeData::LDMDataAccess::DataInfoBox info1 = getData( data, cpuBufferObject1.ptr() );
  void* values1 = cpuBufferObject1->map( SoCpuBufferObject::READ_ONLY );

#ifndef SHORT_DATA
  // For byte data such as tooth.ldm
  const unsigned char* slicePtr1 = static_cast<unsigned char*>( values1 );
#else
  // For unsigned short data such as medicalFootUnsigned.ldm
  const unsigned short* slicePtr1 = static_cast<unsigned short*>( values1 );
#endif

#ifdef USE_DATA2
  SoRef<SoCpuBufferObject> cpuBufferObject2 = new SoCpuBufferObject;
  SoVolumeData* data2 = MedicalHelper::find<SoVolumeData>( pRenderRoot.ptr(), "volData2" );
  SoVolumeData::LDMDataAccess::DataInfoBox info2 = getData( data2, cpuBufferObject2.ptr() );
  void* values2 = cpuBufferObject2->map( SoCpuBufferObject::READ_ONLY );
  const unsigned char* slicePtr2 = static_cast<unsigned char*>( values2 );
#endif

  int64_t min, max;
  data->getMinMax( min, max );
  const unsigned short dataRange = (unsigned short) ( max - min );

  if ( info1.errorFlag == SoVolumeData::LDMDataAccess::CORRECT )
  {
    unsigned char histo[HISTOGRAM_RESOLUTION][HISTOGRAM_RESOLUTION];
    memset( *histo, 0, sizeof( histo ) );
    unsigned short value1, value2;
    unsigned short value1Remaped, value2Remaped;

    const double gradientMax = (float) dataRange / sqrt( 3.f );
    const unsigned short gradientMaxValue = (unsigned short) ( sqrt( double( 3 * ( gradientMax * gradientMax ) ) ) );

    for ( int k = 0; k < info1.bufferDimension[2]; k++ )
    {
      for ( int j = 0; j < info1.bufferDimension[1]; j++ )
      {
        for ( int i = 0; i < info1.bufferDimension[0]; i++ )
        {
          const int index = i + j * info1.bufferDimension[0] + k * info1.bufferDimension[0] * info1.bufferDimension[1];
          value1 = slicePtr1[index];
          value1Remaped = map( value1, (unsigned short) min, (unsigned short) max, 0, HISTOGRAM_RESOLUTION - 1 );

#ifdef USE_DATA2
          value2 = slicePtr2[index];
          value2Remaped = map( value2, (unsigned short) min, (unsigned short) max, 0, HISTOGRAM_RESOLUTION - 1 );
#else
          // Compute gradient into CPU with centralDiff.
          int neighborsIndex[6];
          neighborsIndex[0] = ( i == info1.bufferDimension[0] - 1 ) ? index : index + 1;
          neighborsIndex[1] = ( i == 0 ) ? index : index - 1;

          neighborsIndex[2] =
            ( j == info1.bufferDimension[1] - 1 ) ? i + (j) *info1.bufferDimension[0] + k * info1.bufferDimension[0] * info1.bufferDimension[1] : i + ( j + 1 ) * info1.bufferDimension[0] + k * info1.bufferDimension[0] * info1.bufferDimension[1];

          neighborsIndex[3] = ( j == 0 ) ? i + (j) *info1.bufferDimension[0] + k * info1.bufferDimension[0] * info1.bufferDimension[1] : i + ( j - 1 ) * info1.bufferDimension[0] + k * info1.bufferDimension[0] * info1.bufferDimension[1];

          neighborsIndex[4] =
            ( k == info1.bufferDimension[2] - 1 ) ? i + j * info1.bufferDimension[0] + (k) *info1.bufferDimension[0] * info1.bufferDimension[1] : i + j * info1.bufferDimension[0] + ( k + 1 ) * info1.bufferDimension[0] * info1.bufferDimension[1];

          neighborsIndex[5] = ( k == 0 ) ? i + j * info1.bufferDimension[0] + (k) *info1.bufferDimension[0] * info1.bufferDimension[1] : i + j * info1.bufferDimension[0] + ( k - 1 ) * info1.bufferDimension[0] * info1.bufferDimension[1];

          SbVec3f gradient;
          gradient[0] = float( slicePtr1[neighborsIndex[0]] - slicePtr1[neighborsIndex[1]] );
          gradient[1] = float( slicePtr1[neighborsIndex[2]] - slicePtr1[neighborsIndex[3]] );
          gradient[2] = float( slicePtr1[neighborsIndex[4]] - slicePtr1[neighborsIndex[5]] );
          gradient /= sqrt( 3.f );
          // Lenght of the gradient vector.
          value2 = (unsigned short) ( sqrt( double( ( gradient[0] * gradient[0] ) + ( gradient[1] * gradient[1] ) + ( gradient[2] * gradient[2] ) ) ) );
          value2Remaped = map( value2, 0, gradientMaxValue, 0, HISTOGRAM_RESOLUTION - 1 );
#endif

          if ( histo[value2Remaped][value1Remaped] < UCHAR_MAX )
            histo[value2Remaped][value1Remaped]++;
        }
      }
    }

    texHistogram->image.setValue( SbVec2i32( HISTOGRAM_RESOLUTION, HISTOGRAM_RESOLUTION ), 1, SoSFImage::UNSIGNED_BYTE, histo );
    texHistogram->style = SoImageBackground::STRETCH;
  }

  cpuBufferObject1->unmap();
#ifdef USE_DATA2
  cpuBufferObject2->unmap();
#endif
}

int
main( int /*argc*/, char** argv )
{
  Widget mainWindow = SoXt::init( argv[0] );
  SoDialogViz::init();
  InventorMedical::init();
  SoTransferFunction2D::initClass();

  SoOrthographicCamera* orthoCam = new SoOrthographicCamera;
  orthoCam->nearDistance.setValue( 0.01f );

  SoSeparator* shapesRoot = new SoSeparator;
  shapesRoot->setName( "shapesRoot" );

  // Shape hint use to generate shapes normals.
  SoShapeHints* shapeHint = new SoShapeHints;
  shapesRoot->addChild( shapeHint );
  // Must be less than PI/2: because of perspective projection, angle between body and top/bottom can be up to PI/2 - camFov/2.
  shapeHint->creaseAngle = (float) ( 0.3 * M_PI );
  shapeHint->windingType = SoShapeHints::NO_WINDING_TYPE;
  shapeHint->vertexOrdering = SoShapeHints::COUNTERCLOCKWISE;
  shapeHint->shapeType = SoShapeHints::SOLID;
  shapeHint->faceType = SoShapeHints::CONVEX;

  //------------------------------------------------------------------------------
  // Add line drawers.
  SoSwitch* lineDrawerSwitch = new SoSwitch;
  lineDrawerSwitch->setName( "LINE_DRAWER_SWITCH" );
  lineDrawerSwitch->whichChild = 0;

  // Add lasso screen drawer.
  SoPolyLineScreenDrawer* lasso = new SoLassoScreenDrawer;
  lasso->setName( "LASSO_SCREEN_DRAWER" );
  lasso->onFinish.add( lineFinalizedCallback );
  lasso->simplificationThreshold = 3;
  lineDrawerSwitch->addChild( lasso );

  // Add polygon screen drawer.
  SoPolyLineScreenDrawer* polygon = new SoPolygonScreenDrawer;
  polygon->setName( "POLYGON_SCREEN_DRAWER" );
  polygon->onFinish.add( lineFinalizedCallback );
  lineDrawerSwitch->addChild( polygon );

  SoLightModel* lightModel = new SoLightModel;
  lightModel->model = SoLightModel::BASE_COLOR;

  SoSeparator* materialSep = new SoSeparator;
  pColor = new SoMaterial;
  materialSep->addChild( pColor );

  SoSeparator* sep = new SoSeparator;
  sep->addChild( new SoPerspectiveCamera );
  sep->addChild( orthoCam );
  sep->addChild( shapesRoot );
  sep->addChild( lineDrawerSwitch );
  sep->addChild( lightModel );
  sep->addChild( materialSep );

  SoSeparator* sep2 = new SoSeparator;

  sep2->addChild( new SoPerspectiveCamera );
  sep2->addChild( orthoCam );
  SoSeparator* shapesRoot2 = new SoSeparator;
  shapesRoot2->setName( "shapesRoot_RenderToTex" );
  shapesRoot2->addChild( shapeHint );
  sep2->addChild( shapesRoot2 );
  sep2->addChild( lineDrawerSwitch );
  sep2->addChild( lightModel );
  sep2->addChild( materialSep );

  pRoot = new SoSeparator;
  pRoot->addChild( sep );

  pToRender = new SoSeparator;
  pToRender->addChild( sep2 );

  // Set up viewer.
  Widget parent = MedicalHelper::buildInterface( mainWindow, "$OIVHOME/examples/source/Medical/TransferFunction/medical2DTransferFunction/interface.iv", "viewer0", &pTopLevelDialog0 );

  SoImageBackground* texHistogram = new SoImageBackground;
  pRoot->addChild( texHistogram );

  SoDialogRealSlider* slider;
  slider = (SoDialogRealSlider*) pTopLevelDialog0->searchForAuditorId( "transparency" );
  pColor->transparency.set1Value( 0, slider->getValue() );

  // Connect combobox to lineDrawer switch.
  SoDialogComboBox* modeCombo;
  modeCombo = (SoDialogComboBox*) pTopLevelDialog0->searchForAuditorId( "selectionMode" );
  lineDrawerSwitch->whichChild.connectFrom( &modeCombo->selectedItem );

  // This viewer is made of the data histogram and the screen drawer.
  // The user is able to draw on the screen an area to select the voxel to draw
  // with selected range of data value and gradient.
  SoXtExaminerViewer* transferFunction2DViewer = new SoXtExaminerViewer( parent );
  transferFunction2DViewer->setDecoration( FALSE );
  transferFunction2DViewer->setViewing( FALSE );
  transferFunction2DViewer->setSceneGraph( pRoot.ptr() );
  transferFunction2DViewer->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );
  transferFunction2DViewer->show();
  SoXt::show( mainWindow );

  // This viewer is 'offscreen' and is used on top of the same widget so a renderAction
  // can be launched automatically. It uses the same scene graph as the previous viewer
  // but instead of rendering the screen drawer result, it creates a shape based on the
  // drawn curve and add it to the scene graph using the drawer callback: lineFinalizedCallback.
  SoXtExaminerViewer* transferFunction2DViewer2 = new SoXtExaminerViewer( parent );
  transferFunction2DViewer2->setDecoration( FALSE );
  transferFunction2DViewer2->setViewing( FALSE );
  transferFunction2DViewer2->setSceneGraph( pToRender.ptr() );
  transferFunction2DViewer2->setTransparencyType( SoGLRenderAction::OPAQUE_FIRST );

  SoTransferFunction2D* tex2D = new SoTransferFunction2D;
  tex2D->setName( "transferFunction2D" );

  SoRenderToTextureProperty* pRenderToTexProperty = new SoRenderToTextureProperty;
  tex2D->renderToTextureProperty.setValue( pRenderToTexProperty );

  pRenderToTexProperty->component = SoRenderToTextureProperty::RGB_TRANSPARENCY;
  pRenderToTexProperty->size.setValue( SbVec2s( 256, 256 ) );
  pRenderToTexProperty->updatePolicy = SoRenderToTextureProperty::WHEN_NEEDED;
  pRenderToTexProperty->setName( "renderToTexProperty" );

  pRenderToTexProperty->node.set1Value( 0, (SoNode*) pToRender.ptr() );

  /////////////////////////////////////////////////////////////
  // Data 1
  SoVolumeData* pVolData1 = new SoVolumeData;
  pVolData1->fileName = "$OIVHOME/examples/data/Medical/files/tooth.ldm";
  pVolData1->setName( "volData1" );

#ifdef USE_DATA2
  /////////////////////////////////////////////////////////////
  // Data 2
  SoVolumeData* pVolData2 = new SoVolumeData;
  pVolData2->fileName = "$OIVHOME/examples/data/Medical/files/tooth_gradient.ldm";
  pVolData2->dataSetId = 2;
  pVolData2->setName( "volData2" );
#endif

  // Node in charge of drawing the volume.
  SoVolumeRender* pVolRender = new SoVolumeRender;
  pVolRender->numSlicesControl = SoVolumeRender::AUTOMATIC;
  pVolRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;

  tex2D->internalFormat.setValue( SoTransferFunction2D::RGBA_FORMAT );
  tex2D->minFilter = SoTransferFunction2D::NEAREST;
  tex2D->magFilter = SoTransferFunction2D::NEAREST;

  // Initialize and set the volume shader program.
  SoVolumeRenderingQuality* pVolShader = new SoVolumeRenderingQuality;

  // Specify a fragment shader to combine the intensity and color values.
  // First load the fragment shader code.
  SoFragmentShader* fragmentShader = new SoFragmentShader;
  fragmentShader->sourceProgram.setValue( "$OIVHOME/examples/source/Medical/TransferFunction/medical2DTransferFunction/2DTransferFunctionCustomShader.glsl" );

  // Set the shader parameters.
  fragmentShader->addShaderParameter1i( "tex2D", SoPreferences::getInt( "IVVR_TF2D_TEX_UNIT", 15 ) ); // for now force to 15
  fragmentShader->addShaderParameter1i( "data1", 1 );

  // We need to make sure that the value sent to the GPU are in this range specifically as this is the range
  // of values we conssidered to build on histogram.
  double min, max;
  pVolData1->getMinMax( min, max );
  SoDataRange* range = new SoDataRange;
  range->min.setValue( min );
  range->max.setValue( max );

  pVolShader->shaderObject.set1Value( SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader );

  // Note: This demo currently REQUIRES gradient-based lighting (not deferred lighting).
  pVolShader->preIntegrated = TRUE;
  pVolShader->lighting = TRUE;

#ifdef USE_DATA2
  fragmentShader->addShaderParameter1i( "data2", 2 );
  fragmentShader->setDefine( "USE_DATA2", 1 );
#endif

  // Assemble the scene graph.
  SoMultiDataSeparator* mds = new SoMultiDataSeparator;
  mds->addChild( tex2D );
  mds->addChild( range );
  mds->addChild( pVolShader );
  mds->addChild( pVolData1 );
#ifdef USE_DATA2
  mds->addChild( pVolData2 );
#endif
  mds->addChild( pVolRender );

  pRenderRoot = new SoSeparator;
  pRenderRoot->addChild( mds );

  // This third viewer is the volume rendering viewer and it uses the shape created by transferFunction2DViewer2
  // and a renderToTexture node to fill a texture with the desired voxel values and gradient to render.
  // The result of the renderToTexture is a texture where pixels are set to 1 inside the shape and 0 outside.
  // The fragment shader added before the volumeRender access this texture and for each fragment, look
  // at its corresponding data V and gradient G, extract the value from the texture:
  // texture2D( tex2D, vec2( index1, gradien t) ); and assign an output value accordingly.
  SoDialogCustom* customNode1 = (SoDialogCustom*) pTopLevelDialog0->searchForAuditorId( SbString( "viewer1" ) );
  SoXtExaminerViewer* renderViewer = new SoXtExaminerViewer( customNode1->getWidget() );
  renderViewer->setDecoration( FALSE );
  renderViewer->setSceneGraph( pRenderRoot.ptr() );
  renderViewer->show();

  SoDialogCustom* customNode2 = (SoDialogCustom*) pTopLevelDialog0->searchForAuditorId( SbString( "colorEditor" ) );
  SoXtColorWheel* colorWheel = new SoXtColorWheel( customNode2->getWidget() );
  colorWheel->addFinishCallback( colFinalizedCallback );
  colorWheel->show();
  send2DHistogram( texHistogram );

  SoXt::mainLoop();

  pRoot = NULL;
  pRenderRoot = NULL;
  pToRender = NULL;

  delete renderViewer;
  delete colorWheel;
  delete transferFunction2DViewer;
  delete transferFunction2DViewer2;

  // Release modules.
  SoTransferFunction2D::exitClass();
  InventorMedical::finish();
  SoDialogViz::finish();
  SoXt::finish();

  return 0;
}
