///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////
/*----------------------------------------------------------------------------------------
Example program.
Purpose : Demonstrate how to create and use a color map
author : Jerome Hummel
Updaded by Pascal Estrade (Sep 2014)
August 2002
----------------------------------------------------------------------------------------*/

//header files
#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <VolumeViz/nodes/SoVolumeRendering.h>

#include <Inventor/helpers/SbFileHelper.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>

///////////////////////////////////////////////////////////////////////////////

//file name
SbString DataFile = SbFileHelper::expandString("$OIVHOME/examples/data/Medical/files/head.ldm");
SbString fileName = DataFile;

///////////////////////////////////////////////////////////////////////////////
//Main function
int main(int , char **argv)
{
  // Create the window
  Widget myWindow = SoXt::init( argv[0] );
  if (!myWindow)return 0;
  
  SoVolumeRendering::init();
  InventorMedical::init();

  SoRef<SoSeparator> root = new SoSeparator;

  // Camera for scene
  root->addChild( new SoPerspectiveCamera() );

  // Volume rendering sub-graph
  SoSeparator* volSep = new SoSeparator();
    root->addChild( volSep );

  // Create volumeData node with a filename
  // Note: VolumeViz will try to open and load the data using one
  //       of the builtin file loaders (see SoVolumeReader).
  SoVolumeData* volData = new SoVolumeData();
    volData->fileName.setValue( fileName );
    volSep->addChild( volData );

  // Map full range of actual data values to color map.
  SoDataRange* volRange = new SoDataRange();
    double minval, maxval;
    volData->getMinMax( minval, maxval );
    volRange->min = minval;
    volRange->max = maxval;
    volSep->addChild( volRange );
  
  // Transfer function description
  // Using a predefined color map would use for example this code :
  //   pTransFunc->predefColorMap = SoTransferFunction::GREY;
  //
  // But we want to create a new colorMap here.
  // For demo purposes, this code shows how to create the predefined
  // STANDARD colorMap with 256 entries.
  SoTransferFunction* volTF = new SoTransferFunction;
  int numEntries = 256;
  volTF->colorMap.setNum( 256 * 4 );
  for (int i = 0; i < numEntries; i++) {
    float f = (float)i / 255;
    SbColor rgb; 
    rgb.setHSVValue( 0.916f * f, 1, 1 );
    volTF->colorMap.set1Value( 4*i+0, rgb[0] ); // red
    volTF->colorMap.set1Value( 4*i+1, rgb[1] ); // green
    volTF->colorMap.set1Value( 4*i+2, rgb[2] ); // blue
    volTF->colorMap.set1Value( 4*i+3, 0.3f   ); // alpha
  }
  volTF->minValue = 69;
  volSep->addChild( volTF );

  // Volume rendering parameters
  SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
    volQual->preIntegrated = TRUE;
    volQual->interpolateOnMove = TRUE;
    volSep->addChild( volQual );

  // Create the render node
  SoVolumeRender* volRender = new SoVolumeRender;
    volRender->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
    volSep->addChild( volRender );
  
  // Add logo
  root->addChild( MedicalHelper::exampleLogoNode() );

  // Add orientation marker
  root->addChild( new Gnomon() );

  // Viewer
  SoXtExaminerViewer* viewer = new SoXtExaminerViewer( myWindow );
  viewer->setTitle( "Custom color Map" );
  viewer->setDecoration(FALSE);
  viewer->setSize( MedicalHelper::exampleWindowSize() );
  viewer->setSceneGraph(root.ptr());

  // Adjust camera
  MedicalHelper::orientView( MedicalHelper::SAGITTAL, viewer->getCamera() );
  viewer->viewAll();
  viewer->saveHomePosition();
  viewer->show();
  
  // Loop forever
  SoXt::show( myWindow );
  SoXt::mainLoop();
  delete viewer;
  root = NULL;
  InventorMedical::finish();
  SoVolumeRendering::finish();
  SoXt::finish();
  return 0;
}


