///////////////////////////////////////////////////////////////////////////////
//
// This program is part of the Open Inventor Medical example set.
//
// Open Inventor customers may use this source code to create or enhance
// Open Inventor-based applications.
//
// The medical utility classes are provided as a prebuilt library named
// "fei.inventor.Medical", that can be used directly in an Open Inventor
// application. The classes in the prebuilt library are documented and
// supported by Thermo Fisher Scientific. These classes are also provided as source code.
//
// Please see $OIVHOME/include/Medical/InventorMedical.h for the full text.
//
///////////////////////////////////////////////////////////////////////////////

/*=======================================================================
** Author      : Julien Sallane (feb 2014)
** Updaded by Pascal Estrade (Sep 2014)
**=======================================================================*/

// Multi-transferFunction demo
// Colormap applied to each data voxel is specified by corresponding label voxel.

#include <Inventor/Xt/SoXt.h>
#include <Inventor/Xt/viewers/SoXtExaminerViewer.h>

#include <Inventor/nodes/SoFragmentShader.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>

#include <Inventor/elements/SoViewVolumeElement.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/misc/SbExtrusionGenerator.h>

#include <Inventor/drawers/SoLassoScreenDrawer.h>
#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/SbDataType.h>

#include <VolumeViz/nodes/SoDataRange.h>
#include <VolumeViz/nodes/SoTransferFunction.h>
#include <VolumeViz/nodes/SoVolumeData.h>
#include <VolumeViz/nodes/SoVolumeRender.h>
#include <VolumeViz/nodes/SoVolumeShader.h>
#include <VolumeViz/nodes/SoVolumeRenderingQuality.h>
#include <LDM/nodes/SoMultiDataSeparator.h>

#include <Inventor/helpers/SbFileHelper.h>
#include <Inventor/helpers/SbDataTypeMacros.h>

#include <DialogViz/SoDialogVizAll.h>

#include <Medical/InventorMedical.h>
#include <Medical/helpers/MedicalHelper.h>
#include <Medical/nodes/Gnomon.h>
#include <Medical/nodes/TextBox.h>

///////////////////////////////////////////////////////////////////////////////

const SbString VOLUME_FILE = "$OIVHOME/examples/data/Medical/files/head.ldm";

const SbString SHADER_FILE = 
  "$OIVHOME/examples/source/Medical/TransferFunction/medicalMultiTransferFunctions/multiTransferFunctionFragmentShader.glsl";

const SbString INTERFACE_FILE = 
  "$OIVHOME/examples/source/Medical/TransferFunction/medicalMultiTransferFunctions/interface.iv";

///////////////////////////////////////////////////////////////////////////////
// Application state

static SoRef<SoSeparator> m_root = NULL;

static SoRef<SoTopLevelDialog> m_topLevelDialog = NULL;

static int m_currentLabel = 5;


///////////////////////////////////////////////////////////////////////////////
// This function will be called each time a line created by the line drawer will be finalized.
// It retrieve points of the line in camera space, extrude it and write it in dataSet.
void lineDrawerFinalizeCallback(SoPolyLineScreenDrawer::EventArg& eventArg)
{
  SoPolyLineScreenDrawer* lineDrawer = eventArg.getSource();
  SoHandleEventAction* action = eventArg.getAction();

    // If less than 1 point, shape cannot be generated.
    if ( lineDrawer->point.getNum() < 1 )
    {
        lineDrawer->clear();
        return;
    }

    // retrieve points of line in camera space
    std::vector<SbVec2f> lineInCam( lineDrawer->point.getNum() );
    for ( unsigned int i = 0; i < lineInCam.size(); ++i )
        lineInCam[i].setValue( lineDrawer->point[i][0], lineDrawer->point[i][1] );

    SoVolumeData* vdLabel = MedicalHelper::find<SoVolumeData>(m_root.ptr(), "VD_LABEL");

    // create a new extruded shape :
    // retrieve path of dataset
    SoSearchAction searchAction;
      searchAction.setNode( vdLabel );
      searchAction.apply( m_root.ptr() );
      SoPath* pathToExtrudedShapeRoot = searchAction.getPath();

    // retrieve bounding box of label volume.
    SbBox3f bbox = MedicalHelper::getBoundingBox(vdLabel);

    // create an extruded shape from specified line. Line is extruded along view
    // direction between bounding box enclosing planes.
    SoShape* extrudedShape = SbExtrusionGenerator::createFrom2DPoints( lineInCam,
        pathToExtrudedShapeRoot,
        SoViewVolumeElement::get(action->getState()),
        bbox );
    if ( extrudedShape == NULL )
    {
        lineDrawer->clear();
        return;
    }

    // write shape to dataset
    extrudedShape->ref();
    int editionId = 0;
    vdLabel->startEditing( editionId );
    vdLabel->editSolidShape( extrudedShape, (double)m_currentLabel );
    vdLabel->finishEditing( editionId );

    extrudedShape->unref();

    lineDrawer->clear();
    action->setHandled();
}

///////////////////////////////////////////////////////////////////////////////
// Setup the volume shader node
void setupShader()
{
    // Load fragment shader program
    SoFragmentShader* fragmentShader = new SoFragmentShader;
    fragmentShader->sourceProgram.setValue( SHADER_FILE );

    // Create uniform parameters allowing shader to access textures
    // Set Label texture unit
    SoShaderParameter1i *paramTexVdLabel = new SoShaderParameter1i;
      SoVolumeData* vdLabel = MedicalHelper::find<SoVolumeData>(m_root.ptr(), "VD_LABEL");
      paramTexVdLabel->name = "volumeLabel";
      paramTexVdLabel->value.connectFrom(&vdLabel->dataSetId);
      fragmentShader->parameter.set1Value(0, paramTexVdLabel);

    // Set Data texture unit
    SoShaderParameter1i *paramTexVdData = new SoShaderParameter1i;
      SoVolumeData* vdData = MedicalHelper::find<SoVolumeData>(m_root.ptr(), "VD_DATA");
      paramTexVdData->name = "volumeData";
      paramTexVdData->value.connectFrom(&vdData->dataSetId);
      fragmentShader->parameter.set1Value(1, paramTexVdData);

    // Initialize and set the volume shader program
    SoVolumeRenderingQuality* vrqShader = MedicalHelper::find<SoVolumeRenderingQuality>(m_root.ptr());
    // Specify to what part the shader prog is intended to. Here fragments color only.
    vrqShader->shaderObject.set1Value(SoVolumeShader::FRAGMENT_COMPUTE_COLOR, fragmentShader);
}

///////////////////////////////////////////////////////////////////////////////
// Fill VolumeData vdLabel with the value val using vdData parameters.
template <typename T>
void fillLabelsVolumeData(SoVolumeData* vdLabel, SoVolumeData* vdData, T val)
{
    const SbVec3i32 dataDim = vdData->data.getSize();
    int dataSize = dataDim[0] * dataDim[1] * dataDim[2];// * ((SbDataType)vdData->data.getDataType()).getSize();

    std::vector<T> data(dataSize);
    for(int i = 0; i < dataSize; ++i)
        data[i] = (T)val;

    vdLabel->data.setValue(dataDim, vdData->data.getDataType(), vdData->data.getNumSigBits(), &data[0], SoSFArray3D::COPY);
}


///////////////////////////////////////////////////////////////////////////////
// Class used to interpret event from the GUI.
class auditorClass : public SoDialogAuditor
{
    void dialogComboBox(SoDialogComboBox* cpt)
    {
        if (cpt->auditorID.getValue() == "transferFunction")
            m_currentLabel = cpt->selectedItem.getValue();
    }

    void dialogIntegerSlider(SoDialogIntegerSlider* cpt)
    {
      // Get a list of all the transfer function nodes
      std::vector<SoTransferFunction*> transferFunctions = MedicalHelper::findNodes<SoTransferFunction>(m_root.ptr());
      if (cpt->auditorID.getValue() == "minValue")
          transferFunctions[m_currentLabel]->minValue = cpt->value.getValue();
      else if (cpt->auditorID.getValue() == "maxValue")
          transferFunctions[m_currentLabel]->maxValue = cpt->value.getValue();
    }
};

///////////////////////////////////////////////////////////////////////////////
// Build user interface
Widget BuildInterface(Widget window)
{
  m_topLevelDialog = (SoTopLevelDialog*)SoDialogViz::loadFromFile(INTERFACE_FILE);
  m_topLevelDialog->buildDialog(window, TRUE);
  m_topLevelDialog->label.setValue("Multi transfer functions");

  SoDialogCustom* customNode = (SoDialogCustom*)m_topLevelDialog->searchForAuditorId("viewer");
  Widget viewerWidget = customNode->getWidget();

  auditorClass* auditor = new auditorClass;
  m_topLevelDialog->addAuditor(auditor);
  m_topLevelDialog->show();

  return viewerWidget;
}

////////////////////////////////////////////////////////////////////////
//main function
int main(int, char **argv)
{
    // Create the window
    Widget window = SoXt::init(argv[0]);
    if (!window) return 0;

    std::string errmsg("Unable to open:");

    if (! SbFileHelper::isAccessible(VOLUME_FILE)) {
      SoXt::createSimpleErrorDialog(window, const_cast<char*>(errmsg.c_str()), (char*)VOLUME_FILE.toLatin1());
      return -1;
    }
    if (! SbFileHelper::isAccessible(SHADER_FILE)) {
      SoXt::createSimpleErrorDialog(window, const_cast<char*>(errmsg.c_str()), (char*)SHADER_FILE.toLatin1());
      return -1;
    }
    if (! SbFileHelper::isAccessible(INTERFACE_FILE)) {
      SoXt::createSimpleErrorDialog(window, const_cast<char*>(errmsg.c_str()), (char*)INTERFACE_FILE.toLatin1());
      return -1;
    }

    SoDialogViz::init();
    SoVolumeRendering::init();
    InventorMedical::init();

    // Build scene graph
    m_root = new SoSeparator();

    // Camera
    SoPerspectiveCamera* camera = new SoPerspectiveCamera();
      m_root->addChild( camera );

    // Volume rendering stuff
    SoSeparator* volSep = new SoSeparator();
      m_root->addChild( volSep );

    // Create transfer functions
    const int NUM_MAPS = 6;
    SoTransferFunction::PredefColorMap maps[] = {
      SoTransferFunction::GRAY,
      SoTransferFunction::TEMPERATURE,
      SoTransferFunction::SEISMIC,
      SoTransferFunction::PHYSICS,
      SoTransferFunction::GLOW,
      SoTransferFunction::STANDARD,
    };

    SoGroup* tfGroup = new SoGroup();
      volSep->addChild( tfGroup );

    for (int i = 0; i < NUM_MAPS; ++i) {
      SoTransferFunction* map = new SoTransferFunction();
        map->predefColorMap = maps[i];
        map->transferFunctionId = i;
        map->minValue = 27;
        tfGroup->addChild( map );
    }

    // To manage multiple volumes in a shader we need a special group node.
    SoMultiDataSeparator* multiSep = new SoMultiDataSeparator();
      volSep->addChild( multiSep );

    // Create data volume
    SoVolumeData* volData = new SoVolumeData();
      volData->fileName = VOLUME_FILE;
      MedicalHelper::dicomAdjustVolume( volData );
      volData->dataSetId = 1;
      volData->setName( "VD_DATA" );
      multiSep->addChild( volData );

    // Create label volume (filled in later)
    SoVolumeData* volLabel = new SoVolumeData();
      volLabel->dataSetId = 2;
      volLabel->setName( "VD_LABEL" );
      multiSep->addChild( volLabel );

    SoVolumeRenderingQuality* volQual = new SoVolumeRenderingQuality();
      volQual->ambientOcclusion = TRUE;
      volQual->deferredLighting = TRUE;
      volQual->interpolateOnMove = TRUE;
      multiSep->addChild( volQual );

    SoVolumeRender* volRend = new SoVolumeRender();
      volRend->subdivideTile = TRUE;
      volRend->samplingAlignment = SoVolumeRender::BOUNDARY_ALIGNED;
      volRend->numSlicesControl  = SoVolumeRender::AUTOMATIC;
      volRend->lowResMode = SoVolumeRender::DECREASE_SCREEN_RESOLUTION;
      volRend->opacityThreshold = 0.1f;
      multiSep->addChild( volRend );

    // Screen drawer subgraph
    SoSeparator* drawSep = new SoSeparator();
      drawSep->fastEditing = SoSeparator::CLEAR_ZBUFFER;
      drawSep->setName( "Screen Drawing" );
      m_root->addChild( drawSep );

    // The callback will be called after the line is drawn.
    // There we retreive the points and create the label subVolume.
    SoPolyLineScreenDrawer* drawer = new SoLassoScreenDrawer();
      drawer->isClosed = TRUE;
      drawer->simplificationThreshold = 7;
      // The callback that, after the line is drawn, retreive points and extract label subVolume.
      drawer->onFinish.add(lineDrawerFinalizeCallback);
      drawSep->addChild( drawer );
      
    // Setup fragment shader and corresponding uniform values.
    setupShader();

    // Init label data (extent and tile dimension must match the data volume)
    SB_DATATYPE_CALL(fillLabelsVolumeData,( volLabel, volData, 0 ), static_cast<SbDataType>(volData->data.getDataType()));
    volLabel->ldmResourceParameters.getValue()->tileDimension.setValue(volData->ldmResourceParameters.getValue()->tileDimension.getValue());
    volLabel->extent = volData->extent.getValue();

    // OIV Logo
    m_root->addChild( MedicalHelper::exampleLogoNode() );

    // Orientation (must appear after camera - which is in the loaded scene graph file)
    m_root->addChild( new Gnomon() );

    // Instructions
    TextBox* text = new TextBox();
      text->position.setValue(0, -0.98f, 0); // Normalized device coordinates
      text->alignmentH = TextBox::CENTER;
      text->alignmentV = TextBox::BOTTOM;
      text->addLine("Click and drag to lasso a portion of the volume.");
      text->addLine("Selected color map will be applied to the selected voxels.");
      m_root->addChild(text);

    // Build user interface
    Widget parent = BuildInterface( window );

    // Set up viewer (do not call setSize - size is defined in the UI file)
    SoXtExaminerViewer *myViewer = new SoXtExaminerViewer( parent );
      myViewer->setSceneGraph( m_root.ptr() );
      myViewer->setTitle("Multi transfer functions");
      myViewer->setTransparencyType(SoGLRenderAction::OPAQUE_FIRST);
      myViewer->setDecoration( FALSE );
      myViewer->setViewing( FALSE );

    // Adjust camera
    MedicalHelper::orientView( MedicalHelper::SAGITTAL, myViewer->getCamera(), volData );
    MedicalHelper::dollyZoom( 1.5f, myViewer->getCamera() ); // Fill viewport (viewAll is too conservative)
    myViewer->saveHomePosition();

    // Run then cleanup
    myViewer->show();
    SoXt::show(window);
    SoXt::mainLoop();
    delete myViewer;
    m_root = NULL;
    m_topLevelDialog = NULL;
    InventorMedical::finish();
    SoVolumeRendering::finish();
    SoDialogViz::finish();
    SoXt::finish();
    return 0;
}


